/* globals GravityKit, jQuery, ajaxurl, Qs, google, window */
/**
 * Setups the main variable for this file.
 *
 * @since 3.1.0
 *
 * @type   {Object}
 */
GravityKit.GravityMaps.Controllers = GravityKit.GravityMaps.Controllers || {};
GravityKit.GravityMaps.Controllers.Marker = {};

( ( $, obj ) => {
    "use strict";
	const Map = GravityKit.GravityMaps.Controllers.Map;

	/**
	 * Creates a local alias for Utils.className() method.
	 */
	const className = GravityKit.GravityMaps.Utils.className;

	const {
		doAction,
		addAction,
		applyFilters,
		addFilter,
	} = GravityKit.GravityMaps.hooks;

	/**
	 * Attach all the hooks into the Actions and Filters used by the maps system.
	 *
	 * @since 3.1.0
	 */
	obj.hook = () => {
		addAction( 'gk.maps.controllers.map.after_create_map', 'gravitykit/maps', obj.processMap );
	};

	/**
	 * Process all markers for a given Map.
	 *
	 * It will use the DOM element to determine which map we need to process the markers for.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 */
	obj.processMap = ( $map ) => {
		const data = Map.getData( $map );

		if ( $map.hasClass( className( Map.selectors.generatedMapMarkers ) ) ) {
			return;
		}

		/**
		 * @action gk.maps.controllers.marker.before_process_map Fires before the marker creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map The jQuery object of the map.
		 */
		doAction( 'gk.maps.controllers.marker.before_process_map', $map );

		data.markers_data.forEach( ( marker ) => obj.create( $map, marker ) );

		$map.addClass( className( Map.selectors.generatedMapMarkers ) );

		/**
		 * @action gk.maps.controllers.marker.after_process_map Fires after the marker creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map The jQuery object of the map.
		 */
		doAction( 'gk.maps.controllers.marker.after_process_map', $map );
	};

	/**
	 * Adds an individual marker to a particular map, it will use the jQuery/DOM element of the map.
	 * Will also store that given marker into the `gkMap` data object.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {object} markerData
	 */
	obj.create = ( $map, markerData ) => {
		const settings = Map.getSettings( $map );
		const data = Map.getData( $map );

		/**
		 * @action gk.maps.controllers.marker.create Enables the ability to hook into before the marker creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map
		 * @param {object} markerData
		 */
		doAction( 'gk.maps.controllers.marker.create', $map, markerData );

		/**
		 * @action gk.maps.controllers.marker.after_create Enables the ability to hook into after the marker creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map
		 * @param {object} markerData
		 */
		doAction( 'gk.maps.controllers.marker.after_create', $map, markerData );
	};

	/**
	 * Given a Map jQuery object test to see if we have a particular marker already set, uses the entryId for check.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {object} checkMarker
	 *
	 * @returns {boolean|int}
	 */
	obj.exists = ( $map, checkMarker ) => {
		const data = Map.getData( $map );
		let exists = false;

		data.markers.forEach( ( marker, index ) => {
			if ( checkMarker.entryId && checkMarker.entryId == marker.entryId ) {
				exists = index;
			}
			if ( checkMarker.entry_id && checkMarker.entry_id == marker.entryId ) {
				exists = index;
			}
		} );

		return exists;
	};

	/**
	 * Given a Map jQuery object test to see if we have a particular marker data already set, uses the entryId for check.
	 *
	 * Marker data is used before the marker is configured with Google Maps API.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {object} checkMarker
	 *
	 * @returns {boolean|int}
	 */
	obj.dataExists = ( $map, checkMarker ) => {
		const data = Map.getData( $map );
		let exists = false;

		data.markers_data.forEach( ( marker, index ) => {
			if ( checkMarker.entryId && checkMarker.entryId == marker.entry_id ) {
				exists = index;
			}
			if ( checkMarker.entry_id && checkMarker.entry_id == marker.entry_id ) {
				exists = index;
			}
		} );

		return exists;
	};

	/**
	 * Bounce the marker on mouse over the entry.
	 *
	 * @since 3.1.0
	 *
	 * @param {Event} event
	 *
	 * @return {void}
	 */
	obj.onViewEntryMouseEnter = ( event ) => {
		const $entry = $( event.currentTarget );
		const entryId = $entry.attr( 'id' );

		if ( ! entryId ) {
			return;
		}

		const $wrapper = $entry.parents( Map.selectors.viewWrapper ).eq( 0 );
		const $map = $wrapper.find( Map.selectors.multipleEntryMap ).eq( 0 );

		if ( ! $map.length ) {
			return;
		}

		const data = Map.getData( $map );
		const settings = Map.getSettings( $map );

		if ( ! settings.icon_bounce ) {
			return;
		}

		const markerId = entryId.replace( 'gv_map_', '' );

		const markers = data.markers.filter( marker => marker.entryId === markerId );

		/**
		 * @action gk.maps.controllers.marker.on_view_entry_mouse_enter Enables the ability to hook into before the process of hovering over an entry.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map
		 * @param {array} markers
		 */
		doAction( 'gk.maps.controllers.marker.on_view_entry_mouse_enter', $map, markers );
	};

	/**
	 * Triggers when the window finishes loading.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.load = () => {
		$( document ).on( 'mouseenter', Map.selectors.viewEntry, obj.onViewEntryMouseEnter );
	};

	/**
	 * Hooking needs to happen as early as possible.
	 */
	obj.hook();

	$( window ).on( 'load', obj.load );
} )( jQuery, GravityKit.GravityMaps.Controllers.Marker );
