( ( $, obj, GravityMaps ) => {
	"use strict";

	// Create a global Reference for this globally.
	GravityMaps.searchFields = obj;

	const { Map, Marker, View } = GravityKit.GravityMaps.Controllers;
	const {
		doAction,
		addAction,
		applyFilters,
		addFilter,
	} = GravityKit.GravityMaps.hooks;
	const { className, mileToKm, debug } = GravityKit.GravityMaps.Utils;

	/**
	 * Object holding all the selectors used for this module.
	 *
	 * @since 2.2
	 *
	 * @type {{currentLocationButton: string, currentLocationLat: string, currentLocationAccuracy: string, currentLocationLng: string}}
	 */
	obj.selectors = {
		currentLocationButton: '.gk-maps-search-current-geolocation',
		searchBox: '.gv-search-box',
		searchField: '.gk-maps-search-geolocation-address-autocomplete',
		searchFieldCurrentLocation: '.gk-maps-search-geolocation-address-autocomplete-current-location',
		currentLocationButtonActive: '.gk-maps-search-current-geolocation-active',
		currentLocationLng: '.gk-maps-search-geolocation-lng',
		currentLocationLat: '.gk-maps-search-geolocation-lat',
		currentLocationAccuracy: '.gk-maps-search-current-geolocation-accuracy',
		currentLocationFlag: '.gk-maps-search-current-geolocation-flag',
		currentLocationRadius: '.gk-maps-search-current-geolocation-radius',
		currentLocationUnit: '.gk-maps-search-current-geolocation-unit',
		currentLocationHasSearch: '.gk-maps-search-current-geolocation-has-search',

		viewContainer: View.selectors.container,
		viewWrapper: View.selectors.wrapper,
		viewWrapperById: View.selectors.wrapperById,
		viewContainerById: View.selectors.containerById,

		mapsEntriesContainer: '.gv-map-entries',
		widgetsHeader: '.gv-widgets-header',
		noResults: '.gv-no-results',
		noResultsContainerFlag: '.gk-maps-no-results-container-flag',
	};

	/**
	 * The jQuery object holding the current button for currentLocation.
	 *
	 * @since 2.2
	 *
	 * @type {null|jQuery}
	 */
	obj.$currentLocationButton = null;

	/**
	 * Internationalization for teh Maps Search fields JS.
	 *
	 * @since 2.2
	 *
	 * @type {object}
	 */
	obj.i18n = window.gk_maps_search_fields;

	/**
	 * Pull the current location and saves on a hidden field in the search form.
	 *
	 * @since 2.2
	 *
	 * @param {Event} event
	 *
	 * @return {void}
	 */
	obj.onCurrentLocationClick = ( event ) => {
		event.preventDefault();

		const $target = $( event.target );
		const options = {
			enableHighAccuracy: true,
			timeout: 5000,
			maximumAge: 0
		};

		if ( $target.is( obj.selectors.currentLocationButton ) ) {
			obj.$currentLocationButton = $target;
		} else {
			obj.$currentLocationButton = $( event.target ).parents( obj.selectors.currentLocationButton ).eq( 0 );
		}

		navigator.geolocation.getCurrentPosition( obj.handleCurrentLocationSuccess, obj.handleCurrentLocationError, options );
	};

	obj.resetCurrentLocation = ( $field ) => {
		const $searchBox = $field.parents( obj.selectors.searchBox ).eq( 0 );

		$searchBox.find( obj.selectors.currentLocationButton )
			.removeClass( className( obj.selectors.currentLocationButtonActive ) )
			.prop( 'title', obj.i18n.currentLocationLabel );

		$searchBox.find( obj.selectors.currentLocationFlag ).val( 0 );
	};

	/**
	 * Handles saving the current location on the fields related to the button that was clicked.
	 *
	 * @since 2.2
	 *
	 * @param {Object} position
	 *
	 * @return {void}
	 */
	obj.handleCurrentLocationSuccess = ( position ) => {
		const coordinates = position.coords;
		const $searchBox = obj.$currentLocationButton.parents( obj.selectors.searchBox );
		const $latField = $searchBox.find( obj.selectors.currentLocationLat );
		const $lngField = $searchBox.find( obj.selectors.currentLocationLng );
		const $accuracyField = $searchBox.find( obj.selectors.currentLocationAccuracy );
		const $currentFlagField = $searchBox.find( obj.selectors.currentLocationFlag );
		const $searchField = $searchBox.find( obj.selectors.searchField );

		if ( ! obj.isValidCoordinates( coordinates ) ) {
			doAction( 'gk.maps.invalid_map_coordinates', coordinates );
			return;
		}

		$latField.val( coordinates.latitude );
		$lngField.val( coordinates.longitude );
		$accuracyField.val( coordinates.accuracy );
		$currentFlagField.val( 1 );

		obj.$currentLocationButton
			.addClass( className( obj.selectors.currentLocationButtonActive ) )
			.prop( 'title', obj.i18n.currentLocationLabel );


		if ( obj.$currentLocationButton.data( 'gkCurrentLocationInstantSearch' ) ) {
			obj.$currentLocationButton.parents( 'form' ).eq( 0 ).trigger( 'submit' );
		}

		obj.setupCurrentLocationFieldInteractions( $searchField );
	};

	/**
	 * Displays an error using the default gvMaps error for when invalid Coordinates were passed.
	 *
	 * @since 2.2
	 *
	 * @param {object} coordinates
	 */
	obj.throwInvalidCoordinatesError = ( coordinates ) => {
		gvMapsDisplayErrorNotice( obj.i18n.invalidGeolocationCoordinates );
	};

	/**
	 * Determines if a given object is valid as coordinates.
	 *
	 * @since 2.2
	 *
	 * @param {object} coordinates
	 *
	 * @return {boolean}
	 */
	obj.isValidCoordinates = ( coordinates ) => {
		if ( typeof coordinates !== 'object' ) {
			return false;
		}

		// This will also invalidate 0:0 coordinates on error.
		if ( ! coordinates.latitude && ! coordinates.longitude ) {
			return false;
		}

		return true;
	};

	obj.setupCurrentLocationFieldInteractions = ( $searchField ) => {
		$searchField.each( ( index, element ) => {
			const $field = $( element );
			const $currentButton = $field.siblings( obj.selectors.currentLocationButton );
			if ( ! $currentButton.is( obj.selectors.currentLocationButtonActive ) ) {
				return;
			}

			$field.data( 'currentPlaceholder', $field.attr( 'placeholder' ) );
			$field.attr( 'placeholder', obj.i18n.currentLocationLabel ).addClass( className( obj.selectors.searchFieldCurrentLocation ) );
			$field.val( '' );

			const removeContentsCallback = () => {
				$field.attr( 'placeholder', $field.data( 'currentPlaceholder' ) ).removeClass( className( obj.selectors.searchFieldCurrentLocation ) );
				$field.off( 'focusout.gkMapsSearchField' ).one( 'focusout.gkMapsSearchField', () => {
					$field.attr( 'placeholder', obj.i18n.currentLocationLabel ).addClass( className( obj.selectors.searchFieldCurrentLocation ) );
				} );
				$field.off( 'mousedown.gkMapsSearchField' ).one( 'mousedown.gkMapsSearchField', removeContentsCallback );

			};
			$field.off( 'mousedown.gkMapsSearchField' ).one( 'mousedown.gkMapsSearchField', removeContentsCallback );
		} );
	};

	/**
	 * Returns the options selected currently on the fields for radius search.
	 *
	 * @todo Determine if there a localized way of getting these fields.
	 *
	 * @since 2.2
	 *
	 * @return {{unit: string, accuracy: string, radius: number, long: number, lat: number, hasSearch: boolean}}
	 */
	obj.getCurrentLocationOptions = () => {
		const options = {
			long: parseFloat( $( obj.selectors.currentLocationLng ).val() ),
			lat: parseFloat( $( obj.selectors.currentLocationLat ).val() ),
			radius: parseFloat( $( obj.selectors.currentLocationRadius ).val() ),
			unit: $( obj.selectors.currentLocationUnit ).val(),
			accuracy: $( obj.selectors.currentLocationAccuracy ).val(),
			hasSearch: Boolean( Number( $( obj.selectors.currentLocationHasSearch ).val() ) ),
		};

		return options;
	};

	/**
	 * Throws a error to the user that we couldn't get the coordinates from the browser.
	 *
	 * @todo Error handling need to be visible to the user.
	 *
	 * @since 2.2
	 *
	 * @param {Object} error
	 *
	 * @return {void}
	 */
	obj.handleCurrentLocationError = ( error ) => {

		obj.$currentLocationButton.prop( 'title', obj.i18n.currentLocationTitle );

		confirm( obj.i18n.geolocationCurrentLocationError );

		console.warn( `ERROR(${error.code}): ${error.message}` );
	};

	/**
	 * Cache all the maps used to avoid re-selecting using jQuery.
	 *
	 * @type {{}}
	 */
	obj.$maps = {};

	/**
	 * Gets the jQuery object based on the map object from Google API.
	 *
	 * @since 2.2
	 *
	 * @param map
	 *
	 * @returns {jQuery}
	 */
	obj.getMap = ( map ) => {
		const mapDiv = map.getDiv();
		return $( mapDiv );
	};

	/**
	 * Determine if a given map can have a search included.
	 *
	 * @param {Element} map
	 *
	 * @returns {boolean}
	 */
	obj.shouldSearchOnMove = ( map ) => {
		const $map = obj.getMap( map );

		if ( ! $map.hasClass( 'gk-multi-entry-map' ) ) {
			return false;
		}

		return $map.find( '.gk-maps-search-checkbox' ).is( ':checked' );
	};

	/**
	 * Generates a UUID.
	 *
	 * @since 2.2
	 *
	 * @returns {string}
	 */
	obj.uuid = () => {
		let d = new Date().getTime();
		const uuid = 'xxxxxxxx-xxxx-xxxx-yxxx-xxxxxxxxxxxx'.replace( /[xy]/g, ( c ) => {
			const r = ( d + Math.random() * 16 ) % 16 | 0;
			d = Math.floor( d / 16 );
			return ( c == 'x' ? r : ( r & 0x3 | 0x8 ) ).toString( 16 );
		} );
		return uuid;
	};

	/**
	 * Includes a Search box on Google Maps UI with some actions.
	 *
	 * @param {jQuery} $maps
	 */
	obj.includeSearchOnMoveBox = ( $maps ) => {
		$maps.each( ( index, map ) => {
			const $map = $( map );

			// Checks that we have a valid map.
			if ( ! Map.isValid( $map ) ) {
				return;
			}

			// in case we cant find the search map.
			if ( ! $map.hasClass( 'gk-multi-entry-map' ) ) {
				return;
			}

			const data = Map.getData( $map )

			let hasBox = false;

			// If we already have the box for search on move drawn.
			data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].forEach( ( element ) => {
				if ( ! $( element ).hasClass( 'gk-maps-element-search' ) && ! $( element ).hasClass( 'gk-maps-element-redo-search' ) ) {
					return;
				}
				hasBox = true;
			} );

			if ( hasBox ) {
				return;
			}

			const toggleUuid = obj.uuid();
			const $toggleButton = $( `<label class="gk-maps-element gk-maps-element-search" for="gk-maps-search-${toggleUuid}"><input type="checkbox" class="gk-maps-search-checkbox" id="gk-maps-search-${toggleUuid}"> ${obj.i18n.searchOnMoveLabel}</label>` );
			const redoUuid = obj.uuid();
			const $redoSearch = $( `<button class="gk-maps-element gk-maps-element-redo-search" id="gk-maps-redo-search-${redoUuid}">${obj.i18n.redoSearchLabel}</button>` );

			$toggleButton.on( 'click', () => obj.toggleSearchOnMove( data.map ) );
			$redoSearch.on( 'click', () => {
				const onCompleteCallback = () => {
					data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].clear();
					data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].push( $toggleButton[ 0 ] );

					// Since the button is the same we need to re-enable once completed.
					$redoSearch.prop( 'disabled', false );
				};

				// Make sure you cannot click on this button while it loads.
				$redoSearch.prop( 'disabled', true );

				obj.triggerSearchOnMove( data.map, onCompleteCallback );
			} );
			data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].push( $toggleButton[ 0 ] );

			google.maps.event.addListenerOnce( data.map, 'idle', () => {
				google.maps.event.addListener( data.map, 'bounds_changed', () => {
					$map.addClass( 'gk-map-moved' );

					let boundsChanged = $map.data( 'gkBoundsChange' );
					if ( 'undefined' === typeof boundsChanged ) {
						boundsChanged = false;
					}

					if ( boundsChanged === true ) {
						return;
					}

					if ( ! obj.shouldSearchOnMove( data.map ) ) {
						data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].clear();
						data.map.controls[ google.maps.ControlPosition.TOP_RIGHT ].push( $redoSearch[ 0 ] );

						return;
					}

					$map.data( 'gkBoundsChange', true );

					setTimeout( () => {
						google.maps.event.addListenerOnce( data.map, 'idle', () => {
							$map.data( 'gkBoundsChange', false );
							obj.triggerSearchOnMove( data.map );
						} );
					}, 300 );
				} );
			} );
		} );
	};

	/**
	 * Handles the Search inside a set of bounds.
	 *
	 * @since 2.2
	 *
	 * @param {Element} map
	 * @param {function|null} onCompleteCallback
	 * @param {number} page
	 */
	obj.triggerSearchOnMove = ( map, onCompleteCallback = null, page = 1 ) => {
		const $map = obj.getMap( map );
		const mapData = $map.data( 'gkMap' );
		$map.addClass( className( Map.selectors.multipleEntryMapAvoidRebound ) );

		const bounds = map.getBounds();
		const ne = bounds.getNorthEast();
		const sw = bounds.getSouthWest();
		const mapNonce = $map.data( 'gkNonce' );

		const data = {
			bounds: {
				max_lat: ne.lat(),
				min_lat: sw.lat(),
				min_lng: ne.lng(),
				max_lng: sw.lng(),
			},
			pagenum: page > 0 ? page : 1,
			filter_geolocation: 1,
		};

		const ajaxArguments = {
			url: obj.getRestURL( map ),
			data: data,
			headers: {},
			accepts: 'json',
			dataType: 'json',
			method: 'GET',
			'async': true, // async is keyword
			beforeSend: ( jqXHR, settings ) => obj.ajaxBeforeSend( jqXHR, settings, map ),
			complete: ( jqXHR, textStatus ) => {
				obj.ajaxComplete( jqXHR, textStatus, map );

				if ( ! mapData.is_multi_entry_map ) {
					$map.removeClass( className( Map.selectors.multipleEntryMapAvoidRebound ) );
				}

				if ( onCompleteCallback ) {
					onCompleteCallback();
				}
			},
			success: ( data, textStatus, jqXHR ) => obj.ajaxSuccess( data, textStatus, jqXHR, $map ),
			error: ( jqXHR, settings ) => obj.ajaxError( jqXHR, settings, map ),
			context: $map,
		};

		if ( mapNonce ) {
			ajaxArguments.headers['X-WP-Nonce'] = mapNonce;
		}

		$.ajax( ajaxArguments );
	};

	/**
	 * Triggered on jQuery.ajax() beforeSend action, which we hook into to replace the contents of the modal with a
	 * loading HTML, as well as trigger a before and after hook so third-party developers can always extend all
	 * requests.
	 *
	 * @since 2.2
	 *
	 * @param  {jqXHR}       jqXHR    Request object
	 * @param  {Object} settings Settings that this request will be made with
	 *
	 * @return {void}
	 */
	obj.ajaxBeforeSend = ( jqXHR, settings, map ) => {
		obj.insertLoader( map );
		var $loader = $( document ).find( '.gk-maps-loader' );

		$( document ).trigger( 'beforeAjaxBeforeSend.Search/GravityMaps/GK', [ jqXHR, settings, map ] );

		if ( $loader.length ) {
			$loader.removeClass( 'gk-maps-loader-hidden' );
		}

		$( document ).trigger( 'afterAjaxBeforeSend.Search/GravityMaps/GK', [ jqXHR, settings, map ] );
	};

	obj.insertLoader = ( map ) => {

	};

	/**
	 * Triggered on jQuery.ajax() complete action, which we hook into to reset appropriate variables and remove the
	 * loading HTML, as well as trigger a before and after hook so third-party developers can always extend all requests
	 *
	 * @since 2.2
	 *
	 * @param  {jqXHR}  jqXHR       Request object
	 * @param  {String} textStatus Status for the request
	 *
	 * @return {void}
	 */
	obj.ajaxComplete = ( jqXHR, textStatus, map ) => {
		var $loader = $( document ).find( '.gk-maps-loader' );

		$( document ).trigger( 'beforeAjaxComplete.Search/GravityMaps/GK', [ jqXHR, textStatus, map ] );

		if ( $loader.length ) {
			$loader.addClass( 'gk-maps-loader-hidden' );
		}

		$( document ).trigger( 'afterAjaxComplete.Search/GravityMaps/GK', [ jqXHR, textStatus, map ] );

		// Reset the current ajax request on the manager object.
		obj.currentAjaxRequest = null;
	};

	/**
	 * Triggered on jQuery.ajax() success action, which we hook into to replace the contents of the modal, as well as
	 * trigger a before and after hook so third-party developers can always extend all requests
	 *
	 * @since 2.2
	 *
	 * @param  {String} html       HTML sent from the AJAX request.
	 * @param  {String} textStatus Status for the request.
	 * @param  {jqXHR}  jqXHR      Request object.
	 * @param  {jQuery} $map	   jQuery object of the map.
	 *
	 * @return {void}
	 */
	obj.ajaxSuccess = ( html, textStatus, jqXHR, $map ) => {
		$( document ).trigger( 'beforeAjaxSuccess.Search/GravityMaps/GK', [ html, textStatus, jqXHR, $map ] );

		const originalData = Map.getData( $map );
		const $response = $( html );

		// Remove the radius search and marker if they exist.
		if ( originalData.radiusSearch && originalData.radiusMarker ) {
			originalData.radiusSearch.setMap( null );
			originalData.radiusMarker.setMap( null );
		}

		const $responseViewWrapper = $response.filter( View.selectors.wrapper );
		debug.log( 'View Wrapper [response]: ', $responseViewWrapper );

		const $viewWrapper = View.getWrapper( $map );
		debug.log( 'View Wrapper [existing]: ', $viewWrapper );

		const $countMeta = $response.filter( 'meta[http-equiv="X-Item-Count"]' );
		const responseHasNoResults = $countMeta.length && parseInt( $countMeta.attr( 'content' ), 10 ) === 0
		const observer = Map.getObserver();

		// Unobserve all maps in the view wrapper.
		Map.getMaps( $viewWrapper ).each( ( index, map ) => observer.unobserve( map ) );

		if ( responseHasNoResults ) {
			debug.info( `No results were found.` );
			$responseViewWrapper.addClass( className( View.selectors.noResultsFlag ) );
		} else {
			debug.info( `Results were found` );
		}

		// Get this existing container, so we can replace it later, needs to happen before we move the maps.
		const $existingContainer = $viewWrapper.find( View.selectors.container );

		// Before any replacement, we need to make sure we don't replace the maps.
		const detachedMapsCallbacks = obj.getDetachedMapsCallback( $viewWrapper );

		// Move the permanent components to the new view wrapper.
		const $rebuiltResponse = obj.movePermanentComponents( $viewWrapper, $responseViewWrapper );

		const $replacementContainer = $rebuiltResponse.find( View.selectors.container );
		debug.log( 'View Container [response]: ', $replacementContainer );
		debug.log( 'View Container [existing]: ', $existingContainer );

		// Replace the container with the new one.
		$existingContainer.replaceWith( $replacementContainer );

		$viewWrapper.find( Map.selectors.multipleEntryMap ).each( ( index, map ) => {
			if ( ! detachedMapsCallbacks.length ) {
				return;
			}

			if ( ! detachedMapsCallbacks[ index ] ) {
				return;
			}

			detachedMapsCallbacks[ index ]( $( map ) );
		} );



		// Re-observe all maps in the view wrapper.
		Map.bindObserverToMaps( $viewWrapper );

		$( document ).trigger( 'afterAjaxSuccess.Search/GravityMaps/GK', [ html, textStatus, jqXHR, $map ] );
	};

	obj.getViewType = ( $container ) => {
		if ( $container.hasClass( 'gv-map-container' ) ) {
			return 'map';
		}

		if ( $container.hasClass( 'gv-list-container' ) ) {
			return 'list'
		}

		if ( $container.hasClass( 'gv-table-container' ) ) {
			return 'table'
		}

		return 'other'
	};

	/**
	 *
	 * @param $wrapper
	 * @return {function[]}
	 */
	obj.getDetachedMapsCallback = ( $wrapper ) => {
		const callbacks = [];
		Map.getMaps( $wrapper, true ).each( ( index, mapEl ) => {
			const $map = $( mapEl ).detach();
			const mapData = Map.getData( $map );

			callbacks.push( ( $mapToReplace ) => {
				$map.removeClass( className( Map.selectors.generatedMapMarkers ) );

				// Hide all markers for now.
				mapData.markers.forEach( marker => {
					marker.set( 'gkVisible', false );
					marker.setVisible( false );
				} );

				const receivedData = $mapToReplace.data( 'gkMap' );

				// Overwrite the markers data with the new Markers data.
				mapData.markers_data = receivedData.markers_data;

				$mapToReplace.data( 'gkMap', mapData );

				$mapToReplace.replaceWith( $map );

				if ( mapData.is_multi_entry_map ) {
					$map.addClass( className( Map.selectors.multipleEntryMapAvoidRebound ) );
					Marker.processMap( $map );
				}
			} );
		} );

		return callbacks;
	};

	/**
	 * Look at the given container and make sure any Maps that already exist don't get regenerated.
	 *
	 * @since 3.1.3
	 *
	 * @param {jQuery} $existingWrapper
	 * @param {jQuery} $responseWrapper
	 */
	obj.movePermanentComponents = ( $existingWrapper, $responseWrapper ) => {
		$existingWrapper.find( View.selectors.header ).replaceWith( $responseWrapper.find( View.selectors.header ) );
		$existingWrapper.find( View.selectors.footer ).replaceWith( $responseWrapper.find( View.selectors.footer ) );

		$responseWrapper.find( View.selectors.searchForm ).replaceWith( $existingWrapper.find( View.selectors.searchForm ) );

		return $responseWrapper;
	}

	/**
	 * Triggered on jQuery.ajax() error action, which we hook into to close the modal for now, as well as
	 * trigger a before and after hook so third-party developers can always extend all requests
	 *
	 * @since 1.0
	 *
	 * @param  {jqXHR}  jqXHR    Request object.
	 * @param  {Object} settings Settings that this request was made with.
	 *
	 * @return {void}
	 */
	obj.ajaxError = ( jqXHR, settings, map ) => {
		$( document ).trigger( 'beforeAjaxError.Search/GravityMaps/GK', [ jqXHR, settings, map ] );

		/**
		 * @todo  we need to handle errors here
		 */

		$( document ).trigger( 'afterAjaxError.Search/GravityMaps/GK', [ jqXHR, settings, map ] );
	};

	/**
	 * Get the REST API based on the PHP sent variables.
	 *
	 * @since 2.2
	 *
	 * @param {Element} map
	 *
	 * @returns {string}
	 */
	obj.getRestURL = ( map ) => {
		const $map = obj.getMap( map );
		const data = $map.data( 'gkMap' );
		let limit = 0;
		if ( data.page_size ) {
			limit = data.page_size;
		}

		return obj.i18n.searchRestEndpoint.replace( '{view_id}', data.view_id ).replace( '{limit}', limit );
	};

	obj.toggleSearchOnMove = ( map ) => {
		const $map = obj.getMap( map );
		const shouldDisplay = ! $map.find( '.gk-maps-element-search' ).find( '.gk-maps-search-checkbox' ).is( ':checked' );

		if ( ! shouldDisplay ) {
			$map.addClass( 'gk-multi-entry-map-on-move' );
		} else {
			$map.removeClass( 'gk-multi-entry-map-on-move' );
		}

		const radiusSearch = $map.data( 'gkRadiusSearch' );
		const radiusMarker = $map.data( 'gkRadiusMarker' );

		if ( radiusSearch ) {
			radiusSearch.setVisible( shouldDisplay );
		}
		if ( radiusMarker ) {
			radiusMarker.setVisible( shouldDisplay );
		}
	};

	/**
	 * Handles adding the bounds circle and a center marker.
	 *
	 * This method will wait until the map is idle to draw.
	 *
	 * @since 2.2
	 *
	 * @param {jQuery} $map
	 *
	 * @return {void}
	 */
	obj.drawRadiusOnMap = ( $map ) => {
		const options = obj.getCurrentLocationOptions();

		// Only do anything if we have a Geolocation Search.
		if ( ! options.hasSearch ) {
			return;
		}

		if ( ! $map.hasClass( 'gk-multi-entry-map' ) ) {
			return;
		}

		if ( $map.hasClass( 'gk-multi-entry-map-done' ) ) {
			return;
		}

		const data = Map.getData( $map );

		// When dealing with Null Island don't do anything, we are using == intentionally instead of ===.
		if ( options.lat == 0 && options.long == 0 ) {
			return;
		}

		$map.addClass( 'gk-multi-entry-map-done' );

		obj.includeSearchOnMoveBox( $map );

		google.maps.event.addListenerOnce( data.map, 'idle', () => {
			$( document ).trigger( 'beforeRadiusDraw.GravityKit/Maps', [ data.map ] );

			if ( 'mi' === options.unit ) {
				options.radius = mileToKm( options.radius );
			}

			const radiusSearch = new google.maps.Circle( {
				strokeColor: "#FF0000",
				strokeOpacity: 0.6,
				strokeWeight: 1,
				fillColor: "#FF0000",
				fillOpacity: 0.17,
				map: data.map,
				center: {
					lat: options.lat,
					lng: options.long,
				},
				radius: options.radius * 1000,
				// draggable: true,
				// editable: true,
			} );

			data.radiusSearch = radiusSearch;
			const radiusMarker = new google.maps.Marker( {
				position: {
					lat: options.lat,
					lng: options.long,
				},
				// draggable: true,
				icon: {
					url: obj.i18n.geolocationFenceCenterImage,
					size: new google.maps.Size( 7, 7 ),
					anchor: new google.maps.Point( 4, 4 )
				},
				map: data.map
			} );
			data.radiusMarker = radiusMarker;

			radiusSearch.bindTo( 'center', radiusMarker, 'position' );

			$map.data( 'gkMap', data );

			setTimeout( () => {
				data.map.fitBounds( radiusSearch.getBounds(), 10 );
			}, 300 );

			$( document ).trigger( 'afterRadiusDraw.GravityKit/Maps', [ data.map, radiusSearch ] );
		} );
	};

	/**
	 * Triggers the automatic radius search.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map Which map will be used.
	 *
	 * @return {void}
	 */
	obj.triggerAutomaticRadiusSearch = ( $map ) => {
		if ( ! $map.is( Map.selectors.multipleEntryMap ) ) {
			return;
		}

		const settings = Map.getSettings( $map );

		if ( ! settings?.automatic_radius_search ) {
			return;
		}

		if ( $map.hasClass( 'gk-multi-entry-map-done' ) ) {
			return;
		}

		const $viewWrapper = $map.parents( Map.selectors.viewWrapper ).eq( 0 );
		const $currentLocationField = $viewWrapper.find( obj.selectors.currentLocationButton ).eq( 0 );

		$currentLocationField.data( 'gkCurrentLocationInstantSearch', true );

		$currentLocationField.trigger( 'click' );
	};

	/**
	 * Handles the click on the pagination links.
	 *
	 * @since 3.1.3
	 *
	 * @param {Event} event DOM Event.
	 *
	 * @return {void}
	 */
	obj.onNavigationClick = ( event ) => {
		const $target = $( event.currentTarget );
		if ( ! $target.is( '.page-numbers' ) ) {
			return;
		}

		const href = $target.attr( 'href' );
		if ( ! href ) {
			return;
		}

		const $map = $target.closest( View.selectors.wrapper ).find( `${Map.selectors.mapCanvas}${Map.selectors.multipleEntryMap}` );

		// Couldn't find a map, don't take over other pagination.
		if ( ! Map.isValid( $map ) ) {
			return;
		}

		const { map } = Map.getData( $map );
		const settings = Map.getSettings( $map );

		// Use `gk/gravitymaps/map/control-pagination` to override the View's default pagination behavior.
		const paginationControl = !!( settings.pagination_control ?? false );

		if ( ! paginationControl ) {
			return;
		}

		const url = new URL( href );
		const page = url.searchParams.get( 'pagenum' ) || 1;

		event.preventDefault();
		obj.triggerSearchOnMove( map, null, page );
	};

	/**
	 * Triggers when Ready of the document.
	 *
	 * @since 2.2
	 *
	 * @return {void}
	 */
	obj.ready = () => {
		obj.setupCurrentLocationFieldInteractions( $( obj.selectors.searchField ) );
	};

	$( document ).on( 'click', obj.selectors.currentLocationButton, obj.onCurrentLocationClick );
	$( document ).on( 'click', View.selectors.paginationLink, obj.onNavigationClick );

	$( document ).ready( obj.ready );

	/**
	 * It's not guaranteed, but it actually works for most cases.
	 * Hopefully on version 3.1 we can remove this, since we will use async loading.
	 */
	addAction( 'gk.maps.controllers.marker.after_process_map', 'gravitykit/maps', obj.drawRadiusOnMap );
	addAction( 'gk.maps.controllers.marker.after_process_map', 'gravitykit/maps', obj.includeSearchOnMoveBox );
	addAction( 'gk.maps.controllers.map.invalid_map_coordinates', 'gravitykit/maps', obj.throwInvalidCoordinatesError );
	addAction( 'gk.maps.controllers.map.after_create_map' , 'gravitykit/maps', obj.triggerAutomaticRadiusSearch );

} )( jQuery, {}, window.GravityKit.GravityMaps );

