/* globals GravityKit, jQuery, ajaxurl, Qs, google, window */
/**
 * Setups the main variable for this file.
 *
 * @since 3.1.0
 *
 * @type   {Object}
 */
GravityKit.GravityMaps.Services = GravityKit.GravityMaps.Services || {};
GravityKit.GravityMaps.Services.GoogleMaps = {};

/**
 * This is a global function required for handling Google Maps API errors.
 *
 * It is a simple by-pass to the `GravityKit.GravityMaps.Controllers.Map.triggerError` method.
 *
 * @see https://developers.google.com/maps/documentation/javascript/events#auth-errors
 *
 * @since 3.1.0
 *
 * @return {void}
 */
function gm_authFailure() {
	const Map = GravityKit.GravityMaps.Controllers.Map;
	const GoogleMaps = GravityKit.GravityMaps.Services.GoogleMaps;
	Map.triggerError( GoogleMaps.i18n.google_maps_api_error, Map.getMaps() );
}

( ( $, obj ) => {
    "use strict";

	const { Map, Marker } = GravityKit.GravityMaps.Controllers;
	const {
		doAction,
		addAction,
		applyFilters,
		addFilter,
	} = GravityKit.GravityMaps.hooks;
	const className = GravityKit.GravityMaps.Utils.className;

	obj.i18n = window.gk_maps_services_google_maps_i18n || {};

	/**
	 * Attach all the hooks into the Actions and Filters used by the maps system.
	 *
	 * @since 3.1.0
	 */
	obj.hook = () => {
		addAction( 'gk.maps.controllers.marker.after_process_map', 'gravitykit/maps', obj.triggerIdle );
		addAction( 'gk.maps.controllers.marker.after_process_map', 'gravitykit/maps', obj.processCluster );
		addAction( 'gk.maps.controllers.marker.after_process_map', 'gravitykit/maps', obj.processSpider );
		addAction( 'gk.maps.controllers.marker.on_view_entry_mouse_enter', 'gravitykit/maps', obj.modifyAnimationsForMarkers );
		addAction( 'gk.maps.controllers.marker.create', 'gravitykit/maps', obj.createMarker );
		addFilter( 'gk.maps.controllers.map.get_settings', 'gravitykit/maps', obj.modifyMapSettings );
		addAction( 'gk.maps.controllers.map.create_map', 'gravitykit/maps', obj.createMap );
		addAction( 'gk.maps.controllers.map.set_zoom', 'gravitykit/maps', obj.setZoom );
		addAction( 'gk.maps.controllers.map.set_center', 'gravitykit/maps', obj.setCenter );
	};

	/**
	 * Triggers when the window finishes loading.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.load = () => {

	};

	/**
	 * Given a setTimeout, it will trigger the `idle` event on the map passed in.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map Which map to trigger the event on.
	 *
	 * @return {void}
	 */
	obj.triggerIdle = ( $map ) => {
		const data = Map.getData( $map );

		// It's important to have the timout here to avoid race conditions around clustering.
		setTimeout( () => google.maps.event.trigger( data.map, 'idle' ), 50 );
	};

	/**
	 * Modify the settings for the map when handling Google Maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {Object} settings
	 * @param {jQuery} $map
	 *
	 * @returns {Object}
	 */
	obj.modifyMapSettings = ( settings, $map ) => {
		settings.MapOptions.zoom = parseInt( settings.MapOptions.zoom, 10 );
		settings.MapOptions.mapTypeId = google.maps.MapTypeId[ settings.MapOptions.mapTypeId ];

		if (
			settings.MapOptions.hasOwnProperty( 'zoomControl' )
			&& true === settings.MapOptions.zoomControl
			&& settings.MapOptions.zoomControlOptions
			&& settings.MapOptions.zoomControlOptions.hasOwnProperty( 'position' )
		) {

			/**
			 * Convert map type setting into google.maps object
			 *
			 * With style and position keys.
			 *
			 * For the position value, see [Google V3 API grid of positions](https://developers.google.com/maps/documentation/javascript/reference#ControlPosition)
			 * Options include: BOTTOM_CENTER, BOTTOM_LEFT, BOTTOM_RIGHT, LEFT_BOTTOM, LEFT_CENTER, LEFT_TOP, RIGHT_BOTTOM, RIGHT_CENTER, RIGHT_TOP, TOP_CENTER, TOP_LEFT, TOP_RIGHT
			 */
			settings.MapOptions.zoomControlOptions = {
				'position': google.maps.ControlPosition[ settings.MapOptions.zoomControlOptions.position ],
			};
		}

		return settings;
	};

	/**
	 * Handles the creation of the map for Google Maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 * @param {function} markGeneratedMap The callback to call when the map is created.
	 */
	obj.createMap = ( $map, markGeneratedMap ) => {
		const data = Map.getData( $map );
		const settings = Map.getSettings( $map );

		if ( ! settings.api_key ) {
			Map.triggerError( settings.google_maps_api_key_not_found, $map );
			return;
		} else if ( 'undefined' === typeof window.google ) {
			Map.triggerError( settings.google_maps_script_not_loaded, $map );
			return;
		} else if ( ! settings.can_use_rest ) {
			Map.triggerError( settings.cannot_use_rest_error, $map );
			return;
		} else if ( settings.address_field_missing ) {
			Map.triggerError( settings.address_field_missing, $map );
			return;
		} else if ( settings.hide_until_searched !== '1' && ! data.markers_data.length && ! settings.is_search ) {
			Map.triggerError( settings.entries_missing_coordinates, $map );
			return;
		}

		const options = $.extend( {}, settings.MapOptions, {
			_entryId: data.entry_id,
			_element: $map,
			_bounds: new google.maps.LatLngBounds()
		} );

		data.map = new google.maps.Map( $map[0], options );

		data.infoWindow = new google.maps.InfoWindow( {
			content: '',
			maxWidth: parseInt( settings.infowindow.max_width, 10 ),
		} );

		Map.updateData( $map, data );

		if ( 1 === settings.layers.bicycling ) {
			const bicyclingLayer = new google.maps.BicyclingLayer();
			bicyclingLayer.setMap( data.map );
		}
		if ( 1 === settings.layers.transit ) {
			const transitLayer = new google.maps.TransitLayer();
			transitLayer.setMap( data.map );
		}
		if ( 1 === settings.layers.traffic ) {
			const trafficLayer = new google.maps.TrafficLayer();
			trafficLayer.setMap( data.map );
		}

		Map.setZoom( $map );
		Map.setCenter( $map );

		/**
		 * Wait until the map is stopped moving before tagging it as generated and triggering the after_init_map hook.
		 */
		google.maps.event.addListenerOnce( data.map, 'idle', markGeneratedMap );
	};

	/**
	 * Create a marker for Google Maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {object} markerData
	 *
	 * @return {void}
	 */
	obj.createMarker = ( $map, markerData ) => {
		const settings = Map.getSettings( $map );
		const data = Map.getData( $map );

		if ( ! data.markers ) {
			data.markers = [];
		}

		let marker;

		if ( data.is_multi_entry_map ) {
			const index = Marker.exists( $map, markerData );

			if ( false !== index && data.markers[ index ] ) {
				marker = data.markers[ index ];
				marker.setMap( data.map );
				marker.set( 'gkVisible', true );
				marker.setVisible( true );
			}
		}

		if ( ! marker ) {
			const geo = new google.maps.LatLng( markerData.lat, markerData.long );
			const icon = {
				url: settings.icon,
			};

			if ( markerData.icon ) {

				if ( markerData.icon.url && markerData.icon.url.length ) {
					icon.url = markerData.icon.url;
				}

				if ( markerData.icon.scaledSize && markerData.icon.scaledSize.length === 2 ) {
					icon.size = new google.maps.Size( markerData.icon.scaledSize[ 0 ], markerData.icon.scaledSize[ 1 ] );
				}

				if ( markerData.icon.origin && markerData.icon.origin.length === 2 ) {
					icon.origin = new google.maps.Point( markerData.icon.origin[ 0 ], markerData.icon.origin[ 1 ] );
				}

				if ( markerData.icon.anchor && markerData.icon.anchor.length === 2 ) {
					icon.anchor = new google.maps.Point( markerData.icon.anchor[ 0 ], markerData.icon.anchor[ 1 ] );
				}

				if ( markerData.icon.scaledSize && markerData.icon.scaledSize.length === 2 ) {
					icon.scaledSize = new google.maps.Size( markerData.icon.scaledSize[ 0 ], markerData.icon.scaledSize[ 1 ] );
				}
			}

			/**
			 * Enables the ability to filter the options used to build a marker.
			 *
			 * @since 2.2
			 *
			 * @param {jQuery} $map
			 * @param {object} data
			 * @param {object} markerData
			 */
			const markerOptions = applyFilters( 'gk.maps.services.google_maps.create_marker_options', {
				map: data.map,
				icon: icon,
				url: markerData.url,
				position: geo,
				gkVisible: true,
				entryId: markerData.entry_id,
				content: markerData.content,
			}, $map, data, markerData );

			marker = new google.maps.Marker( markerOptions );

			data.markers.push( marker );

			Map.updateData( $map, data );
			obj.bindMarkerEvents( $map, marker );
		}

		// Extend map bounds using marker position
		data.map._bounds.extend( marker.position );

		if ( ! $map.hasClass( className( Map.selectors.multipleEntryMapAvoidRebound ) ) ) {
			data.map.fitBounds( data.map._bounds );
		}

		// Add this particular marker to all the search maps in the container.
		if ( ! data.is_multi_entry_map ) {
			Map.getSearchMaps( data.$container, true ).each( ( index, searchMap ) => {
				const $searchMap = $( searchMap );
				const searchMapData = $searchMap.data( 'gkMap' );

				if ( ! Marker.exists( $searchMap, marker ) ) {
					searchMapData.markers.concat( marker );
					searchMapData.markers_data.concat( markerData );
				}

				$searchMap.data( 'gkMap', searchMapData );
			} );
		}
	};

	/**
	 * Open infowindow or go to entry link when marker has been clicked
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map   jQuery object of the map container.
	 * @param {object} marker google.maps.Marker Google maps marker object
	 * @param {string} marker.content Infowindow markup string
	 * @param {object} marker.map A google.maps.Map object
	 * @param {string} marker.url Full URL to the marker's single entry page
	 * @param {object} marker.position A google.maps.LatLng object
	 * @param {int|string} marker.entryId Entry ID # or slug
	 *
	 * @return {void}
	 */
	obj.onMarkerClick = ( $map, marker ) => {
		const data = Map.getData( $map );
		const settings = Map.getSettings( $map );
		const content = obj.infoWindowGetContent( $map, marker );

		// Open infowindow if content is set
		if ( content ) {
			data.infoWindow.setContent( content );
			data.infoWindow.open( marker.map, marker );

			return;
		}

		// Go to entry link
		data.infoWindow.close();
		window.open( marker.url, settings.marker_link_target );
	};

	/**
	 * Modify the animations for the markers related to Google Maps Service.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {Object} markers
	 */
	obj.modifyAnimationsForMarkers = ( $map, markers ) => {
		markers.forEach( marker => {
			if ( marker.animating ) {
				return;
			}

			marker.setAnimation( google.maps.Animation.BOUNCE );

			// stop the animation after one bounce
			setTimeout( () => marker.setAnimation( null ),750 );
		} );
	};

	/**
	 * Check if the infowindow content is empty and if so add a link to the single entry (by default)
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map
	 * @param {object} marker
	 *
	 * @returns {string} Prepared Infowindow HTML, with empty image tags removed and default text added to empty links
	 */
	obj.infoWindowGetContent = ( $map, marker ) => {
		const data = Map.getData( $map );
		const settings = Map.getSettings( $map );

		/**
		 * Do we accept empty infowindows?
		 * @see \GravityKit\GravityMaps\Render_Map::parse_map_options
		 */
		if ( ! settings.infowindow.no_empty ) {
			return marker.content;
		}

		const $content = $( marker.content );

		$content
			.find( 'img[src=""]' ).remove() // Remove empty images
			.end()
			.addClass( function () {
				if ( 0 === $content.find( 'img' ).length ) {
					return 'gv-infowindow-no-image';
				}
			} )
			.find( 'a.gv-infowindow-entry-link:not([allow-empty]):empty' ).text( settings.infowindow.empty_text ); // Empty links get some text, unless "allow-empty" attribute is set

		return $content.prop( 'outerHTML' );
	};

	/**
	 * Add event listeners to Google Markers.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map   jQuery object of the map container.
	 * @param {object} marker google.maps.Marker
	 */
	obj.bindMarkerEvents = ( $map, marker ) => {
		if ( Map.isSingleEntry( $map ) ) {
			return;
		}
		const data = Map.getData( $map );

		google.maps.event.addListenerOnce( data.map, 'idle', () => {
			// The marker has been clicked.
			google.maps.event.addListener( marker, 'spider_click', () => obj.onMarkerClick( $map, marker ) );
			google.maps.event.addListener( marker, 'click', () => obj.onMarkerClick( $map, marker ) );

			// on Mouse over
			google.maps.event.addListener( marker, 'mouseover', obj.onMarkerMouseOver( $map, marker ) );

			// on mouseout
			google.maps.event.addListener( marker, 'mouseout', obj.onMarkerMouseOut( $map, marker ) );

			// Close infowindow when clicking the map
			google.maps.event.addListener( data.map, 'click', () => data.infoWindow.close() );

			/**
			 * @action gk.maps.services.google_maps.bind_marker_events Enables the ability to add events bound to a marker.
			 *
			 * @since 3.1.0
			 *
			 * @param {jQuery} $map  jQuery object of the map container.
			 * @param {object} marker google.maps.Marker
			 */
			doAction( 'gk.maps.services.google_maps.bind_marker_events', $map, marker );
		} );
	};

	/**
	 * Highlights the assigned entry on mouse over a Marker
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map   jQuery object of the map container.
	 * @param {object} marker google.maps.Marker.
	 *
	 * @returns {Function}
	 */
	obj.onMarkerMouseOver = ( $map, marker ) => () => $( '#gv_map_' + marker.entryId ).addClass( 'gv-highlight-entry' );

	/**
	 * Remove the highlight of the assigned entry on mouse out a Marker
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map   jQuery object of the map container.
	 * @param {object} marker google.maps.Marker.
	 *
	 * @returns {Function}
	 */
	obj.onMarkerMouseOut = ( $map, marker ) => () => $( '#gv_map_' + marker.entryId ).removeClass( 'gv-highlight-entry' );

	/**
	 * Fixes issue where fitBounds() zooms in too far after adding markers
	 *
	 * @see http://stackoverflow.com/a/4065006/480856
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.setZoom = ( $map ) => {
		const data = Map.getData( $map );

		if ( typeof data.map === 'undefined' ) {
			return;
		}

		google.maps.event.addListenerOnce( data.map, 'idle', () => {
			// Don't zoom for multiple entry maps.
			if ( Map.isMultiEntry( $map ) ) {
				return;
			}

			const settings = Map.getSettings( $map );

			if ( data.map.getZoom() > settings.MapOptions.zoom ) {
				data.map.setZoom( settings.MapOptions.zoom );
			}
		} );
	};

	/**
	 * Centers the map properly.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.setCenter = (  $map ) => {
		const data = Map.getData( $map );

		if ( typeof data.map === 'undefined' ) {
			return;
		}

		google.maps.event.addListenerOnce( data.map, 'idle', () => {
			// Don't zoom for multiple entry maps.
			if ( Map.isMultiEntry( $map ) ) {
				return;
			}

			const settings = Map.getSettings( $map );

			if ( 'undefined' !== typeof settings.MapOptions.center && settings.MapOptions.center.lat && settings.MapOptions.center.lng ) {
				data.map.setCenter( settings.MapOptions.center );
			} else if ( $map.hasClass( 'gk-no-markers' ) ) {
				data.map.setCenter( { lat: 0, lng: 0 } );
			}
		} );
	};

	/**
	 * Sets up the Clustering of a given set of maps.
	 *
	 * Note: This is not native to Google Maps, it uses a Google Supported API but not native.
	 *
	 * @since 3.1.0
	 *
	 * @link https://developers.google.com/maps/documentation/javascript/marker-clustering
	 *
	 * @param {jQuery} $map
	 */
	obj.processCluster = ( $map ) => {
		$map = $( $map );
		if ( Map.isSingleEntry( $map ) ) {
			return;
		}
		const data = Map.getData( $map );

		if ( ! data || ! data.markers ) {
			return;
		}

		const settings = Map.getSettings( $map );

		// Cluster markers if option is set and map contains markers
		if ( ! settings.MapOptions.markerClustering || ! data.markers ) {
			return;
		}

		// remove if this particular map already has a cluster.
		if ( data.cluster ) {
			data.cluster.clearMarkers();
		}

		google.maps.event.addListenerOnce( data.map, 'idle', () => {
			data.cluster = new MarkerClusterer( data.map, data.markers, {
				imagePath: settings.markerClusterIconPath,
				maxZoom: settings.MapOptions.markerClusteringMaxZoom || settings.MapOptions.zoom,
			} );

			Map.updateData( $map, data );
		} );
	};

	/**
	 * Configures the Spider, which handles when too many markers are on top of each other.
	 *
	 * Note: This uses a third party API, not default to Google API.
	 *
	 * @since 3.1.0
	 *
	 * @link https://github.com/jawj/OverlappingMarkerSpiderfier
	 *
	 * @param {jQuery} $map
	 *
	 * @return {void}
	 */
	obj.processSpider = ( $map ) => {
		// It's important to have the timout here to avoid race conditions around clustering.
		setTimeout( () => {
			const data = Map.getData( $map );
			google.maps.event.addListener( data.map, 'idle', () => {
				// Spiderfy markers
				const oms = new OverlappingMarkerSpiderfier( data.map, {
					markersWontMove: true,
					markersWontHide: true,
					keepSpiderfied: true,
				} );

				data.markers.forEach( marker => oms.addMarker( marker ) );
			} );
		}, 55 );
	};

	/**
	 * Hooking needs to happen as early as possible.
	 */
	obj.hook();

	$( window ).on( 'load', obj.load );
} )( jQuery, GravityKit.GravityMaps.Services.GoogleMaps );
