<?php
namespace GravityKit\GravityMaps;

/**
 * Class Assets
 *
 * @since   3.1.0
 *
 * @package GravityKit\GravityMaps
 */
class Assets extends AbstractSingleton {

	/**
	 * Loader object.
	 *
	 * @since 3.1.0
	 *
	 * @var Loader
	 */
	protected $loader;

	/**
	 * Boots the assets class.
	 *
	 * @since 3.1.0
	 *
	 */
	public function register(): void {
	}

	/**
	 * Registers assets to WordPress.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	public function register_to_wp(): void {
		// Avoid doing it twice.
		if ( wp_script_is( 'gk-maps-base', 'registered' ) ) {
			return;
		}

		wp_register_script(
			'gk-maps-base',
			$this->get_url( 'assets/js/base.js' ),
			[
				'jquery',
				'wp-hooks',
			],
			$this->get_version()
		);

		wp_register_script(
			'gk-maps-utils',
			$this->get_url( 'assets/js/gk-maps-utils.js' ),
			[],
			$this->get_version()
		);

		wp_register_script(
			'gk-maps-controllers-map',
			$this->get_url( '/assets/js/controllers/map.js' ),
			[
				'jquery',
				'gk-maps-base',
				'gk-maps-utils',
			],
			$this->get_version(),
			true
		);

		wp_register_script(
			'gk-maps-controllers-view',
			$this->get_url( '/assets/js/controllers/view.js' ),
			[
				'jquery',
				'gk-maps-base',
				'gk-maps-utils',
				'gk-maps-controllers-map',
			],
			$this->get_version(),
			true
		);

		wp_register_script(
			'gk-maps-controllers-marker',
			$this->get_url( '/assets/js/controllers/marker.js' ),
			[
				'jquery',
				'gk-maps-base',
				'gk-maps-utils',
				'gk-maps-controllers-map',
			],
			$this->get_version(),
			true
		);
	}

	/**
	 * Stores the loader object.
	 *
	 * @since 3.1.0
	 *
	 * @param Loader $loader
	 *
	 */
	public function set_loader( Loader $loader ): void {
		$this->loader = $loader;
	}

	/**
	 * Returns the loader object.
	 *
	 * @since 3.1.0
	 *
	 * @return Loader
	 */
	public function get_loader(): Loader {
		return $this->loader;
	}

	/**
	 * Get the version of the plugin.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_version(): string {
		return $this->get_loader()->plugin_version;
	}

	/**
	 * Determine if we need to minify the assets.
	 *
	 * @since 3.1.0
	 *
	 * @return bool
	 */
	protected function should_minify(): bool {
		return ! ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG );
	}

	/**
	 * Given the path to a given asset, determine it's final URL.
	 *
	 * @since 3.1.0
	 *
	 * @param string    $asset  Path to the asset, it should be relative to the root of the plugin.
	 * @param bool|null $minify Whether we want to minify the asset, if null, it will be determined automatically.
	 *
	 * @return string
	 */
	public function get_url( string $asset, ?bool $minify = null ): string {
		if ( is_null( $minify ) ) {
			$minify = $this->should_minify();
		}

		$url = $this->get_loader()->url . ltrim( $asset, '/' );

		if ( $minify ) {
			$url = $this->modify_with_minified_suffix( $url );
		}

		return $url;
	}

	/**
	 * Given the URL to a given asset, determine it's final URL with the minified suffix.
	 *
	 * @since 3.1.0
	 *
	 * @param string $asset URL of the Asset in question.
	 *
	 * @return string
	 */
	protected function modify_with_minified_suffix( string $asset ): string {
		// There is a `.min.` in this file name, so we don't need to do anything.
		if ( stripos( $asset, '.min.' ) !== false ) {
			return $asset;
		}

		// Only replace for these two types.
		if ( substr( $asset, -3, 3 ) === '.js' ) {
			$asset = substr_replace( $asset, '.min', -3, 0 );
		} elseif ( substr( $asset, -4, 4 ) === '.css' ) {
			$asset = substr_replace( $asset, '.min', -4, 0 );
		}

		return $asset;
	}
}