<?php

namespace GravityKit\GravityMaps;

use Exception;
use GravityKit\GravityMaps\Geocoder\HttpAdapter\HttpAdapterInterface;

/**
 * HTTP adapter for geocoder
 *
 * @since 1.2
 */
class HTTP_Adapter implements HttpAdapterInterface {
	protected $headers = [];

	/**
	 * {@inheritDoc}
	 */
	public function getContent( $url ) {
		$request_settings = array(
			'user-agent' => 'GravityView Maps', // OpenStreetMap requests unique User-Agent ID
			'sslverify'  => false,
		);

		if ( $this->hasHeaders() ) {
			$request_settings['headers'] = $this->getHeaders();
		}

		/**
		 * @filter `gravityview/maps/request_settings` Modify request settings used to get content
		 * @since  1.2
		 * @see    WP_Http::request()
		 *
		 * @param array  $request_settings Args passed to wp_remote_request()
		 * @param string $url              URL to fetch
		 */
		$request_settings = apply_filters( 'gravityview/maps/request_settings', $request_settings, $url );

		$response = wp_remote_request( $url, $request_settings );

		if ( is_wp_error( $response ) ) {
			throw new Exception( $response->get_error_message() );
		}

		$status = wp_remote_retrieve_response_code( $response );

		if ( 200 !== $status ) {
			throw new Exception( sprintf( 'The server returned a %s status.', $status ) );
		}

		$content = wp_remote_retrieve_body( $response );

		$this->_log_http_errors( $content, $url );

		return $content;
	}

	/**
	 * Includes headers to be sent with the request.
	 *
	 * @since TBD
	 *
	 * @param string $name  The name of the header.
	 * @param string $value The value of the header.
	 *
	 * @return void
	 */
	public function addHeader( string $name, string $value ): void {
		$this->headers[ $name ] = $value;
	}

	/**
	 * Gets the headers to be sent with the request.
	 *
	 * @since TBD
	 *
	 * @return array
	 */
	protected function getHeaders(): array {
		return $this->headers;
	}

	/**
	 * Determines if the adapter has headers to send.
	 *
	 * @since TBD
	 *
	 * @return bool
	 */
	protected function hasHeaders(): bool {
		return ! empty( $this->headers );
	}

	/**
	 * {@inheritDoc}
	 */
	public function getName() {
		return 'gravityview';
	}

	/**
	 * The Geocoding library doesn't trigger exceptions if any providers work. We want to log some errors.
	 *
	 * @since 1.2
	 *
	 * @param string $status  The URL of the request
	 *
	 * @param string $content Error content
	 * @param string $content Response from provider
	 */
	private function _log_http_errors( $content, $url = '' ) {
		$json = json_decode( $content );

		// Not all providers return JSON; Open Street Map returns XML
		if ( empty( $json ) ) {
			return;
		}

		// Log any failed requests
		if ( ! empty( $json->error_message ) || ( isset( $json->success ) && ! $json->success ) ) {
			do_action( 'gravityview_log_error', sprintf( '%s: Unsuccessful request to %s', __METHOD__, $url ), $content );
		}
	}
}
