<?php
namespace GravityKit\GravityMaps\Map_Services;

use GV\View;

use WP_Post;

/**
 * Class Factory
 *
 * @since   3.1.0
 *
 * @package GravityKit\GravityMaps\Maps
 */
class Factory {
	/**
	 * Parses the view into a View object.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return View|null
	 */
	protected static function parse_view( $view ): ?View {
		if ( is_numeric( $view ) ) {
			return View::by_id( $view );
		}

		if ( $view instanceof View ) {
			return $view;
		}

		return View::from_post( $view );
	}

	/**
	 * Gets the map service class for the view.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return string|null
	 */
	public static function get_view_service_class( $view ): ?string {
		$view = static::parse_view( $view );

		if ( null === $view ) {
			return null;
		}

		$default_map_instance = Google_Maps::class;

		/**
		 * Currently we are not allowing other map services, but when we do, we will need to add it here.
		 */

		/**
		 * @filter `gk/gravitymaps/maps/service-factory/get-map-service-class-for-view` Enables the filtering of which class will handle the map service for a given view.
		 *
		 * @since 3.1.0
		 *
		 * @param string $default_map_instance The default map service class.
		 * @param View   $view                 The view object.
		 */
		return apply_filters( 'gk/gravitymaps/maps/service-factory/get-map-service-class-for-view', $default_map_instance, $view );
	}

	/**
	 * Given a view, returns the map service instance.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return Map_Service_Abstract|null
	 */
	public static function from_view( $view ): ?Map_Service_Abstract {
		$view = static::parse_view( $view );

		if ( null === $view ) {
			return null;
		}

		$service_class = static::get_view_service_class( $view );
		if ( null === $service_class ) {
			return null;
		}

		$map = call_user_func( [ $service_class, 'from_view' ], $view );

		/**
		 * @filter `gk/gravitymaps/maps/service-factory/get-map-service-instance-for-view` Enables the filtering of the instance of the map service for a given view.
		 *
		 * @since 3.1.0
		 *
		 * @param Map_Service_Abstract $map  The map service instance.
		 * @param View                 $view The view object.
		 */
		return apply_filters( 'gk/gravitymaps/maps/service-factory/get-map-service-instance-for-view', $map, $view );
	}
}