<?php

namespace GravityKit\GravityMaps\Map_Services;

use GV\REST\Core as Rest_Core;
use GravityKit\GravityMaps\Assets;
use WP_Scripts;

/**
 * Class Google_Maps
 *
 * @since   3.1.0
 *
 * @package GravityKit\GravityMaps\Maps
 */
class Google_Maps extends Map_Service_Abstract {
	/**
	 * @inheritDoc
	 *
	 * @since 3.1.0
	 */
	protected static $slug = 'google';

	/**
	 * The handle for the Google Maps script.
	 *
	 * @since 3.1.0
	 *
	 * @var string
	 */
	protected $script_handle = 'gv-google-maps';

	/**
	 * Register the main global Script for Google Maps usage.
	 *
	 * @since 3.1.2
	 *
	 */
	public function register_global_script(): void {
		$this->determine_script_handle();
		wp_register_script(
			$this->get_script_handle(),
			$this->get_script_url(),
			[],
			null
		);
	}

	/**
	 * @inheritDoc
	 *
	 * @since 3.1.0
	 */
	public function register_assets(): void {
		$this->register_global_script();

		$assets = Assets::instance();

		wp_register_script(
			'gv-google-maps-clusterer',
			$assets->get_url( 'assets/lib/markerclusterer.min.js' ),
			[],
			$assets->get_version()
		);

		wp_register_script(
			'gv-google-maps-spiderfier',
			$assets->get_url( 'assets/lib/oms.min.js' ),
			[],
			$assets->get_version()
		);

		wp_register_script(
			'gk-maps-search-fields',
			$assets->get_url( "assets/js/gk-maps-search-fields.js" ),
			[
				'jquery',
				'gk-maps-base',
				'gk-maps-utils',
				'gk-maps-controllers-view',
				'gk-maps-services-google-maps',
			],
			$assets->get_version()
		);

		wp_register_script(
			'gk-maps-address-auto-complete',
			$assets->get_url( "assets/js/gk-maps-address-auto-complete.js" ),
			[
				'jquery',
				'gk-maps-base',
				$this->get_script_handle(),
			],
			$assets->get_version()
		);

		wp_register_script(
			'gk-maps-services-google-maps',
			$assets->get_url( '/assets/js/services/google-maps.js' ),
			[
				'gk-maps-controllers-map',
				'gk-maps-controllers-marker',
				$this->get_script_handle(),
				'gv-google-maps-clusterer',
				'gv-google-maps-spiderfier',
			],
			$assets->get_version(),
			true
		);

		wp_register_style(
			'gravityview-maps',
			$assets->get_url( '/assets/css/gv-maps.css', false ),
			[],
			$assets->get_version()
		);

		wp_register_script(
			'gk-maps-iso-3166-1-alpha-2',
			$assets->get_url( 'assets/lib/iso-3166-1-alpha-2.js', false ),
			[ 'underscore', 'gk-maps-base' ],
			$assets->get_version()
		);

		wp_register_script(
			'gk-maps-form-address-auto-complete',
			$assets->get_url( 'assets/js/gk-maps-form-address-auto-complete.js' ),
			[
				'jquery',
				'gk-maps-iso-3166-1-alpha-2',
				$this->get_script_handle(),
				'gk-maps-base',
			],
			$assets->get_version()
		);
	}

	/**
	 * @inheritDoc
	 *
	 * @since 3.1.0
	 */
	public function enqueue_assets(): void {
		$assets = Assets::instance();

		if ( $this->is_view_service() ) {
			wp_enqueue_script( 'gk-maps-address-auto-complete' );

			wp_enqueue_style( 'gravityview-maps' );
			wp_enqueue_script( 'gk-maps-search-fields' );

			wp_localize_script(
				'gk-maps-search-fields',
				'gk_maps_search_fields',
				[
					'geolocationCurrentLocationError' => esc_attr__( 'There is no location support on this device or it is disabled. Please check your settings.', 'gk-gravitymaps' ),
					'invalidGeolocationCoordinates  ' => esc_attr__( 'There was an unknown error with retrieving your coordinates, please check if you are using any AdBlockers or if you are blocking Google API requests on your Browser.', 'gk-gravitymaps' ),
					'geolocationFenceCenterImage'     => $assets->get_url( '/assets/img/mapicons/geolocation-fence-center.png', false ),
					'searchRestEndpoint'              => Rest_Core::get_url() . '/views/{view_id}/entries.html?limit={limit}',
					'currentLocationLabel'            => esc_attr__( 'Current location', 'gk-gravitymaps' ),
					'currentLocationTitle'            => esc_attr__( 'Click to use current location', 'gk-gravitymaps' ),
					'searchOnMoveLabel'               => esc_attr__( 'Search as map moves', 'gk-gravitymaps' ),
					'redoSearchLabel'                 => esc_attr__( 'Redo search in map', 'gk-gravitymaps' ),
				]
			);

			wp_enqueue_script( 'gravityview-maps' );
			wp_localize_script( 'gk-maps-base', 'gk_gravitymaps_base', [
				'isDebug' => defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG,
			] );
			wp_localize_script( 'gk-maps-factory', 'gravitykit_gravitymaps_factory_default_arguments', [] );
			wp_localize_script( 'gk-maps-services-google-maps', 'gk_maps_services_google_maps_i18n', [
				'google_maps_api_error'         => esc_html__( 'Google Maps API returned an error. Please check the browser console for more information.', 'gk-gravitymaps' ),
			] );
		} elseif ( $this->is_form_service() ) {
			wp_enqueue_script( 'gk-maps-form-address-auto-complete' );
		}
	}

	/**
	 * Get Google Maps API key.
	 *
	 * @since 3.1.0
	 *
	 * @return string|null
	 */
	public function get_api_key(): ?string {
		global $gravityview_maps;

		if ( ! function_exists( 'gravityview' ) ) {
			return null;
		}

		$providers = $gravityview_maps->component_instances['Settings']->get_providers_api_keys();

		/**
		 * @filter `gravityview/maps/render/google_api_key` Modify the Google API key used when registering the `gv-google-maps` script
		 *
		 * @param string $key If the Google API key setting is set in GravityView Settings, use it. Otherwise: ''
		 */
		$api_key = apply_filters_deprecated( 'gravityview/maps/render/google_api_key', [ $providers['google_maps/key'] ], '3.1.0' );

		/**
		 * @filter `gk/gravitymaps/map-services/google-maps/api_key` Modify the Google API key used when registering the `gv-google-maps` script.
		 *
		 * @param string $key If the Google API key setting is set in GravityView Settings, use it. Otherwise: ''
		 */
		return apply_filters( 'gk/gravitymaps/map-services/google-maps/api_key', $providers['google_maps/key'] );
	}

	/**
	 * Fetches the Google Script URL.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	protected function get_script_url(): string {
		$args = [
			'libraries' => 'places',
			'key'       => $this->get_api_key(),

			/**
			 * Don't remove this piece, it prevents error on missing callback.
			 * As `callback` is required for the Maps Script but for our code it's not.
			 *
			 * @url https://developers.google.com/maps/documentation/javascript/url-params
			 */
			'callback'  => 'Function.prototype',
		];

		$url = set_url_scheme( 'https://maps.googleapis.com/maps/api/js' );
		$url = add_query_arg( $args, $url );

		return $url;
	}

	/**
	 * Gets the current Google Script Handle.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_script_handle(): ?string {
		return $this->script_handle;
	}

	/**
	 * Sets the Google Maps API JS handle.
	 *
	 * @since 3.1.0
	 *
	 * @param string $handle Handle for the Google Maps v3 API script.
	 *
	 * @return void
	 */
	protected function set_script_handle( string $handle ): void {
		$this->script_handle = $handle;
	}

	/**
	 * Gets a regular expression to match the Google Maps JS URL.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	protected function get_maps_script_url_regex(): string {
		return '/maps\.google(apis)?\.com\/maps\/api\/js/ism';
	}

	/**
	 * Determines the Google JS handle internally, to avoid conflicting with other plugins.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	public function determine_script_handle(): void {
		global $wp_scripts;

		if ( ! ( $wp_scripts instanceof WP_Scripts ) ) {
			$wp_scripts = new WP_Scripts();
		}

		// Default: use our own script.
		$handle = $original_handle = $this->get_script_handle();

		/**
		 * Find other plugins that have registered Google Maps.
		 *
		 * @since 1.2
		 */
		foreach ( $wp_scripts->registered as $key => $script ) {
			if ( preg_match( $this->get_maps_script_url_regex(), $script->src ) ) {
				$existing_args = wp_parse_args( wp_parse_url( $script->src, PHP_URL_QUERY ) );
				$existing_libs = [];
				if ( ! empty( $existing_args['libraries'] ) && is_string( $existing_args['libraries'] ) ) {
					$existing_libs = explode( ',', $existing_args['libraries'] );
				}
				if ( ! in_array( 'places', $existing_libs, true ) ) {
					$existing_libs[] = 'places';
				}
				$existing_libs   = implode( ',', $existing_libs );

				// Overwrite the existing script SRC to include the Libs we want.
				$wp_scripts->registered[ $key ]->src = add_query_arg( [ 'libraries' => $existing_libs, ], $script->src );

				// Set the handle to the existing script.
				$handle = $script->handle;
				do_action( 'gravityview_log_debug', __METHOD__ . ': Using non-GravityView Maps script: ' . $handle, $script );
				break;
			}
		}

		/**
		 * @filter `gravityview_maps_google_script_handle` If your site already has Google Maps v3 API script enqueued, you can specify the handle here.
		 *
		 * @param string $script_slug Default: `gv-google-maps`
		 */
		$handle = apply_filters( 'gravityview_maps_google_script_handle', $handle );

		/**
		 * Any scripts that we add that depend on the original handle get modified.
		 */
		foreach ( $wp_scripts->registered as $key => $script ) {
			// Find Scripts that depend on this original handle.
			$index = array_search( $original_handle, $script->deps, true );
			if ( false === $index ) {
				continue;
			}

			// Replace with the new handle.
			$script->deps[ $index ] = $handle;

			// Overwrite the script.
			$wp_scripts->registered[ $key ] = $script;
		}

		// Deregister our Google Maps.
		if ( $original_handle !== $handle ) {
			wp_deregister_script( $original_handle );
		}

		$this->set_script_handle( $handle );
	}


	/**
	 * Convert zoom control settings to values expected by Google Maps
	 *
	 * @see   https://developers.google.com/maps/documentation/javascript/controls#Adding_Controls_to_the_Map
	 *
	 * @since 1.4.2
	 * @since 3.1.0 Moved from Render_Map to here.
	 *
	 * @param array $map_settings Array of map settings
	 *
	 * @return bool|null `TRUE`: show zoom control; `FALSE`: hide zoom control; `NULL`: let map decide
	 */
	public static function parse_map_zoom_control( $map_settings ): ?bool {
		switch ( rgar( $map_settings, 'map_zoom_control' ) ) {
			// Force don't show zoom
			case 'none':
				$zoomControl = false;
				break;

			// Force zoom to display
			case 'small':
			case 'large': // Backward compatibility
				$zoomControl = true;
				break;
			// Let the map decide
			default:
				$zoomControl = null;
				break;
		}

		return $zoomControl;
	}
}