<?php
namespace GravityKit\GravityMaps\Map_Services;

use GravityKit\GravityMaps\Admin;
use GravityKit\GravityMaps\Data;
use GV\View;

use WP_Post;

/**
 * Class Map_Service_Abstract
 *
 * @since   3.1.0
 *
 * @package GravityKit\GravityMaps\Maps
 */
abstract class Map_Service_Abstract {

	/**
	 * View object.
	 *
	 * @since 3.1.0
	 *
	 * @var ?View
	 */
	protected $view = null;

	/**
	 * Form array from Gravity Forms
	 *
	 * @since 3.1.2
	 *
	 * @var ?array
	 */
	protected $form = null;

	/**
	 * Data object.
	 *
	 * @since 3.1.0
	 *
	 * @var Data
	 */
	protected $data;

	/**
	 * What is the maps service for this instance of Maps.
	 *
	 * @since 3.1.0
	 *
	 * @var string
	 */
	protected static $slug;

	/**
	 * Gets the slug for the maps service.
	 * E.g.: google, mapbox, etc.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public static function get_slug(): string {
		return static::$slug;
	}

	/**
	 * Given a view, returns this map instance.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return Map_Service_Abstract
	 */
	public static function from_view( $view ): Map_Service_Abstract {
		$map = new static;
		$map->set_view( $view );
		$map->setup_data();

		return $map;
	}

	/**
	 * Given a form, returns this map instance.
	 *
	 * @since 3.1.2
	 *
	 * @param int|string|null $form The ID of the form we will build this service for.
	 *
	 * @return Map_Service_Abstract
	 */
	public static function from_form( $form ): Map_Service_Abstract {
		$map = new static;
		$map->set_form( $form );

		return $map;
	}

	/**
	 * Determines if this is a view service.
	 *
	 * @since 3.1.2
	 *
	 * @return bool
	 */
	public function is_view_service(): bool {
		return ! empty( $this->view );
	}

	/**
	 * Determines if this is a form service.
	 *
	 * @since 3.1.2
	 *
	 * @return bool
	 */
	public function is_form_service(): bool {
		return ! empty( $this->form );
	}

	/**
	 * Sets the form object.
	 *
	 * @since 3.1.2
	 *
	 * @param int|string|null $form The ID of the form we will build this service for.
	 *
	 * @return void
	 */
	protected function set_form( $form ): void {
		if ( is_numeric( $form ) ) {
			$this->form = \GFAPI::get_form( $form );
		}
	}

	/**
	 * Sets the view object.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return void
	 */
	protected function set_view( $view ): void {
		if ( is_numeric( $view ) ) {
			$this->view = View::by_id( $view );
		} elseif ( $view instanceof View ) {
			$this->view = $view;
		} else {
			$this->view = View::from_post( $view );
		}
	}

	/**
	 * Configures the data object.
	 *
	 * Currently, we need to use the View Template to avoid extra queries and backwards compatibility.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	protected function setup_data(): void {
		$this->data = Data::get_instance( $this->get_view() );
	}

	/**
	 * Gets the data object, which holds the markers.
	 *
	 * @todo The Data object should no longer exist, we should have a Marker Factory allows us to pull Markers from the database for a given service.
	 *
	 * @since 3.1.0
	 *
	 * @return Data
	 */
	public function get_data(): Data {
		return $this->data;
	}

	/**
	 * Gets the view in which this instance of Maps is being used.
	 *
	 * @since 3.1.0
	 *
	 * @return ?View
	 */
	public function get_view(): ?View {
		return $this->view;
	}

	/**
	 * Gets the form which this instance of Maps is being used.
	 *
	 * @since 3.1.2
	 *
	 * @return ?array
	 */
	public function get_form(): ?array {
		return $this->form;
	}

	/**
	 * Gets the view settings, this is only required because it only fetches the Map related settings.
	 *
	 * @since 3.1.0
	 *
	 * @return array
	 */
	public function get_view_related_settings(): array {
		return Admin::get_map_settings( $this->get_view()->ID );
	}

	/**
	 * Gets all the markers for this map.
	 *
	 * @since 3.1.0
	 *
	 * @return array
	 */
	public function get_markers(): array {
		return $this->get_data()->markers;
	}

	/**
	 * Gets all the markers for a given entry.
	 *
	 * @since 3.1.0
	 *
	 * @param int|string $entry
	 *
	 * @return array
	 */
	public function get_markers_by_entry( $entry ): array {
		return $this->get_data()->get_markers_by_entry( $entry );
	}

	/**
	 * Determine if the service can use the REST API. Currently, it checks the global settings about rest api and the view settings.
	 *
	 * @since 3.1.0
	 *
	 * @return bool
	 */
	public function can_use_rest_api(): bool {
		$view = $this->get_view();

		if ( ! $view ) {
			return false;
		}

		if ( gravityview()->plugin->settings->get( 'rest_api' ) && $view->settings->get( 'rest_disable' ) === '1' ) {
			return false;
		}

		if ( ! gravityview()->plugin->settings->get( 'rest_api' ) && $view->settings->get( 'rest_enable' ) !== '1' ) {
			return false;
		}

		return true;
	}

	/**
	 * Hooks the asset register and enqueue to WordPress hooks.
	 * The earliest you can register assets safely in WordPress is `init` so we hook to that.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	public function hook_assets(): void {
		$register_hook = 'init';

		if ( did_action( $register_hook ) || doing_action( $register_hook ) ) {
			$register_hook = $this->get_assets_hook();
		}

		// Prevents the double hooking.
		if (
			has_action( $register_hook, [ $this, 'register_assets' ] )
			&& has_action( $this->get_assets_hook(), [ $this, 'enqueue_assets' ] )
		) {
			return;
		}

		// At this point we know that we are past the registration and enqueuing point when it was called, so Register and Enqueue now.
		if ( did_action( $register_hook ) && ! doing_action( $register_hook ) ) {
			$this->register_assets();
			$this->enqueue_assets();

			return;
		}

		// If we are not past the registration and enqueuing point, then hook to the registration point.
		add_action( $register_hook, [ $this, 'register_assets' ], 15 );
		add_action( $this->get_assets_hook(), [ $this, 'enqueue_assets' ], 20 );
	}

	/**
	 * Gets the hook for enqueuing the assets.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_assets_hook(): string {
		return 'wp_enqueue_scripts';
	}

	/**
	 * Register the map service assets.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	abstract public function register_assets(): void;

	/**
	 * Enqueue the map service assets.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	abstract public function enqueue_assets(): void;
}