<?php

namespace GravityKit\GravityMaps;


use GravityKit\GravityMaps\Foundation\Helpers\Arr;

class Markers {

	/**
	 * Local storage of the cached Marker Values.
	 *
	 * @since 2.0
	 *
	 * @var array
	 */
	protected static $items = [];

	/**
	 * Gets the cached value for a given marker.
	 *
	 * @since 2.0
	 *
	 * @param int|string $entry_id
	 *
	 * @return false|object
	 */
	public static function get( $entry_id ) {
		if ( ! isset( static::$items[ (int) $entry_id ] ) ) {
			return false;
		}

		return static::$items[ (int) $entry_id ];
	}

	/**
	 * Primes the cache for all the Map Markers for a given view to be queried at once.
	 *
	 * @since 2.0
	 *
	 * @param array $entries
	 * @param array $fields
	 *
	 */
	public static function prime_cache( array $entries, array $fields ) {
		$entries_as_array = array_map( static function ( $entry ) {
			return $entry->as_entry();
		}, $entries );
		$entries_ids      = wp_list_pluck( $entries_as_array, 'id' );
		$entries_ids      = array_filter( $entries_ids, 'is_numeric' );

		$entries_ids = array_filter( $entries_ids, static function ( $entry_id ) {
			return false === Markers::get( $entry_id );
		} );

		$meta_keys    = [];
		$field_values = [];

		foreach ( $fields as $field ) {
			$id = Arr::get( (array) $field, 'id' );
			if ( empty( $id ) ) {
				continue;
			}

			if ( $field instanceof \GF_Field_Address ) {
				$meta_keys = \GravityKit\GravityMaps\Geolocation\Fields::instance()->get_meta_keys( $id );
			} elseif ( $field instanceof \GF_Field ) {
				$field_values[] = $id;
			}
		}

		if ( ! empty( $meta_keys ) ) {
			$values = static::get_meta_values_for_entries( $entries_ids, $meta_keys );
			static::$items += $values;
		}

		if ( ! empty( $field_values ) ) {
			foreach ( $entries as $entry ) {
				foreach ( $field_values as $field_id ) {
					if ( ! isset( $entry[ $field_id ] ) ) {
						continue;
					}

					$values[] = $entry[ $field_id ];
				}

				if ( ! empty( $values ) ) {
					static::$items[ (int) $entry['id'] ] = $values;
				}
			}
		}
	}

	/**
	 * Mimics `gform_get_meta_values_for_entries` but runs in a more performant way for the purpose of caching geolocation.
	 *
	 * @since 3.1.4
	 *
	 * @param array $entry_ids
	 * @param array $meta_keys
	 *
	 * @return array
	 */
	public static function get_meta_values_for_entries( array $entry_ids, array $meta_keys ): array {
		if ( empty( $entry_ids ) ) {
			return [];
		}

		if ( empty( $meta_keys ) ) {
			return [];
		}

		global $wpdb;
		$table_name     = \RGFormsModel::get_entry_meta_table_name();
		$meta_key_where = [];

		foreach ( $meta_keys as $meta_key ) {
			$meta_key         = esc_sql( $meta_key );
			$meta_key_where[] = "`meta_key` = '{$meta_key}'";
		}

		$entry_ids_sql = implode( ',', $entry_ids );

		$meta_key_where_sql = implode( ' OR ', $meta_key_where );

		$sql_query = "  SELECT
	                    `entry_id`, `meta_key`, `meta_value`
	                    FROM {$table_name}
	                    WHERE
	                        entry_id IN ( {$entry_ids_sql} )
	                    	AND ( {$meta_key_where_sql} )
					";

		$results = $wpdb->get_results( $sql_query );

		return array_reduce( $results, static function ( $carry, $item ) {
			$item->entry_id = (int) $item->entry_id;

			if ( ! isset( $carry[ $item->entry_id ] ) ) {
				$carry[ $item->entry_id ] = (object) [
					'entry_id' => $item->entry_id,
				];
			}

			$carry[ $item->entry_id ]->{$item->meta_key} = $item->meta_value;

			return $carry;
		}, [] );
	}

	/**
	 * Filters the markers to only keep valid ones.
	 *
	 * @since 2.2
	 *
	 * @param mixed $marker
	 *
	 * @return bool
	 */
	public static function filter_valid_marker_data( $marker ): bool {
		if ( empty( $marker ) ) {
			return false;
		}

		if ( empty( $marker['lat'] ) ) {
			return false;
		}

		if ( empty( $marker['long'] ) ) {
			return false;
		}

		return true;
	}
}
