<?php

namespace GravityKit\GravityMaps\Query_Modifiers;

use GravityKit\GravityMaps\Form_Fields;

use GFFormsModel;
use GF_Query_Condition;
use GF_Query;
use GravityKit\GravityMaps\Geolocation\Fields;

/**
 * Class Search_GF_Query_Bounds_Condition
 *
 * @since 2.0
 *
 */
class Bounds_Condition extends GF_Query_Condition {
	/**
	 * The mode we handle fields.
	 *
	 * @since 2.2
	 *
	 * @var string
	 */
	protected $mode = 'internal';

	/**
	 * Sets the fields that need to be used based on the GF forms array of IDs provided.
	 *
	 * @since 2.0
	 *
	 * @var array[]
	 */
	protected $fields;

	/**
	 * Sets the bounds that will be used for creating the SQL.
	 *
	 * @since 2.0
	 *
	 * @var array[]
	 */
	protected $bounds;

	/**
	 * @since 2.2.1
	 *
	 * @var array The aliases we need to join.
	 */
	protected $aliases = [];

	/**
	 * Sets which fields we will use for the SQL for this particular conditional.
	 *
	 * @since 2.0
	 *
	 * @param int[]|string[] $fields Field IDS that will be turned to array of data.
	 * @param string         $type   From which type of field we are looking for.
	 *
	 * @return $this
	 */
	public function set_fields( $fields, string $type = 'internal' ): self {
		$callbacks = Fields::instance()->get_meta_keys_callbacks( $type );

		// Don't set fields when there are no callbacks.
		if ( ! $callbacks ) {
			return $this;
		}

		$this->mode = $type;

		$this->fields = array_map( static function ( $id ) use ( $callbacks ) {
			return [
				'id' => $id,
				'lat' => $callbacks['lat']( $id ),
				'long' => $callbacks['long']( $id ),
			];
		}, (array) $fields );
		return $this;
	}

	/**
	 * Sets the Bounds for this particular conditional.
	 *
	 * @since 2.0
	 *
	 * @param array $bounds
	 *
	 * @return $this
	 */
	public function set_bounds( array $bounds ) {
		$this->bounds = $bounds;
		return $this;
	}

	/**
	 * Add a alias to the ones we need a join for.
	 *
	 * @since 2.2.1
	 *
	 * @param string $alias    The alias generated by the query.
	 * @param string $field_id The field ID.
	 * @param string $sql      The SQL that the alias is used on.
	 *
	 */
	public function add_alias( $alias, $field_id, $sql ): void {
		$this->aliases[ $field_id ] = [
			'name' => $alias,
			'sql' => $sql,
		];
	}

	/**
	 * Gets all the aliases related to the current conditional.
	 *
	 * @since 2.2.1
	 *
	 * @return array
	 */
	public function get_aliases(): array {
		return $this->aliases;
	}

	/**
	 * Generate the SQL based on the params set for this particular type of conditional.
	 *
	 * @param GF_Query The query.
	 *
	 * @return string The SQL this condition generates.
	 */
	public function sql( $query ) {
		if ( ! isset( $this->bounds, $this->fields ) ) {
			return null;
		}

		$sql = [];

		/*
		 * Reference: https://stackoverflow.com/questions/4834772/get-all-records-from-mysql-database-that-are-within-google-maps-getbounds
		 */

		foreach ( $this->fields as $field ) {
			$alias_lat  = $query->_alias( $field['lat'], 0, 'geo' );
			$alias_long = $query->_alias( $field['long'], 0, 'geo' );

			$lat_meta_value  = "$alias_lat.meta_value";
			$long_meta_value = "$alias_long.meta_value";

			$bounds_conditional = '
				((CASE WHEN %1$s < %2$s
				        THEN %5$s BETWEEN %1$s AND %2$s
				        ELSE %5$s BETWEEN %2$s AND %1$s
				END)
				AND
				(CASE WHEN %3$s < %4$s
				        THEN %6$s BETWEEN %3$s AND %4$s
				        ELSE %6$s BETWEEN %4$s AND %3$s
				END))
			';

			$bounds_query = sprintf( $bounds_conditional, $this->bounds['max_lat'], $this->bounds['min_lat'], $this->bounds['min_lng'], $this->bounds['max_lng'], $lat_meta_value, $long_meta_value );
			$sql[]        = $alias_sql = $bounds_query;

			$this->add_alias( $alias_lat, $field['lat'], $alias_sql );
			$this->add_alias( $alias_long, $field['long'], $alias_sql );
		}

		// Add the join aliases to the query.
		add_filter( 'gform_gf_query_sql', [ $this, 'include_join_aliases' ], 25 );

		return ' ( ' . implode( ' ' . static::_OR . ' ', $sql ) . ' ) ';
	}

	/**
	 * Ensures that the SQL aliases are included in the query.
	 *
	 * @since 2.2.1
	 *
	 * @param $sql
	 *
	 * @return array
	 */
	public function include_join_aliases( $sql ) {
		$aliases        = $this->get_aliases();
		$meta_table     = GFFormsModel::get_entry_meta_table_name();
		$entry_table    = GFFormsModel::get_entry_table_name();
		$regex          = "/FROM +(?:`?{$entry_table}`?) +AS +`?([^` ]+)`?/i";
		$entry_id_alias = preg_match( $regex, $sql['from'], $matches ) ? $matches[1] : 't1';

		foreach ( $aliases as $field_id => $alias ) {
			// Only add the alias if the query contains the alias SQL.
			if ( strpos( $sql['where'], $alias['sql'] ) === false ) {
				continue;
			}

			$new_join = " LEFT JOIN `{$meta_table}` `{$alias['name']}` ON `{$alias['name']}`.entry_id = `{$entry_id_alias}`.`id` AND `{$alias['name']}`.meta_key = '$field_id'";

			// Don't add the join twice.
			if ( strpos( $sql['join'], $new_join ) !== false ) {
				continue;
			}

			$sql['join'] .= $new_join;
		}

		// This should only run once.
		remove_filter( 'gform_gf_query_sql', [ $this, 'include_join_aliases' ], 25 );

		return $sql;
	}
}