<?php
namespace GravityKit\GravityMaps\Template;

use GravityView_View;
use GravityKit\GravityMaps\Map_Services\Factory;
use GravityKit\GravityMaps\Map_Services\Google_Maps;
use GravityKit\GravityMaps\Map_Services\Map_Service_Abstract;
use GravityKit\GravityMaps\Markers;
use GV\View;
use GVCommon;

use WP_Post;

/**
 * Class Map
 *
 * @since   3.1.0
 *
 * @package GravityKit\GravityMaps\Template
 */
class Map {
	/**
	 * Default pagination control setting.
	 * To alter this value look at filter `gk/gravitymaps/map/control-pagination`.
	 *
	 * @since 3.1.4
	 *
	 * @var bool
	 */
	protected const CONTROL_PAGINATION = false;

	/**
	 * Stores all the views that were loaded, so we avoid double setup of a view map.
	 *
	 * @since 3.1.3
	 *
	 * @var array The views that were loaded.

	 */
	protected static $views_loaded = [];

	/**
	 * Stores all the existing map ids that were rendered.
	 *
	 * @since 3.1.0
	 *
	 * @var string[] The existing map IDs.
	 */
	protected static $existing_ids = [];

	/**
	 * Cache the resulting value of the will_render method.
	 *
	 * @since 3.1.3
	 *
	 * @var bool
	 */
	protected $will_render;

	/**
	 * Gets all the existing IDs in the request.
	 *
	 * @since 3.1.0
	 *
	 * @return string[]
	 */
	public static function get_existing_ids(): array {
		return static::$existing_ids;
	}

	/**
	 * Given an Entry ID returns a Map Template instance.
	 *
	 * @since 3.1.0
	 *
	 * @param string|int                   $entry_id Entry ID.
	 * @param WP_Post|View|int|string|null $view     The view.
	 *
	 * @return static
	 */
	public static function from_entry_id( $entry_id, $view = null ): self {
		if ( null === $view ) {
			$view = gravityview_get_view_id();
		}

		$map = new static();
		$map->set_entry_id( $entry_id );
		$map->set_view( $view );
		$map->setup_service();

		return $map;
	}

	/**
	 * Given an Entry ID returns a Map Template instance.
	 *
	 * @since 3.1.0
	 * @since 3.1.6 Included the preload_service parameter.
	 *
	 * @param WP_Post|View|int|string|null $view The view.
	 * @param bool                         $preload_service Whether to preload the service for Markers/Maps.
	 *
	 * @return static
	 */
	public static function from_view( $view, bool $preload_service = true ): self {
		$view = static::normalize_view( $view );
   		if ( static::$views_loaded[ $view->ID ] ?? false ) {
			return static::$views_loaded[ $view->ID ];
		}

		$map = new static();
		$map->set_view( $view );

		if ( true === $preload_service ) {
			$map->setup_service();
		}

		return $map;
	}

	/**
	 * Unique ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @var string The map ID.
	 */
	protected $id;

	/**
	 * Instance of the view associated with this map.
	 *
	 * @since 3.1.0
	 *
	 * @var ?View The view object.
	 */
	protected $view = null;

	/**
	 * Instance of the map service associated with this map.
	 *
	 * @since 3.1.0
	 *
	 * @var Map_Service_Abstract The map service object.
	 */
	protected $service;

	/**
	 * Entry ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @var null|int The entry ID
	 */
	protected $entry_id = null;

	/**
	 * Constructor, mostly here to make sure the ID is set.
	 *
	 * @since 3.1.0
	 */
	public function __construct() {
		$this->setup_id();
	}

	/**
	 * Sets up the ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	protected function setup_id(): void {
		do {
			$id = uniqid( '', false );
		} while ( array_key_exists( $id, static::get_existing_ids() ) );

		$this->id = $id;
	}

	/**
	 * Gets the random generated ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Using the random ID, prefix it for a unique HTML ID.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_html_id() {
		return 'gv-map-canvas-' . $this->get_id();
	}

	/**
	 * Determines if the map has an entry ID.
	 *
	 * @since 3.1.0
	 *
	 * @return bool
	 */
	public function has_entry(): bool {
		return ! empty( $this->get_entry_id() );
	}

	/**
	 * Sets the entry ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @param int|string $entry_id The entry ID.
	 *
	 * @return void
	 */
	public function set_entry_id( $entry_id ): void {
		$this->entry_id = (int) $entry_id;
	}

	/**
	 * Gets the entry ID for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return int
	 */
	public function get_entry_id(): ?int {
		return $this->entry_id;
	}

	/**
	 * Sets the view object.
	 *
	 * @since 3.1.0
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return void
	 */
	protected function set_view( $view ): void {
		$this->view = static::normalize_view( $view );
	}

	/**
	 * Normalizes the view object.
	 *
	 * @since 3.1.3
	 *
	 * @param WP_Post|View|int|string $view The view.
	 *
	 * @return ?View
	 */
	protected static function normalize_view( $view ): ?View {
		if ( is_numeric( $view ) ) {
			$view = View::by_id( $view );
		} elseif ( ! $view instanceof View ) {
			$view = View::from_post( $view );
		}

		if ( ! $view instanceof View ) {
			return null;
		}

		return $view;
	}

	/**
	 * Gets the view associated with this Map.
	 *
	 * @since 3.1.0
	 *
	 * @return ?View
	 */
	public function get_view(): ?View {
		return $this->view;
	}

	/**
	 * Gets the HTML Classes for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return string
	 */
	public function get_html_classes(): string {
		// 'gv-map-canvas' is a legacy class for compatibility.
		$classes = [ 'gv-map-canvas', 'gk-map-canvas' ];

		if ( $this->has_entry() ) {
			$classes[] = 'gk-map-canvas-' . $this->get_entry_id();
		} else {
			$classes[] = 'gk-multi-entry-map';
		}

		$classes = array_map( 'sanitize_html_class', $classes );
		$classes = array_filter( $classes );

		return implode( ' ', $classes );
	}

	/**
	 * Escape the data for use in HTML attributes.
	 *
	 * @since 3.1.0
	 *
	 * @param mixed $data Data to escape.
	 *
	 * @return false|string
	 */
	protected function esc_json_attr( $data ) {
		return wp_json_encode( $data, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE );
	}

	/**
	 * Configures the map service that will render this map, currently only using the view.
	 *
	 * @since 3.1.0
	 * @since 3.1.6 Method is now public.
	 *
	 * @return void
	 */
	public function setup_service(): void {
		$this->service = Factory::from_view( $this->get_view() );

		if ( ! $this->service ) {
			do_action( 'gravityview_log_debug', 'Map service not found for view', $this->get_view() );
		}
	}

	/**
	 * Do we have a proper map service associated with the Map Template?
	 *
	 * @since 3.1.0
	 *
	 * @return bool
	 */
	public function has_service(): bool {
		return $this->service instanceof Map_Service_Abstract;
	}

	/**
	 * Gets the map service that will render this map.
	 *
	 * @since 3.1.0
	 *
	 * @return Map_Service_Abstract
	 */
	public function get_service(): Map_Service_Abstract {
		return $this->service;
	}

	/**
	 * Convert zoom control settings to values expected by Google Maps
	 *
	 * @see   https://developers.google.com/maps/documentation/javascript/controls#Adding_Controls_to_the_Map
	 *
	 * @since 3.1.0
	 *
	 * @param array $map_settings Array of map settings
	 *
	 * @return bool|null `TRUE`: show zoom control; `FALSE`: hide zoom control; `NULL`: let map decide
	 */
	protected function parse_legacy_compatibility_map_zoom_control( $map_settings ): ?bool {
		$zoom_control = rgar( $map_settings, 'map_zoom_control' );

		if ( null === $zoom_control ) {
			return null;
		}

		$zoom_control_map = [
			'none' => false,
			'small' => true,
			'large'	=> true,
		];

		return $zoom_control_map[ $zoom_control ] ?? null;
	}

	/**
	 * Determines if this given map should have take ownership of the pagination on the view.
	 *
	 * @since 3.1.4
	 *
	 * @return bool
	 */
	public function should_control_pagination(): bool {
		/**
		 * Filter the pagination control setting for the map.
		 *
		 * @filter `gk/gravitymaps/map/control-pagination`
		 *
		 * @since  3.1.4
		 *
		 * @param bool $pagination_override Whether to override the View's native pagination behavior.
		 * @param Map  $map                 The map object.
		 */
		return (bool) apply_filters( 'gk/gravitymaps/map/control-pagination', static::CONTROL_PAGINATION, $this );
	}

	/**
	 * Gets the global settings for Map that will be passed to the JS.
	 *
	 * @since 3.1.0
	 *
	 * @return array
	 */
	public function get_settings(): array {
		// get view map settings
		$map_settings = $this->get_service()->get_view_related_settings();

		/**
		 * Default settings
		 */
		$map_options = [
			'pagination_control'      => $this->should_control_pagination(),
			'MapOptions'              => [
				'zoomControl' => Google_Maps::parse_map_zoom_control( $map_settings ),
			],
			'automatic_radius_search' => (bool) ( $map_settings['map_automatic_radius_search'] ?? false ),
			'api_key'                 => $this->get_service()->get_api_key(), // @todo This needs to be Just Google Maps
			'icon'                    => $map_settings['map_marker_icon'],
			'markerClusterIconPath'   => plugins_url( 'assets/img/mapicons/m', $GLOBALS['gravityview_maps']->path ),
			'layers'                  => [
				'bicycling' => (int) ( 'bicycling' === $map_settings['map_layers'] ),
				'transit'   => (int) ( 'transit' === $map_settings['map_layers'] ),
				'traffic'   => (int) ( 'traffic' === $map_settings['map_layers'] ),
			],
			'is_single_entry'         => gravityview_is_single_entry(),
			'icon_bounce'             => true,
			// Return false to disable icon bounce
			'sticky'                  => ! empty( $map_settings['map_canvas_sticky'] ),
			// todo: make sure we are running the map template
			'template_layout'         => ! empty( $map_settings['map_canvas_position'] ) ? $map_settings['map_canvas_position'] : '',
			// todo: make sure we are running the map template
			'marker_link_target'      => '_top',
			// @since 1.4 allow to specify a different marker link target
			'mobile_breakpoint'       => 600,
			// @since 1.4.2 Set the mobile breakpoint, in pixels
			'infowindow'              => [
				'no_empty'   => true,
				// @since 1.4 check if the infowindow is empty, and if yes, force a link to the single entry
				'empty_text' => __( 'View Details', 'gk-gravitymaps' ),
				//@since 1.4, If the infowindow is empty, generate a link to the single entry with this text
				'max_width'  => 300,
				//@since 1.4, Max width of the infowindow (in px)
			],
		];
		$map_options['errors'] = [];

		if ( has_filter( 'gravityview/maps/markers/lat_long/fields_id' ) ) {
			$map_options['errors'][] = [
				'code'     => 'invalid_hook_lat_long_fields',
				'message'  => esc_html__( 'Hook `gravityview/maps/markers/lat_long/fields_id` is deprecated since version 3.1.0! Use `gk/gravitymaps/markers/coordinates/fields-ids` instead.', 'gk-gravitymaps' ),
				'hide_map' => false,
			];
		}

		/**
		 * @filter `gravityview/maps/render/options` Modify the map options used by Google. Uses same parameters as the [Google MapOptions](https://developers.google.com/maps/documentation/javascript/reference#MapOptions)
		 *
		 * @param array $map_options Map Options
		 */
		$map_options = apply_filters( 'gravityview/maps/render/options', $map_options );

		$default_MapOptions = [
			'backgroundColor'           => null,
			'center'                    => null,
			'disableDefaultUI'          => null,
			'disableDoubleClickZoom'    => empty( $map_settings['map_doubleclick_zoom'] ),
			'draggable'                 => ! empty( $map_settings['map_draggable'] ),
			'draggableCursor'           => null,
			'draggingCursor'            => null,
			'heading'                   => null,
			'keyboardShortcuts'         => null,
			'mapMaker'                  => null,
			'mapTypeControl'            => null,
			'mapTypeControlOptions'     => null,
			'mapTypeId'                 => strtoupper( $map_settings['map_type'] ),
			'maxZoom'                   => ! isset( $map_settings['map_maxzoom'] ) ? 16 : (int) $map_settings['map_maxzoom'],
			'minZoom'                   => ! isset( $map_settings['map_minzoom'] ) ? 3 : (int) $map_settings['map_minzoom'],
			'noClear'                   => null,
			'overviewMapControl'        => null,
			'overviewMapControlOptions' => null,
			'panControl'                => ! empty( $map_settings['map_pan_control'] ),
			'panControlOptions'         => null,
			'rotateControl'             => null,
			'rotateControlOptions'      => null,
			'scaleControl'              => null,
			'scaleControlOptions'       => null,
			'scrollwheel'               => ! empty( $map_settings['map_scrollwheel_zoom'] ),
			'streetView'                => null,
			'streetViewControl'         => ! empty( $map_settings['map_streetview_control'] ),
			'streetViewControlOptions'  => null,
			'styles'                    => empty( $map_settings['map_styles'] ) ? null : json_decode( $map_settings['map_styles'] ),
			'tilt'                      => null,
			'zoom'                      => ! isset( $map_settings['map_zoom'] ) ? 15 : (int) $map_settings['map_zoom'],
			'zoomControl'               => null,
			'zoomControlOptions'        => null,
			'markerClustering'          => ! empty( $map_settings['map_marker_clustering'] ),
			'markerClusteringMaxZoom'   => empty( $map_settings['map_marker_clustering_maxzoom'] ) ? null : (int) $map_settings['map_marker_clustering_maxzoom'],
		];

		/**
		 * Enforce specific Google-available parameters, then remove null options
		 */
		$map_options['MapOptions'] = array_filter( shortcode_atts( $default_MapOptions, $map_options['MapOptions'] ), static function ( $value ) {
			return ! is_null( $value );
		} );

		unset( $default_MapOptions );

		$translations = [
			'display_errors'                => GVCommon::has_cap( [ 'gravityforms_edit_settings', 'gravityview_view_settings' ] ),
			'is_search'                     => Utils::is_search(),
			'google_maps_api_key_not_found' => esc_html__( 'Google Maps API key was not found. Please make sure that it is configured in GravityView settings.', 'gk-gravitymaps' ),
			'google_maps_script_not_loaded' => esc_html__( 'Google Maps script failed to load.', 'gk-gravitymaps' ),
			'google_maps_api_error'         => esc_html__( 'Google Maps API returned an error. Please check the browser console for more information.', 'gk-gravitymaps' ),
			'entries_missing_coordinates'   => esc_html__( 'None of the address fields have latitude/longitude coordinates. Please make sure that at least one address is geocoded before a map can be displayed.', 'gk-gravitymaps' ),
			'cannot_use_rest_error'         => esc_html__( 'Rest API cannot be disabled to be able to use the Maps functionality.', 'gk-gravitymaps' ),
			'hide_until_searched'           => (bool) $this->get_view()->settings->get( 'hide_until_searched' ),
			'can_use_rest'                  => $this->get_service()->can_use_rest_api(),
		];

		if ( empty( $map_settings['map_address_field'] ) ) {
			$translations['address_field_missing'] = esc_html__( 'The "Address Field" setting has not been configured for this View. In View Settings, click on the Maps tab, set the fields you would like to display on this map, then save the View.', 'gk-gravitymaps' );
		}

		return array_merge( $map_options, $translations );
	}

	/**
	 * Gets the markers for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return array
	 */
	public function get_markers(): array {
		if ( ! $this->has_entry() ) {
			$markers = $this->get_service()->get_markers();
		} else {
			$markers = $this->get_service()->get_markers_by_entry( $this->get_entry_id() );
		}

		$markers = array_values( array_map( static function ( $marker ) {
			return $marker->to_array();
		}, $markers ) );

		return array_filter( $markers, [ Markers::class, 'filter_valid_marker_data' ] );
	}

	/**
	 * Get the data that will be passed to the JS part of the map rendering.
	 *
	 * @since 3.1.0
	 *
	 * @return array
	 */
	public function get_canvas_data(): array {
		if ( ! $this->has_service() ) {
			return [];
		}

		$data = [
			'view_id' => $this->get_view()->ID,
			'entry_id' => $this->get_entry_id(),
			'is_multi_entry_map' => ! $this->has_entry(),
			'markers_data' => $this->get_markers(),
			'markers' => [],
		];

		if ( ! $this->has_entry() ) {
			$paging            = GravityView_View::getInstance()->getPaging();
			$data['page_size'] = $paging['page_size'];
		}

		return $data;
	}

	/**
	 * Determines if a map should be displayed or not.
	 *
	 * @since 3.1.0
	 *
	 * @return bool
	 */
	public function should_display(): bool {
		if ( ! $this->has_service() ) {
			return false;
		}

		$hide_until_searched = $this->get_view()->settings->get( 'hide_until_searched' );

		if (
			$this->has_entry()
			&& ! empty( $hide_until_searched )
			&& ! Utils::is_search()
		) {
			return false;
		}

		return true;
	}

	/**
	 * Gets the HTML for the map.
	 *
	 * @since 3.1.0
	 *
	 * @return string|null
	 */
	public function get_html(): ?string {
		if ( ! $this->get_service() ) {
			do_action( 'gravityview_log_debug', 'HTML not available for maps without a proper service.', $this );

			return null;
		}

		ob_start();
		?>
		<div
			id="<?php echo esc_attr( $this->get_html_id() ); ?>"
			class="<?php echo esc_attr( $this->get_html_classes() ); ?>"
			data-entryid="<?php echo esc_attr( $this->get_entry_id() ); ?>"
			data-gk-map="<?php echo esc_attr( $this->esc_json_attr( $this->get_canvas_data() ) ); ?>"
			data-gk-map-settings="<?php echo esc_attr( $this->esc_json_attr( $this->get_settings() ) ); ?>"
		></div>
		<?php

		$html = ob_get_clean();

		return $html;
	}

	/**
	 * Adds this map to the list of existing maps.
	 *
	 * @since 3.1.0
	 *
	 */
	public function add_into_existing(): void {
		static::$existing_ids[ $this->get_id() ] = $this;
	}

	/**
	 * Checks if the current View is a Map template or has any map object configured so that we could speed up decisions on the frontend.
	 *
	 * This method will cache its result since this is an intense operation to be running all the time.
	 *
	 * @since 3.1.3
	 *
	 * @param bool $retest Whether to retest the map for the current view.
	 *
	 * @return bool
	 */
	public function will_render( bool $retest = false ): bool {
		$view_id = $this->get_view()->ID;
		if ( $retest && ! empty( $this->will_render ) ) {
			return $this->will_render;
		}

		if ( 'map' === get_post_meta( $view_id, '_gravityview_directory_template', true ) ) {
			$this->will_render = true;
			return $this->will_render;
		}

		$check_objects_callback = static function( $objects, string $field_id ): bool {
			if ( empty( $objects ) ) {
				return false;
			}

			if ( ! is_array( $objects ) ) {
				return false;
			}

			foreach ( $objects as $areas ) {
				if ( ! is_array( $areas ) ) {
					continue;
				}

				foreach ( $areas as $object ) {
					if ( $field_id === $object['id'] ) {
						return true;
					}
				}
			}

			return false;
		};

		$widgets = get_post_meta( $view_id, '_gravityview_directory_widgets', true );
		if ( $check_objects_callback( $widgets ?? null, 'map' ) ) {
			$this->will_render = true;
			return $this->will_render;
		}

		$fields = get_post_meta( $view_id, '_gravityview_directory_fields', true );
		if ( $check_objects_callback( $fields ?? null, 'entry_map' ) ) {
			$this->will_render = true;
			return $this->will_render;
		}

		$this->will_render = false;
		return $this->will_render;
	}

	/**
	 * Renders the map.
	 *
	 * @since 3.1.0
	 *
	 * @return void
	 */
	public function render(): void {
		if ( ! $this->should_display() ) {
			return;
		}

		$html = $this->get_html();

		if ( $html ) {
			$this->add_into_existing();
		}

		echo $html;
	}
}
