<?php

namespace GravityKit\GravityMaps\Views;

use GravityKit\GravityMaps\AbstractSingleton;
use GravityKit\GravityMaps\Loader;
use GravityKit\GravityMaps\Template\Map as Template_Map;
use GravityKit\GravityMaps\Template_Map_Default;
use GravityKit\GravityMaps\Template_Preset_Business_Map;
use GV\Request;
use GV\Template_Context;
use GV\View;

/**
 * Class Controller
 *
 * @since   3.0
 *
 * @package GravityKit\GravityMaps\Views
 */
class Controller extends AbstractSingleton {
	/**
	 * @inheritDoc
	 */
	protected function register(): void {
		$this->hooks();
		$this->boot();
	}

	/**
	 * Boots related classes that need to be initialized.
	 *
	 * @since 3.0
	 */
	protected function boot(): void {
		CanvasPositionModifier::instance();
	}

	/**
	 * Configures hooks.
	 *
	 * @since 3.0
	 */
	protected function hooks(): void {
		$this->add_filters();
		$this->add_actions();
	}

	/**
	 * Adds filters.
	 *
	 * @since 3.0
	 */
	protected function add_filters(): void {
		add_filter( 'gravityview/template/view/class', [ $this, 'filter_modify_view_class' ], 15, 3 );
		add_filter( 'gravityview_template_paths', [ $this, 'add_template_path' ] );
		add_filter( 'gravityview/template/fields_template_paths', [ $this, 'add_template_path' ] );

		add_action( 'gk/gravitymaps/render/map-canvas', [ $this, 'render_map' ], 10, 2 );
		add_action( 'gravityview/view/get', [ $this, 'preload_map_service' ], 25, 1 );
	}

	/**
	 * Adds actions.
	 *
	 * @since 3.0
	 */
	protected function add_actions(): void {
		// Register the Maps View template type to GV core.
		add_action( 'init', [ $this, 'register_templates' ], 20 );
	}

	/**
	 * Modifies the View Class to use the Map Class.
	 *
	 * @since 3.0
	 *
	 * @param string  $class   The chosen class - Default: {@see \GV\View_Table_Template}
	 * @param View    $view    The view about to be rendered.
	 * @param Request $request The associated request.
	 *
	 * @return string
	 */
	public function filter_modify_view_class( $class, $view, $request ): string {
		if ( Map::$slug !== $view->settings->get( 'template' ) ) {
			return $class;
		}

		return Map::class;
	}

	/**
	 * Include this extension templates path.
	 *
	 * @since 3.0
	 *
	 * @param array $file_paths List of template paths ordered
	 *
	 * @return array
	 */
	public function add_template_path( $file_paths ): array {
		// Index 100 is the default GravityView template path.
		$file_paths[133] = plugin_dir_path( Loader::instance()->path ) . 'templates';

		return $file_paths;
	}

	/**
	 * Register the Maps View template type to GV core
	 *
	 * @since 3.0
	 *
	 * @return void
	 */
	public function register_templates(): void {
		// When GravityView is enabled but not active due to version mismatch, the class will not exist.
		if ( ! class_exists( '\GravityKit\GravityMaps\Template_Map_Default' ) ) {
			return;
		}

		new Template_Map_Default;
		new Template_Preset_Business_Map;
	}


	/**
	 * Output the map placeholder HTML
	 * If entry is defined, add the entry ID to the <div> tag to allow JS logic to render the entry marker only
	 *
	 * @since 1.6.2 added $context parameter
	 * @since 3.1.0 moved from Render_Map to here.
	 *
	 * @param array|null              $entry   Gravity Forms entry object
	 * @param string|Template_Context $context Current context, if set. Otherwise, empty string.
	 *
	 * @return void
	 */
	public function render_map( $entry = null, $context = '' ) {
		if ( ! empty( $entry ) ) {
			$map = Template_Map::from_entry_id( $entry['id'] );
		} else {
			$map = Template_Map::from_view( empty( $context ) ? gravityview_get_view_id() : $context->view );
		}

		// If this map has a service (Google Maps, Leaflet, etc), enqueue the assets.
		if ( $map->has_service() ) {
			$map->get_service()->hook_assets();
		}

		$map->render();
	}

	/**
	 * Preload the map service for the view.
	 *
	 * This method will unhook and re-hook itself to avoid an infinite loop.
	 *
	 * @since 3.1.3
	 *
	 * @param View $view The view.
	 *
	 * @return void
	 */
	public function preload_map_service( $view ): void {
		if ( empty( $view ) ) {
			return;
		}

		remove_action( 'gravityview/view/get', [ $this, 'preload_map_service' ], 25 );

		$map = Template_Map::from_view( $view, false );

		// If this map has a service (Google Maps, Leaflet, etc), enqueue the assets.
		if ( $map->will_render() ) {
			$map->setup_service();

			if ( $map->has_service() ) {
				$map->get_service()->hook_assets();
			}
		}

		add_action( 'gravityview/view/get', [ $this, 'preload_map_service' ], 25, 1 );
	}
}