<?php
/*
Plugin Name: US Map
Plugin URI: https://codecanyon.net/item/interactive-us-map-wordpress-plugin/10359489
Description: Customize each state (colors, link, etc) through the map dashboard and use the shortcode in your page.
Version: 2.7
Author: WP Map Plugins
Author URI: https://www.wpmapplugins.com/
Text Domain: us-map
Domain Path: /languages
*/

declare(strict_types=1);

namespace USMap;

use USMap\USMap;

if (!defined('USMAP_VERSION')) {
    define('USMAP_VERSION', '2.6');
}

if (!defined('USMAP_DIR')) {
    define('USMAP_DIR', plugin_dir_path(__FILE__));
}

if (!defined('USMAP_URL')) {
    define('USMAP_URL', plugin_dir_url(__FILE__));
}

(new USMap())->init();

class USMap {

    const PLUGIN_NAME = 'US Map';

    private $options = null;

    public function init() {
        $this->initActions();
        $this->initShortcodes();
        $this->initOptions();
    }

    private function initActions() {
        if( !function_exists('wp_get_current_user') ) {
            include(ABSPATH . "wp-includes/pluggable.php");
        }
        add_action( 'admin_menu', array($this, 'addOptionsPage') );
        add_action( 'admin_footer', array($this, 'addJsConfigInFooter') );
        add_action( 'wp_footer', array($this, 'addSpanTag') );
        add_action( 'admin_enqueue_scripts', array($this, 'initAdminScript') );
        add_action( 'init', array($this, 'loadTextdomain') );
    }

    private function initShortcodes() {
        add_shortcode('us_map', array($this, 'USMapShortcode'));
    }

    private function initOptions() {
        $defaultOptions = $this->getDefaultOptions();
        $this->options  = get_option('us_map');

        if (current_user_can( 'manage_options' )){
            $this->updateOptions($defaultOptions);
        }

        if (!is_array($this->options)) {
            $this->options = $defaultOptions;
        }

        $this->prepareOptionsListForTpl();
    }

    public function addJsConfigInFooter() {
        echo wp_kses_post('<span id="tipusmap" style="margin-top:-32px"></span>');
        include __DIR__ . "/includes/js-config.php";
    }

    public function addOptionsPage() {
        add_menu_page(
            USMap::PLUGIN_NAME,
            USMap::PLUGIN_NAME,
            'manage_options',
            'us-map',
            array($this, 'optionsScreen'),
            USMAP_URL . 'public/images/map-icon.png'
        );
    }

    /**
     * @return array
     */
    private function getDefaultOptions() {
        $default = array(
            'usbrdrclr'    => '#6B8B9E',
            'usshowvisns'  => 1,
            'usvisns'      => '#666666',
            'usvisnshover' => '#113e6b',
            'usshowlakes'  => 1,
            'uslakesfill'  => '#66CCFF',
            'usshowalaska' => 1,
            'usshowhawaii' => 1,
            'usshowdc'     => 1,
            'uscallouts'   => 1,
        );

        $areas = array(
            'ALABAMA', 'ALASKA', 'ARIZONA', 'ARKANSAS', 'CALIFORNIA', 'COLORADO', 'CONNECTICUT', 'DELAWARE', 'FLORIDA', 'GEORGIA', 'HAWAII', 'IDAHO', 'ILLINOIS', 'INDIANA', 'IOWA', 'KANSAS', 'KENTUCKY', 'LOUISIANA', 'MAINE', 'MARYLAND', 'MASSACHUSETTS', 'MICHIGAN', 'MINNESOTA', 'MISSISSIPPI', 'MISSOURI', 'MONTANA', 'NEBRASKA', 'NEVADA', 'NEW HAMPSHIRE', 'NEW JERSEY', 'NEW MEXICO', 'NEW YORK', 'NORTH CAROLINA', 'NORTH DAKOTA', 'OHIO', 'OKLAHOMA', 'OREGON', 'PENNSYLVANIA', 'RHODE ISLAND', 'SOUTH CAROLINA', 'SOUTH DAKOTA', 'TENNESSEE', 'TEXAS', 'UTAH', 'VERMONT', 'VIRGINIA', 'WASHINGTON', 'WEST VIRGINIA', 'WISCONSIN', 'WYOMING', 'WASHINGTON DC'
        );

        foreach ($areas as $k => $area) {
            $default['upclr_' . ($k + 1)]  = '#E0F3FF';
            $default['ovrclr_' . ($k + 1)] = '#8FBEE8';
            $default['dwnclr_' . ($k + 1)] = '#477CB2';
            $default['url_' . ($k + 1)]    = '';
            $default['turl_' . ($k + 1)]   = '_self';
            $default['info_' . ($k + 1)]   = $area;
            $default['enbl_' . ($k + 1)]   = 1;
        }

        return $default;
    }

    private function updateOptions(array $defaultOptions) {
        if (isset($_POST['us_map']) && isset($_POST['submit-clrs'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['upclr_' . $i]  = $_POST['upclr_all'];
                $_POST['ovrclr_' . $i] = $_POST['ovrclr_all'];
                $_POST['dwnclr_' . $i] = $_POST['dwnclr_all'];

                $i++;
            }

            update_option('us_map',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map']) && isset($_POST['submit-url'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['url_' . $i]  = $_POST['url_all'];
                $_POST['turl_' . $i] = $_POST['turl_all'];

                $i++;
            }

            update_option('us_map',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map']) && isset($_POST['submit-info'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['info_' . $i] = $_POST['info_all'];

                $i++;
            }

            update_option('us_map',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map']) && !isset($_POST['preview_map'])) {
            update_option('us_map',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map']) && isset($_POST['restore_default'])) {
            update_option('us_map', $defaultOptions);

            $this->options = $defaultOptions;
        }
    }

    private function prepareOptionsListForTpl() {
        $this->options['prepared_list'] = array();
        $i                              = 1;
        while (isset($this->options['url_' . $i])) {
            $this->options['prepared_list'][] = array(
                'index'  => $i,
                'info_'  => $this->options['info_' . $i],
                'url'    => $this->options['url_' . $i],
                'turl'   => $this->options['turl_' . $i],
                'upclr'  => $this->options['upclr_' . $i],
                'ovrclr' => $this->options['ovrclr_' . $i],
                'dwnclr' => $this->options['dwnclr_' . $i],
                'enbl'   => isset($this->options['enbl_' . $i]),
            );

            $i++;
        }
    }

    public function USMapShortcode() {
        wp_enqueue_style('us-map-style-frontend', USMAP_URL . 'public/css/map-style.css', false, '1.0', 'all');
        wp_enqueue_script('us-map-interact', USMAP_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);

        ob_start();

        include __DIR__ . "/includes/map.php";
        include __DIR__ . "/includes/js-config.php";

        return ob_get_clean();
    }

    public function optionsScreen() {
        include __DIR__ . "/includes/admin.php";
    }

    public function initAdminScript() {
        if ( current_user_can( 'manage_options') && ( esc_attr(isset($_GET['page'])) && esc_attr($_GET['page']) == 'us-map') ):
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_style('thickbox');
            wp_enqueue_script('thickbox');
            wp_enqueue_script('media-upload');

            wp_enqueue_style('us-map-dashboard-style', USMAP_URL . 'public/css/dashboard-style.css', false, '1.0', 'all');
            wp_enqueue_style('us-map-style', USMAP_URL . 'public/css/map-style.css', false, '1.0', 'all');
            wp_enqueue_style('wp-tinyeditor', USMAP_URL . 'public/css/tinyeditor.css', false, '1.0', 'all');

            wp_enqueue_script('us-map-interact', USMAP_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);
            wp_enqueue_script('us-map-tiny.editor', USMAP_URL . 'public/js/editor/tinymce.min.js', 10, '1.0', true);
            wp_enqueue_script('us-map-script', USMAP_URL . 'public/js/editor/scripts.js', array('wp-color-picker'), false, true);
        endif;
    }

    public function addSpanTag()
    {
        echo wp_kses_post('<span id="tipusmap"></span>');
    }
    
    public function loadTextdomain() {
        load_plugin_textdomain( 'us-map', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }
}
