<?php
/*
Plugin Name: US Map III
Plugin URI: https://codecanyon.net/item/interactive-us-map-wordpress-plugin/10359489
Description: Customize each state (colors, link, etc) through the map dashboard and use the shortcode in your page.
Version: 2.7
Author: WP Map Plugins
Author URI: https://www.wpmapplugins.com/
Text Domain: us-map-iii
Domain Path: /languages
*/

declare(strict_types=1);

namespace USMap_III;

use USMap_III\USMap_III;

if (!defined('USMAP_III_VERSION')) {
    define('USMAP_III_VERSION', '2.6');
}

if (!defined('USMAP_III_DIR')) {
    define('USMAP_III_DIR', plugin_dir_path(__FILE__));
}

if (!defined('USMAP_III_URL')) {
    define('USMAP_III_URL', plugin_dir_url(__FILE__));
}

(new USMap_III())->init();

class USMap_III {

    const PLUGIN_NAME = 'US Map III';

    private $options = null;

    public function init() {
        $this->initActions();
        $this->initShortcodes();
        $this->initOptions();
    }

    private function initActions() {
        if( !function_exists('wp_get_current_user') ) {
            include(ABSPATH . "wp-includes/pluggable.php");
        }
        add_action( 'admin_menu', array($this, 'addOptionsPage') );
        add_action( 'admin_footer', array($this, 'addJsConfigInFooter') );
        add_action( 'wp_footer', array($this, 'addSpanTag') );
        add_action( 'admin_enqueue_scripts', array($this, 'initAdminScript') );
        add_action( 'init', array($this, 'loadTextdomain') );
    }

    private function initShortcodes() {
        add_shortcode('us_map_iii', array($this, 'USMap_IIIShortcode'));
    }

    private function initOptions() {
        $defaultOptions = $this->getDefaultOptions();
        $this->options  = get_option('us_map_iii');

        if (current_user_can( 'manage_options' )){
            $this->updateOptions($defaultOptions);
        }

        if (!is_array($this->options)) {
            $this->options = $defaultOptions;
        }

        $this->prepareOptionsListForTpl();
    }

    public function addJsConfigInFooter() {
        echo wp_kses_post('<span id="tipusmap-iii" style="margin-top:-32px"></span>');
        include __DIR__ . "/includes/js-config.php";
    }

    public function addOptionsPage() {
        add_menu_page(
            USMap_III::PLUGIN_NAME,
            USMap_III::PLUGIN_NAME,
            'manage_options',
            'us-map-iii',
            array($this, 'optionsScreen'),
            USMAP_III_URL . 'public/images/map-icon.png'
        );
    }

    /**
     * @return array
     */
    private function getDefaultOptions() {
        $default = array(
            'usbrdrclr_iii'    => '#6B8B9E',
            'usshowvisns_iii'  => 1,
            'usvisns_iii'      => '#666666',
            'usvisnshover_iii' => '#113e6b',
            'usshowlakes_iii'  => 1,
            'uslakesfill_iii'  => '#66CCFF',
            'usshowalaska_iii' => 1,
            'usshowhawaii_iii' => 1,
            'usshowdc_iii'     => 1,
            'uscallouts_iii'   => 1,
        );

        $areas = array(
            'ALABAMA', 'ALASKA', 'ARIZONA', 'ARKANSAS', 'CALIFORNIA', 'COLORADO', 'CONNECTICUT', 'DELAWARE', 'FLORIDA', 'GEORGIA', 'HAWAII', 'IDAHO', 'ILLINOIS', 'INDIANA', 'IOWA', 'KANSAS', 'KENTUCKY', 'LOUISIANA', 'MAINE', 'MARYLAND', 'MASSACHUSETTS', 'MICHIGAN', 'MINNESOTA', 'MISSISSIPPI', 'MISSOURI', 'MONTANA', 'NEBRASKA', 'NEVADA', 'NEW HAMPSHIRE', 'NEW JERSEY', 'NEW MEXICO', 'NEW YORK', 'NORTH CAROLINA', 'NORTH DAKOTA', 'OHIO', 'OKLAHOMA', 'OREGON', 'PENNSYLVANIA', 'RHODE ISLAND', 'SOUTH CAROLINA', 'SOUTH DAKOTA', 'TENNESSEE', 'TEXAS', 'UTAH', 'VERMONT', 'VIRGINIA', 'WASHINGTON', 'WEST VIRGINIA', 'WISCONSIN', 'WYOMING', 'WASHINGTON DC'
        );

        foreach ($areas as $k => $area) {
            $default['upclr_' . ($k + 1)]  = '#E0F3FF';
            $default['ovrclr_' . ($k + 1)] = '#8FBEE8';
            $default['dwnclr_' . ($k + 1)] = '#477CB2';
            $default['url_' . ($k + 1)]    = '';
            $default['turl_' . ($k + 1)]   = '_self';
            $default['info_' . ($k + 1)]   = $area;
            $default['enbl_' . ($k + 1)]   = 1;
        }

        return $default;
    }

    private function updateOptions(array $defaultOptions) {
        if (isset($_POST['us_map_iii']) && isset($_POST['submit-clrs'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['upclr_' . $i]  = $_POST['upclr_all'];
                $_POST['ovrclr_' . $i] = $_POST['ovrclr_all'];
                $_POST['dwnclr_' . $i] = $_POST['dwnclr_all'];

                $i++;
            }

            update_option('us_map_iii',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map_iii']) && isset($_POST['submit-url'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['url_' . $i]  = $_POST['url_all'];
                $_POST['turl_' . $i] = $_POST['turl_all'];

                $i++;
            }

            update_option('us_map_iii',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map_iii']) && isset($_POST['submit-info'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['info_' . $i] = $_POST['info_all'];

                $i++;
            }

            update_option('us_map_iii',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map_iii']) && !isset($_POST['preview_map'])) {
            update_option('us_map_iii',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['us_map_iii']) && isset($_POST['restore_default'])) {
            update_option('us_map_iii', $defaultOptions);

            $this->options = $defaultOptions;
        }
    }

    private function prepareOptionsListForTpl() {
        $this->options['prepared_list'] = array();
        $i                              = 1;
        while (isset($this->options['url_' . $i])) {
            $this->options['prepared_list'][] = array(
                'index'  => $i,
                'info_'  => $this->options['info_' . $i],
                'url'    => $this->options['url_' . $i],
                'turl'   => $this->options['turl_' . $i],
                'upclr'  => $this->options['upclr_' . $i],
                'ovrclr' => $this->options['ovrclr_' . $i],
                'dwnclr' => $this->options['dwnclr_' . $i],
                'enbl'   => isset($this->options['enbl_' . $i]),
            );

            $i++;
        }
    }

    public function USMap_IIIShortcode() {
        wp_enqueue_style('us-map-iii-style-frontend', USMAP_III_URL . 'public/css/map-style.css', false, '1.0', 'all');
        wp_enqueue_script('us-map-iii-interact', USMAP_III_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);

        ob_start();

        include __DIR__ . "/includes/map.php";
        include __DIR__ . "/includes/js-config.php";

        return ob_get_clean();
    }

    public function optionsScreen() {
        include __DIR__ . "/includes/admin.php";
    }

    public function initAdminScript() {
        if ( current_user_can( 'manage_options') && ( esc_attr(isset($_GET['page'])) && esc_attr($_GET['page']) == 'us-map-iii') ):
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_style('thickbox');
            wp_enqueue_script('thickbox');
            wp_enqueue_script('media-upload');

            wp_enqueue_style('us-map-iii-dashboard-style', USMAP_III_URL . 'public/css/dashboard-style.css', false, '1.0', 'all');
            wp_enqueue_style('us-map-iii-style', USMAP_III_URL . 'public/css/map-style.css', false, '1.0', 'all');
            wp_enqueue_style('wp-tinyeditor', USMAP_III_URL . 'public/css/tinyeditor.css', false, '1.0', 'all');

            wp_enqueue_script('us-map-iii-interact', USMAP_III_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);
            wp_enqueue_script('us-map-iii-tiny.editor', USMAP_III_URL . 'public/js/editor/tinymce.min.js', 10, '1.0', true);
            wp_enqueue_script('us-map-iii-script', USMAP_III_URL . 'public/js/editor/scripts.js', array('wp-color-picker'), false, true);
        endif;
    }

    public function addSpanTag()
    {
        echo wp_kses_post('<span id="tipusmap-iii"></span>');
    }
    
    public function loadTextdomain() {
        load_plugin_textdomain( 'us-map-iii', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }
}
