import JetIcon from "components/jet-icon";

import {
	getAvailableImageSizes,
	getImageSizeName,
	getImageURLBySizeName
} from "utils/imageSizes";

const className = 'jet-guten-banner';

const {
	registerBlockType,
} = wp.blocks;

const {
	__
} = wp.i18n;

const {
	Dashicon,
	IconButton,
	Toolbar,
	PanelBody,
	RangeControl,
	SelectControl,
	FontSizePicker,
	Button
} = wp.components;

const {
	BlockControls,
	InspectorControls,
	AlignmentToolbar,
	RichText,
	MediaUpload,
	PanelColorSettings,
	ColorPalette,
	URLInput
} = wp.editor;

const EFFECTS = [{
		value: "lily",
		label: __("Lily")
	},
	{
		value: "sadie",
		label: __("Sadie")
	},
	{
		value: "layla",
		label: __("Layla")
	},
	{
		value: "oscar",
		label: __("Oscar")
	},
	{
		value: "marley",
		label: __("Marley")
	},
	{
		value: "ruby",
		label: __("Ruby")
	},
	{
		value: "roxy",
		label: __("Roxy")
	},
	{
		value: "bubba",
		label: __("Bubba")
	},
	{
		value: "romeo",
		label: __("Romeo")
	},
	{
		value: "sarah",
		label: __("Sarah")
	},
	{
		value: "chico",
		label: __("Chico")
	}
];

const FONT_SIZES = [{
		name: 'small',
		shortName: 'S',
		size: 14,
	},
	{
		name: 'regular',
		shortName: 'M',
		size: 16,
	},
	{
		name: 'large',
		shortName: 'L',
		size: 36,
	},
	{
		name: 'larger',
		shortName: 'XL',
		size: 48,
	},
];

let imageSizeName;

registerBlockType("jet-guten/banner", {
	title: __("Banner"),
	icon: <JetIcon icon="banner" />,
	category: "layout",
	attributes: {
		effect: {
			type: "string",
			default: "lily"
		},
		title: {
			type: "array",
			source: "children",
			selector: `.${className}__title`
		},
		text: {
			type: "array",
			source: "children",
			selector: `.${className}__text`
		},
		imageID: {
			type: "number"
		},
		imageURL: {
			type: "string",
			source: "attribute",
			selector: `img.${className}__img`,
			attribute: "src"
		},
		availableImageSizes: {
			type: "array",
			default: []
		},
		link: {
			type: "string",
			source: "attribute",
			selector: `a.${className}__link`,
			attribute: "href"
		},
		titleNodeName: {
			type: 'string',
			default: 'H2',
		},
		titleColor: {
			type: "string"
		},
		titleFontSize: {
			type: 'number',
		},
		textColor: {
			type: "string"
		},
		textFontSize: {
			type: 'number',
		},
		overlayColor: {
			type: "string"
		},
		overlayOpacity: {
			type: "number",
			default: 30
		},
		contentAlign: {
			type: "string",
			default: "center"
		}
	},
	edit: props => {
		const { setAttributes, isSelected } = props;
		const {
			effect,
			title,
			text,
			imageID,
			imageURL,
			availableImageSizes,
			link,
			titleNodeName,
			titleColor,
			titleFontSize,
			textColor,
			textFontSize,
			overlayColor,
			overlayOpacity,
			contentAlign
		} = props.attributes;

		if (!availableImageSizes.length) {
			getAvailableImageSizes(imageID, newAvailableImageSizes => {
				setAttributes({ availableImageSizes: newAvailableImageSizes })
			});
		}

		if (!imageSizeName) {
			imageSizeName = getImageSizeName(imageURL, availableImageSizes);
		}

		return [
			isSelected && (
				<BlockControls key="controls">
					<AlignmentToolbar
						value={contentAlign}
						onChange={newAlign =>
							setAttributes({ contentAlign: newAlign })
						}
					/>
					<Toolbar>
						<MediaUpload
							onSelect={media =>
								{
									getAvailableImageSizes(media.id, newAvailableImageSizes => {
										setAttributes({
											availableImageSizes: newAvailableImageSizes,
											imageURL: getImageURLBySizeName(imageSizeName, newAvailableImageSizes),
											imageID: media.id
										})
									});
								}
							}
							type="image"
							value={imageID}
							render={({ open }) => (
								<IconButton
									className="components-toolbar__control"
									label={__("Edit image")}
									icon="edit"
									onClick={open}
								/>
							)}
						/>
					</Toolbar>
				</BlockControls>
			),
			isSelected && imageURL && (
				<InspectorControls key={"inspector"}>
					<PanelBody
						title={__("General Settings")}
						initialOpen={false}
					>
						<SelectControl
							label={ __( 'Image Size' ) }
							value={imageURL}
							options={availableImageSizes.map(({ value, label }) => ({
								value: value,
								label: label
							}))}
							onChange={ newImageURL => {
								imageSizeName = getImageSizeName(newImageURL, availableImageSizes);
								setAttributes({ imageURL: newImageURL });
							}}
						/>
						<SelectControl
							label={ __( 'Effect' ) }
							value={effect}
							options={EFFECTS.map(({ value, label }) => ({
								value: value,
								label: label
							}))}
							onChange={newEffect => {
								setAttributes({ effect: newEffect });
							}}
						/>
					</PanelBody>
					<PanelBody
						title={ __( 'Title Settings' ) }
						initialOpen={false}
					>
						<p>{ __( 'Heading Tag' ) }</p>
						<Toolbar
							controls={ '123456'.split( '' ).map( ( level ) => ( {
								icon: 'heading',
								isActive: 'H' + level === titleNodeName,
								onClick: () => setAttributes( { titleNodeName: 'H' + level } ),
								subscript: level,
							} ) ) }
						/>
						<p>{ __( 'Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ titleFontSize }
							onChange={fontSizeValue =>
								setAttributes({ titleFontSize: fontSizeValue })
							}
						/>
						<PanelColorSettings
							title={__("Color")}
							colorSettings={ [
								{
									value: titleColor,
									onChange: function( colorValue ) {
										setAttributes({
											titleColor: colorValue
										})
									},
									label: __( 'Title Color' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
					<PanelBody
						title={ __( 'Description Settings' ) }
						initialOpen={false}
					>
						<p>{ __( 'Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ textFontSize }
							onChange={fontSizeValue =>
								setAttributes({ textFontSize: fontSizeValue })
							}
						/>
						<PanelColorSettings
							title={__("Color")}
							colorSettings={ [
								{
									value: textColor,
									onChange: function( colorValue ) {
										setAttributes({
											textColor: colorValue
										})
									},
									label: __( 'Text Color' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
					<PanelBody
						title={__("Overlay Settings")}
						initialOpen={false}
					>
						<RangeControl
							label={__("Overlay Opacity")}
							value={overlayOpacity}
							onChange={opacityValue =>
								setAttributes({
									overlayOpacity: opacityValue
								})
							}
							min={0}
							max={100}
							step={1}
						/>
						<PanelColorSettings
							title={__("Color")}
							colorSettings={ [
								{
									value: overlayColor,
									onChange: function( colorValue ) {
										setAttributes({
											overlayColor: colorValue
										})
									},
									label: __( 'Overlay Color' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
				</InspectorControls>
			),
			!imageURL && (
				<div class={className}>
					<MediaUpload
						onSelect={media =>
							{
								getAvailableImageSizes(media.id, newAvailableImageSizes => {
									setAttributes({
										availableImageSizes: newAvailableImageSizes,
										imageURL: getImageURLBySizeName(imageSizeName, newAvailableImageSizes),
										imageID: media.id
									})
								});
							}
						}
						type="image"
						value={imageID}
						render={({ open }) => (
							<Button
								className={"button button-large"}
								onClick={open}
							>
								{__("Upload Image")}
							</Button>
						)}
					/>
				</div>
			),
			imageURL && (
				<figure
					className={`${className} ${className}-${effect}`}
				>
					<div
						className={`${className}__overlay`}
						style={{
							backgroundColor: overlayColor,
							opacity: overlayOpacity / 100
						}}
					/>
					<img className={`${className}__img`} src={imageURL} />
					<figcaption class={`${className}__content`}>
						<div class={`${className}__content-wrap`}>
							<div className={`${className}__title`}>
								<RichText
									tagName={ titleNodeName.toLowerCase() }
									className={`${className}__title__inner`}
									placeholder={__("Write banner title…")}
									isSelected={false}
									value={title}
									onChange={titleValue =>
										setAttributes({
											title: titleValue
										})
									}
									style={{
										color: titleColor,
										fontSize: titleFontSize ? titleFontSize + 'px' : undefined,
										textAlign: contentAlign
									}}
								/>
							</div>
							<div className={`${className}__text`}>
								<RichText
									tagName="p"
									className={`${className}__text__inner`}
									placeholder={__("Write banner text…")}
									isSelected={false}
									value={text}
									onChange={textValue =>
										setAttributes({
											text: textValue
										})
									}
									style={{
										color: textColor,
										fontSize: textFontSize ? textFontSize + 'px' : undefined,
										textAlign: contentAlign
									}}
								/>
							</div>
						</div>
					</figcaption>
				</figure>
			),
			isSelected &&
				imageURL && (
					<form
						className="jet-guten-editor__inline-link"
						onSubmit={event => event.preventDefault()}
					>
						<Dashicon
							className="jet-guten-editor__inline-link__icon"
							icon="admin-links"
						/>
						<URLInput
							autoFocus={false}
							value={link}
							onChange={value => setAttributes({ link: value })}
						/>
					</form>
				)
		];
	},
	save: props => {
		const {
			attributes: {
				effect,
				title,
				text,
				link,
				imageURL,
				titleNodeName,
				titleColor,
				titleFontSize,
				textColor,
				textFontSize,
				overlayColor,
				overlayOpacity,
				contentAlign
			}
		} = props;

		return (
			<figure
				className={`${className} ${className}-${effect}`}
			>
				<div
					className={`${className}__overlay`}
					style={{
						backgroundColor: overlayColor,
						opacity: overlayOpacity / 100
					}}
				/>
				<img className={`${className}__img`} src={imageURL} />
				{link && (
					<a
						className={`${className}__link`}
						href={link}
					></a>
				)}
				<figcaption class={`${className}__content`}>
					<div class={`${className}__content-wrap`}>
						<RichText.Content
							className={`${className}__title`}
							tagName={ titleNodeName.toLowerCase() }
							style={ {
								color: titleColor,
								fontSize: titleFontSize ? titleFontSize + 'px' : undefined,
								textAlign: contentAlign
							} }
							value={ title }
						/>
						<p
							className={`${className}__text`}
							style={{
								color: textColor,
								fontSize: textFontSize ? textFontSize + 'px' : undefined,
								textAlign: contentAlign
							}}
						>{text}</p>
					</div>
				</figcaption>
			</figure>
		);
	}
});