import JetIcon from "components/jet-icon";

const className = 'jet-guten-circle-progress';

const {
	registerBlockType,
} = wp.blocks;

const {
	__
} = wp.i18n;

const {
	Toolbar,
	PanelBody,
	RangeControl,
	TextControl,
	FontSizePicker,
	SelectControl
} = wp.components;

const {
	InspectorControls,
	RichText,
	PanelColorSettings,
	ColorPalette
} = wp.editor;

const FONT_SIZES = [
	{
		name: 'small',
		shortName: 'S',
		size: 14,
	},
	{
		name: 'regular',
		shortName: 'M',
		size: 16,
	},
	{
		name: 'large',
		shortName: 'L',
		size: 36,
	},
	{
		name: 'larger',
		shortName: 'XL',
		size: 48,
	}
];

registerBlockType("jet-guten/circle-progress", {
	title: __("Circle Progress"),
	icon: <JetIcon icon="circle-progress" />,
	category: "layout",
	attributes: {
		valueNumber: {
			type: "number",
			default: 50
		},
		maxValue: {
			type: "number",
			default: 100
		},
		valueNumberPrefix: {
			type: "string",
			source: "html",
			selector: `.${className}__value__prefix`,
			default: ""
		},
		valueNumberSuffix: {
			type: "string",
			source: "html",
			selector: `.${className}__value__suffix`,
			default: ""
		},
		circleSize: {
			type: "number",
			default: 180
		},
		stokeColor: {
			type: "string",
			default: "#6ec1e4"
		},
		stokeWidth: {
			type: "number",
			default: 7
		},
		backgroundStokeColor: {
			type: "string",
			default: "#e6e9ec"
		},
		backgroundStokeWidth: {
			type: "number",
			default: 7
		},
		counterTitle: {
			type: "string",
			source: "html",
			selector: `.${className}__title`,
			default: ''
		},
		counterSubtitle: {
			type: "string",
			source: "html",
			selector: `.${className}__subtitle`,
			default: ''
		},
		circleFill: {
			type: "string",
			default: "#ffffff",
		},
		numberColor: {
			type: "string"
		},
		numberFontSize: {
			type: 'number',
		},
		prefixColor: {
			type: "string"
		},
		prefixFontSize: {
			type: 'number',
		},
		suffixColor: {
			type: "string"
		},
		suffixFontSize: {
			type: 'number',
		},
		valuePosition: {
			type: 'string',
			default: 'inside',
		},
		titleNodeName: {
			type: 'string',
			default: 'H3',
		},
		titleColor: {
			type: "string"
		},
		titleFontSize: {
			type: 'number',
		},
		subtitleColor: {
			type: "string"
		},
		subtitleFontSize: {
			type: 'number',
		},
	},
	edit: props => {
		const { setAttributes, isSelected } = props;
		const {
			valueNumber,
			maxValue,
			valueNumberPrefix,
			valueNumberSuffix,
			circleSize,
			stokeColor,
			stokeWidth,
			backgroundStokeColor,
			backgroundStokeWidth,
			counterTitle,
			counterSubtitle,
			numberColor,
			numberFontSize,
			prefixColor,
			prefixFontSize,
			suffixColor,
			suffixFontSize,
			titleNodeName,
			titleColor,
			titleFontSize,
			subtitleColor,
			subtitleFontSize,
			circleFill,
			valuePosition
		} = props.attributes;

		const maxStroke = Math.max(stokeWidth, backgroundStokeWidth),
			  radius = circleSize / 2 - maxStroke / 2,
			  center = circleSize / 2,
			  circumference = Math.round( 2 * Math.PI * radius ),
			  progress = Math.round( circumference * ( 1 - valueNumber / maxValue ) ),
			  xmlns = 'http://www.w3.org/2000/svg';

		return [
			isSelected && (
				<InspectorControls key={"inspector"}>
					<PanelBody
						title={__("General Settings")}
					>
						<RangeControl
							label={__("Value Number")}
							value={valueNumber}
							onChange={newValueNumber =>
								setAttributes({
									valueNumber: newValueNumber
								})
							}
							min={0}
							max={maxValue}
							step={1}
						/>
						<TextControl
							type="number"
							label={__("Max Value")}
							value={maxValue}
							onChange={newMaxValue =>
								setAttributes({
									maxValue: newMaxValue
								})
							}
						/>
						<TextControl
							type="text"
							label={__("Value Number Prefix")}
							value={valueNumberPrefix}
							placeholder={__("Prefix")}
							onChange={newValueNumberPrefix =>
								setAttributes({
									valueNumberPrefix: newValueNumberPrefix
								})
							}
						/>
						<TextControl
							type="text"
							label={__("Value Number Suffix")}
							value={valueNumberSuffix}
							placeholder={__("Suffix")}
							onChange={newValueNumberSuffix =>
								setAttributes({
									valueNumberSuffix: newValueNumberSuffix
								})
							}
						/>
					</PanelBody>
					<PanelBody
						title={__("Circle Settings")}
						initialOpen={false}
					>
						<RangeControl
							label={__("Circle Size")}
							value={circleSize}
							onChange={newCircleSize => {
								setAttributes({
									circleSize: newCircleSize
								})
								if (stokeWidth*2 > newCircleSize) {
									setAttributes({
										stokeWidth: Math.ceil(newCircleSize/2)
									})
								}
								if (backgroundStokeWidth*2 > newCircleSize) {
									setAttributes({
										backgroundStokeWidth: Math.ceil(newCircleSize/2)
									})
								}
							} }
							min={100}
							max={580}
							step={1}
						/>
						<RangeControl
							label={__("Line Width")}
							value={stokeWidth}
							onChange={newStokeWidth =>
								setAttributes({
									stokeWidth: newStokeWidth
								})
							}
							min={1}
							max={circleSize/2}
							step={1}
						/>
						<RangeControl
							label={__("Background Line Width")}
							value={backgroundStokeWidth}
							onChange={newBackgroundStokeWidth =>
								setAttributes({
									backgroundStokeWidth: newBackgroundStokeWidth
								})
							}
							min={1}
							max={circleSize/2}
							step={1}
						/>
						<PanelColorSettings
							title={__("Colors")}
							colorSettings={ [
								{
									value: stokeColor,
									onChange: function( colorValue ) {
										setAttributes({
											stokeColor: colorValue
										})
									},
									label: __( 'Line' ),
								},
								{
									value: backgroundStokeColor,
									onChange: function( colorValue ) {
										setAttributes({
											backgroundStokeColor: colorValue
										})
									},
									label: __( 'Background Line' ),
								},
								{
									value: circleFill,
									onChange: function( colorValue ) {
										setAttributes({
											circleFill: colorValue
										})
									},
									label: __( 'Circle Fill' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
					<PanelBody
						title={__("Value Settings")}
						initialOpen={false}
					>
						<SelectControl
							label={ __( 'Value Position' ) }
							value={ valuePosition }
							options={ [
								{
									value: 'inside',
									label: __( 'Inside progress circle' ),
								},
								{
									value: 'below',
									label: __( 'Below progress circle' ),
								},
							] }
							onChange={newvaluePosition =>
								setAttributes({
									valuePosition: newvaluePosition
								})
							}
						/>
						<p>{ __( 'Number Font Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ numberFontSize }
							onChange={newNumberFontSize =>
								setAttributes({
									numberFontSize: newNumberFontSize
								})
							}
						/>
						<p>{ __( 'Prefix Font Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ prefixFontSize }
							onChange={newPrefixFontSize =>
								setAttributes({
									prefixFontSize: newPrefixFontSize
								})
							}
						/>
						<p>{ __( 'Suffix Font Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ suffixFontSize }
							onChange={newSuffixFontSize =>
								setAttributes({
									suffixFontSize: newSuffixFontSize
								})
							}
						/>
						<PanelColorSettings
							title={__("Colors")}
							colorSettings={ [
								{
									value: numberColor,
									onChange: function( colorValue ) {
										setAttributes({
											numberColor: colorValue
										})
									},
									label: __( 'Number' ),
								},
								{
									value: prefixColor,
									onChange: function( colorValue ) {
										setAttributes({
											prefixColor: colorValue
										})
									},
									label: __( 'Prefix' ),
								},
								{
									value: suffixColor,
									onChange: function( colorValue ) {
										setAttributes({
											suffixColor: colorValue
										})
									},
									label: __( 'Suffix' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
					<PanelBody
						title={__("Label Settings")}
						initialOpen={false}
					>
						<p>{ __( 'Title Tag' ) }</p>
						<Toolbar
							controls={ '123456'.split( '' ).map( ( level ) => ( {
								icon: 'heading',
								isActive: 'H' + level === titleNodeName,
								onClick: () => setAttributes( { titleNodeName: 'H' + level } ),
								subscript: level,
							} ) ) }
						/>
						<p>{ __( 'Title Font Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ titleFontSize }
							onChange={newTitleFontSize =>
								setAttributes({
									titleFontSize: newTitleFontSize
								})
							}
						/>
						<p>{ __( 'Subtitle Font Size' ) }</p>
						<FontSizePicker
							fontSizes={ FONT_SIZES }
							value={ subtitleFontSize }
							onChange={newSubtitleFontSize =>
								setAttributes({
									subtitleFontSize: newSubtitleFontSize
								})
							}
						/>
						<PanelColorSettings
							title={__("Colors")}
							colorSettings={ [
								{
									value: titleColor,
									onChange: function( colorValue ) {
										setAttributes({
											titleColor: colorValue
										})
									},
									label: __( 'Title' ),
								},
								{
									value: subtitleColor,
									onChange: function( colorValue ) {
										setAttributes({
											subtitleColor: colorValue
										})
									},
									label: __( 'Subtitle' ),
								},
							] }
						>
						</PanelColorSettings>
					</PanelBody>
				</InspectorControls>
			),
			<div className={[ className, 'label-poition-' + valuePosition ].join( ' ' )}>
				<div className={`${className}__bar`}>
					<svg
						className={`${className}__circle`}
						width={circleSize}
						height={circleSize}
						xmlns={xmlns}
					>
						<circle
							className="jet-guten-circle-progress__meter"
							cx={center}
							cy={center}
							r={radius}
							stroke={backgroundStokeColor}
							stroke-width={backgroundStokeWidth}
							fill={circleFill}
						/>
						<circle
							className="jet-guten-circle-progress__value"
							cx={center}
							cy={center}
							r={radius}
							stroke={stokeColor}
							stroke-width={stokeWidth}
							data-value={valueNumber}
							stroke-dasharray={circumference}
							stroke-dashoffset={progress}
							fill="none"
						/>
					</svg>
					<div className={`${className}__value`}>
						<span
							className={`${className}__value__prefix`}
							style={{
								color: prefixColor,
								fontSize: prefixFontSize ? prefixFontSize + 'px' : undefined
							}}
						>{valueNumberPrefix}</span>
						<span
							className={`${className}__value__number`}
							style={{
								color: numberColor,
								fontSize: numberFontSize ? numberFontSize + 'px' : undefined
							}}
						>{valueNumber}</span>
						<span
							className={`${className}__value__suffix`}
							style={{
								color: suffixColor,
								fontSize: suffixFontSize ? suffixFontSize + 'px' : undefined
							}}
						>{valueNumberSuffix}</span>
					</div>
				</div>
				<div className={`${className}__content`}>
					<RichText
						tagName={ titleNodeName.toLowerCase() }
						className={`${className}__title`}
						isSelected={false}
						placeholder={__("Write title…")}
						value={counterTitle}
						onChange={newCounterTitle =>
							setAttributes({
								counterTitle: newCounterTitle
							})
						}
						style={{
							color: titleColor,
							fontSize: titleFontSize ? titleFontSize + 'px' : undefined
						}}
					/>
					<RichText
						tagName='div'
						className={`${className}__subtitle`}
						placeholder={__("Write subtitle…")}
						isSelected={false}
						value={counterSubtitle}
						onChange={newCounterSubtitle =>
							setAttributes({
								counterSubtitle: newCounterSubtitle
							})
						}
						style={{
							color: subtitleColor,
							fontSize: subtitleFontSize ? subtitleFontSize + 'px' : undefined
						}}
					/>
				</div>
			</div>
		];
	},
	save: props => {
		const {
			attributes: {
				valueNumber,
				maxValue,
				valueNumberPrefix,
				valueNumberSuffix,
				circleSize,
				stokeColor,
				stokeWidth,
				backgroundStokeColor,
				backgroundStokeWidth,
				counterTitle,
				counterSubtitle,
				numberColor,
				numberFontSize,
				prefixColor,
				prefixFontSize,
				suffixColor,
				suffixFontSize,
				titleNodeName,
				titleColor,
				titleFontSize,
				subtitleColor,
				subtitleFontSize,
				circleFill,
				valuePosition
			}
		} = props;

		const maxStroke = Math.max(stokeWidth, backgroundStokeWidth),
			  radius = circleSize / 2 - maxStroke / 2,
			  center = circleSize / 2,
			  circumference = Math.round( 2 * Math.PI * radius ),
			  progress = Math.round( circumference * ( 1 - valueNumber / maxValue ) ),
			  xmlns = 'http://www.w3.org/2000/svg';

		return (
			<div className={[ className, 'label-poition-' + valuePosition ].join( ' ' )}>
				<div className={`${className}__bar`}>
					<svg
						className={`${className}__circle`}
						width={circleSize}
						height={circleSize}
						xmlns={xmlns}
					>
						<circle
							className="jet-guten-circle-progress__meter"
							cx={center}
							cy={center}
							r={radius}
							stroke={backgroundStokeColor}
							stroke-width={backgroundStokeWidth}
							fill={circleFill}
						/>
						<circle
							className="jet-guten-circle-progress__value"
							cx={center}
							cy={center}
							r={radius}
							stroke={stokeColor}
							stroke-width={stokeWidth}
							data-value={valueNumber}
							stroke-dasharray={circumference}
							stroke-dashoffset={progress}
							fill="none"
						/>
					</svg>
					<div className={`${className}__value`}>
						<span
							className={`${className}__value__prefix`}
							style={{
								color: prefixColor,
								fontSize: prefixFontSize ? prefixFontSize + 'px' : undefined
							}}
						>{valueNumberPrefix}</span>
						<span
							className={`${className}__value__number`}
							style={{
								color: numberColor,
								fontSize: numberFontSize ? numberFontSize + 'px' : undefined
							}}
						>{valueNumber}</span>
						<span
							className={`${className}__value__suffix`}
							style={{
								color: suffixColor,
								fontSize: suffixFontSize ? suffixFontSize + 'px' : undefined
							}}
						>{valueNumberSuffix}</span>
					</div>
				</div>
				<div className={`${className}__content`}>
					<RichText.Content
						tagName={ titleNodeName.toLowerCase() }
						className={`${className}__title`}
						value={counterTitle}
						style={{
							color: titleColor,
							fontSize: titleFontSize ? titleFontSize + 'px' : undefined
						}}
					/>
					<div
						className={`${className}__subtitle`}
						style={{
							color: subtitleColor,
							fontSize: subtitleFontSize ? subtitleFontSize + 'px' : undefined
						}}
					>{counterSubtitle}</div>
				</div>
			</div>
		);
	}
});