/**
 * Plugin's jquery functions and included jQuery libraries
 */


/**
 * Open a certain marker (where openMarker(1, 5) mean: open the 5th marker from the 1st map displayed) 
 * @param e the index of the map
 * @param t the index of the marker to be open in the map
 */
function openMarker(e, t) {
    jQuery(".responsive-map").each(function(n) {
        if (jQuery(this).data("gmap") && e == n + 1) {
            google.maps.event.trigger(jQuery(this).gMapResp("getMarker", t), "click")
        }
    })
}


/**
 * Adds a "your location" button
 * @param map the map in which to create the "your location" button
 */
function resmap_addLocatemeButton(map) {

    var controlDiv = document.createElement('div');
    controlDiv.id = 'locatemediv';
    
    var firstChild = document.createElement('button');
    firstChild.id = 'locatemeFirstChild';
    firstChild.title = 'Your Location';
    firstChild.type = "button";
    controlDiv.appendChild(firstChild);
    
    var secondChild = document.createElement('div');
    secondChild.id = 'locatemeSecondChild';
    firstChild.appendChild(secondChild);
    
    var thirdChild = document.createElement('div');
    thirdChild.id = 'locatemeThirdChild';
    secondChild.appendChild(thirdChild);
    
    firstChild.addEventListener('click', function() {
    	
        if(navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(function(position) {
                var latlng = new google.maps.LatLng(position.coords.latitude, position.coords.longitude);
                map.setCenter(latlng);
                
                var locatemeIcon = {
                		path: 'M-20,0a20,20 0 1,0 40,0a20,20 0 1,0 -40,0',
                        strokeColor: '#BBBBBB',
                        strokeWeight: 6,
                        fillColor: '#666666',
                        fillOpacity: 0.8,
                        scale: 0.7
            	  };

                var marker = new google.maps.Marker({
                    position: latlng,
                    icon: locatemeIcon,
                    map: map,
                    title: ''
                  });
            });
        }
    });
    
    controlDiv.index = 1;
    map.controls[google.maps.ControlPosition.RIGHT_BOTTOM].push(controlDiv);
}


/**
 * Create address search control 
 * @param map the map in which to create the search box
 */
function resmap_createSearchBox(map) {
    var control = document.createElement('div');
    var input = document.createElement('input');
    control.appendChild(input);
    control.setAttribute('id', 'locationDiv');
    input.setAttribute('id', 'locationInput');
    map.controls[google.maps.ControlPosition.LEFT_BOTTOM].push(control);

    var ac = new google.maps.places.Autocomplete(input, { types: ['geocode'] });

    google.maps.event.addListener(ac, 'place_changed', function() {
        var place = ac.getPlace();
        if (place && place.geometry && place.geometry.viewport) {
          map.fitBounds(place.geometry.viewport);
        } else if (place.geometry) {
          map.setCenter(place.geometry.location);
          map.setZoom(17);
        }
      });

    input.onkeyup = resmap_submitGeocode(input);
}

/**
 * Geocodes the address entered in the input field 
 * @param input the search box 
 */
function resmap_submitGeocode(input) {
    return function(e) {
        var enterPressed;

        if (window.event) {
            enterPressed = (window.event.which == 13 || window.event.keyCode == 13);
        } else {
            enterPressed = (e.which == 13);
        }

        if (enterPressed && typeof geocoder !== "undefined") {
            geocoder.geocode({
                address: input.value
            }, function(results, status) {
                if (status == google.maps.GeocoderStatus.OK) {
                    map.fitBounds(results[0].geometry.viewport);
                } 
            });
        }
    }
}
 
//==ClosureCompiler==
//@compilation_level ADVANCED_OPTIMIZATIONS
//@externs_url https://raw.githubusercontent.com/google/closure-compiler/master/contrib/externs/maps/google_maps_api_v3.js
//==/ClosureCompiler==

/**
* @name MarkerClusterer for Google Maps v3
* @version version 1.0
* @author Luke Mahe
* @fileoverview
* The library creates and manages per-zoom-level clusters for large amounts of
* markers.
* <br/>
* This is a v3 implementation of the
* <a href="http://gmaps-utility-library-dev.googlecode.com/svn/tags/markerclusterer/"
* >v2 MarkerClusterer</a>.
*/

/**
* @license
* Copyright 2010 Google Inc. All Rights Reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/


/**
* A Marker Clusterer that clusters markers.
*
* @param {google.maps.Map} map The Google map to attach to.
* @param {Array.<google.maps.Marker>=} opt_markers Optional markers to add to
*   the cluster.
* @param {Object=} opt_options support the following options:
*     'gridSize': (number) The grid size of a cluster in pixels.
*     'maxZoom': (number) The maximum zoom level that a marker can be part of a
*                cluster.
*     'zoomOnClick': (boolean) Whether the default behaviour of clicking on a
*                    cluster is to zoom into it.
*     'averageCenter': (boolean) Whether the center of each cluster should be
*                      the average of all markers in the cluster.
*     'minimumClusterSize': (number) The minimum number of markers to be in a
*                           cluster before the markers are hidden and a count
*                           is shown.
*     'styles': (object) An object that has style properties:
*       'url': (string) The image url.
*       'height': (number) The image height.
*       'width': (number) The image width.
*       'anchor': (Array) The anchor position of the label text.
*       'textColor': (string) The text color.
*       'textSize': (number) The text size.
*       'backgroundPosition': (string) The position of the backgound x, y.
*       'iconAnchor': (Array) The anchor position of the icon x, y.
* @constructor
* @extends google.maps.OverlayView
*/
function MarkerClusterer(map, opt_markers, opt_options) {
// MarkerClusterer implements google.maps.OverlayView interface. We use the
// extend function to extend MarkerClusterer with google.maps.OverlayView
// because it might not always be available when the code is defined so we
// look for it at the last possible moment. If it doesn't exist now then
// there is no point going ahead :)
this.extend(MarkerClusterer, google.maps.OverlayView);
this.map_ = map;

/**
* @type {Array.<google.maps.Marker>}
* @private
*/
this.markers_ = [];

/**
*  @type {Array.<Cluster>}
*/
this.clusters_ = [];

this.sizes = [53, 56, 66, 78, 90];

/**
* @private
*/
this.styles_ = [];

/**
* @type {boolean}
* @private
*/
this.ready_ = false;

var options = opt_options || {};

/**
* @type {number}
* @private
*/
this.gridSize_ = options['gridSize'] || 60;

/**
* @private
*/
this.minClusterSize_ = options['minimumClusterSize'] || 2;


/**
* @type {?number}
* @private
*/
this.maxZoom_ = options['maxZoom'] || null;

this.styles_ = options['styles'] || [];

/**
* @type {string}
* @private
*/
this.imagePath_ = options['imagePath'] ||
   this.MARKER_CLUSTER_IMAGE_PATH_;

/**
* @type {string}
* @private
*/
this.imageExtension_ = options['imageExtension'] ||
   this.MARKER_CLUSTER_IMAGE_EXTENSION_;

/**
* @type {boolean}
* @private
*/
this.zoomOnClick_ = true;

if (options['zoomOnClick'] != undefined) {
 this.zoomOnClick_ = options['zoomOnClick'];
}

/**
* @type {boolean}
* @private
*/
this.averageCenter_ = false;

if (options['averageCenter'] != undefined) {
 this.averageCenter_ = options['averageCenter'];
}

this.setupStyles_();

this.setMap(map);

/**
* @type {number}
* @private
*/
this.prevZoom_ = this.map_.getZoom();

// Add the map event listeners
var that = this;
google.maps.event.addListener(this.map_, 'zoom_changed', function() {
 var zoom = that.map_.getZoom();

 if (that.prevZoom_ != zoom) {
   that.prevZoom_ = zoom;
   that.resetViewport();
 }
});

google.maps.event.addListener(this.map_, 'idle', function() {
 that.redraw();
});

// Finally, add the markers
if (opt_markers && opt_markers.length) {
 this.addMarkers(opt_markers, false);
}
}


/**
* The marker cluster image path.
*
* @type {string}
* @private
*/
MarkerClusterer.prototype.MARKER_CLUSTER_IMAGE_PATH_ = '../images/m';


/**
* The marker cluster image path.
*
* @type {string}
* @private
*/
MarkerClusterer.prototype.MARKER_CLUSTER_IMAGE_EXTENSION_ = 'png';


/**
* Extends a objects prototype by anothers.
*
* @param {Object} obj1 The object to be extended.
* @param {Object} obj2 The object to extend with.
* @return {Object} The new extended object.
* @ignore
*/
MarkerClusterer.prototype.extend = function(obj1, obj2) {
return (function(object) {
 for (var property in object.prototype) {
   this.prototype[property] = object.prototype[property];
 }
 return this;
}).apply(obj1, [obj2]);
};


/**
* Implementaion of the interface method.
* @ignore
*/
MarkerClusterer.prototype.onAdd = function() {
this.setReady_(true);
};

/**
* Implementaion of the interface method.
* @ignore
*/
MarkerClusterer.prototype.draw = function() {};

/**
* Sets up the styles object.
*
* @private
*/
MarkerClusterer.prototype.setupStyles_ = function() {
if (this.styles_.length) {
 return;
}

for (var i = 0, size; size = this.sizes[i]; i++) {
 this.styles_.push({
   url: this.imagePath_ + (i + 1) + '.' + this.imageExtension_,
   height: size,
   width: size
 });
}
};

/**
*  Fit the map to the bounds of the markers in the clusterer.
*/
MarkerClusterer.prototype.fitMapToMarkers = function() {
var markers = this.getMarkers();
var bounds = new google.maps.LatLngBounds();
for (var i = 0, marker; marker = markers[i]; i++) {
 bounds.extend(marker.getPosition());
}

this.map_.fitBounds(bounds);
};


/**
*  Sets the styles.
*
*  @param {Object} styles The style to set.
*/
MarkerClusterer.prototype.setStyles = function(styles) {
this.styles_ = styles;
};


/**
*  Gets the styles.
*
*  @return {Object} The styles object.
*/
MarkerClusterer.prototype.getStyles = function() {
return this.styles_;
};


/**
* Whether zoom on click is set.
*
* @return {boolean} True if zoomOnClick_ is set.
*/
MarkerClusterer.prototype.isZoomOnClick = function() {
return this.zoomOnClick_;
};

/**
* Whether average center is set.
*
* @return {boolean} True if averageCenter_ is set.
*/
MarkerClusterer.prototype.isAverageCenter = function() {
return this.averageCenter_;
};


/**
*  Returns the array of markers in the clusterer.
*
*  @return {Array.<google.maps.Marker>} The markers.
*/
MarkerClusterer.prototype.getMarkers = function() {
return this.markers_;
};


/**
*  Returns the number of markers in the clusterer
*
*  @return {Number} The number of markers.
*/
MarkerClusterer.prototype.getTotalMarkers = function() {
return this.markers_.length;
};


/**
*  Sets the max zoom for the clusterer.
*
*  @param {number} maxZoom The max zoom level.
*/
MarkerClusterer.prototype.setMaxZoom = function(maxZoom) {
this.maxZoom_ = maxZoom;
};


/**
*  Gets the max zoom for the clusterer.
*
*  @return {number} The max zoom level.
*/
MarkerClusterer.prototype.getMaxZoom = function() {
return this.maxZoom_;
};


/**
*  The function for calculating the cluster icon image.
*
*  @param {Array.<google.maps.Marker>} markers The markers in the clusterer.
*  @param {number} numStyles The number of styles available.
*  @return {Object} A object properties: 'text' (string) and 'index' (number).
*  @private
*/
MarkerClusterer.prototype.calculator_ = function(markers, numStyles) {
var index = 0;
var count = markers.length;
var dv = count;
while (dv !== 0) {
 dv = parseInt(dv / 10, 10);
 index++;
}

index = Math.min(index, numStyles);
return {
 text: count,
 index: index
};
};


/**
* Set the calculator function.
*
* @param {function(Array, number)} calculator The function to set as the
*     calculator. The function should return a object properties:
*     'text' (string) and 'index' (number).
*
*/
MarkerClusterer.prototype.setCalculator = function(calculator) {
this.calculator_ = calculator;
};


/**
* Get the calculator function.
*
* @return {function(Array, number)} the calculator function.
*/
MarkerClusterer.prototype.getCalculator = function() {
return this.calculator_;
};


/**
* Add an array of markers to the clusterer.
*
* @param {Array.<google.maps.Marker>} markers The markers to add.
* @param {boolean=} opt_nodraw Whether to redraw the clusters.
*/
MarkerClusterer.prototype.addMarkers = function(markers, opt_nodraw) {
for (var i = 0, marker; marker = markers[i]; i++) {
 this.pushMarkerTo_(marker);
}
if (!opt_nodraw) {
 this.redraw();
}
};


/**
* Pushes a marker to the clusterer.
*
* @param {google.maps.Marker} marker The marker to add.
* @private
*/
MarkerClusterer.prototype.pushMarkerTo_ = function(marker) {
marker.isAdded = false;
if (marker['draggable']) {
 // If the marker is draggable add a listener so we update the clusters on
 // the drag end.
 var that = this;
 google.maps.event.addListener(marker, 'dragend', function() {
   marker.isAdded = false;
   that.repaint();
 });
}
this.markers_.push(marker);
};


/**
* Adds a marker to the clusterer and redraws if needed.
*
* @param {google.maps.Marker} marker The marker to add.
* @param {boolean=} opt_nodraw Whether to redraw the clusters.
*/
MarkerClusterer.prototype.addMarker = function(marker, opt_nodraw) {
this.pushMarkerTo_(marker);
if (!opt_nodraw) {
 this.redraw();
}
};


/**
* Removes a marker and returns true if removed, false if not
*
* @param {google.maps.Marker} marker The marker to remove
* @return {boolean} Whether the marker was removed or not
* @private
*/
MarkerClusterer.prototype.removeMarker_ = function(marker) {
var index = -1;
if (this.markers_.indexOf) {
 index = this.markers_.indexOf(marker);
} else {
 for (var i = 0, m; m = this.markers_[i]; i++) {
   if (m == marker) {
     index = i;
     break;
   }
 }
}

if (index == -1) {
 // Marker is not in our list of markers.
 return false;
}

marker.setMap(null);

this.markers_.splice(index, 1);

return true;
};


/**
* Remove a marker from the cluster.
*
* @param {google.maps.Marker} marker The marker to remove.
* @param {boolean=} opt_nodraw Optional boolean to force no redraw.
* @return {boolean} True if the marker was removed.
*/
MarkerClusterer.prototype.removeMarker = function(marker, opt_nodraw) {
var removed = this.removeMarker_(marker);

if (!opt_nodraw && removed) {
 this.resetViewport();
 this.redraw();
 return true;
} else {
return false;
}
};


/**
* Removes an array of markers from the cluster.
*
* @param {Array.<google.maps.Marker>} markers The markers to remove.
* @param {boolean=} opt_nodraw Optional boolean to force no redraw.
*/
MarkerClusterer.prototype.removeMarkers = function(markers, opt_nodraw) {
var removed = false;

for (var i = 0, marker; marker = markers[i]; i++) {
 var r = this.removeMarker_(marker);
 removed = removed || r;
}

if (!opt_nodraw && removed) {
 this.resetViewport();
 this.redraw();
 return true;
}
};


/**
* Sets the clusterer's ready state.
*
* @param {boolean} ready The state.
* @private
*/
MarkerClusterer.prototype.setReady_ = function(ready) {
if (!this.ready_) {
 this.ready_ = ready;
 this.createClusters_();
}
};


/**
* Returns the number of clusters in the clusterer.
*
* @return {number} The number of clusters.
*/
MarkerClusterer.prototype.getTotalClusters = function() {
return this.clusters_.length;
};


/**
* Returns the google map that the clusterer is associated with.
*
* @return {google.maps.Map} The map.
*/
MarkerClusterer.prototype.getMap = function() {
return this.map_;
};


/**
* Sets the google map that the clusterer is associated with.
*
* @param {google.maps.Map} map The map.
*/
MarkerClusterer.prototype.setMap = function(map) {
this.map_ = map;
};


/**
* Returns the size of the grid.
*
* @return {number} The grid size.
*/
MarkerClusterer.prototype.getGridSize = function() {
return this.gridSize_;
};


/**
* Sets the size of the grid.
*
* @param {number} size The grid size.
*/
MarkerClusterer.prototype.setGridSize = function(size) {
this.gridSize_ = size;
};


/**
* Returns the min cluster size.
*
* @return {number} The grid size.
*/
MarkerClusterer.prototype.getMinClusterSize = function() {
return this.minClusterSize_;
};

/**
* Sets the min cluster size.
*
* @param {number} size The grid size.
*/
MarkerClusterer.prototype.setMinClusterSize = function(size) {
this.minClusterSize_ = size;
};


/**
* Extends a bounds object by the grid size.
*
* @param {google.maps.LatLngBounds} bounds The bounds to extend.
* @return {google.maps.LatLngBounds} The extended bounds.
*/
MarkerClusterer.prototype.getExtendedBounds = function(bounds) {
var projection = this.getProjection();

// Turn the bounds into latlng.
var tr = new google.maps.LatLng(bounds.getNorthEast().lat(),
   bounds.getNorthEast().lng());
var bl = new google.maps.LatLng(bounds.getSouthWest().lat(),
   bounds.getSouthWest().lng());

// Convert the points to pixels and the extend out by the grid size.
var trPix = projection.fromLatLngToDivPixel(tr);
trPix.x += this.gridSize_;
trPix.y -= this.gridSize_;

var blPix = projection.fromLatLngToDivPixel(bl);
blPix.x -= this.gridSize_;
blPix.y += this.gridSize_;

// Convert the pixel points back to LatLng
var ne = projection.fromDivPixelToLatLng(trPix);
var sw = projection.fromDivPixelToLatLng(blPix);

// Extend the bounds to contain the new bounds.
bounds.extend(ne);
bounds.extend(sw);

return bounds;
};


/**
* Determins if a marker is contained in a bounds.
*
* @param {google.maps.Marker} marker The marker to check.
* @param {google.maps.LatLngBounds} bounds The bounds to check against.
* @return {boolean} True if the marker is in the bounds.
* @private
*/
MarkerClusterer.prototype.isMarkerInBounds_ = function(marker, bounds) {
return bounds.contains(marker.getPosition());
};


/**
* Clears all clusters and markers from the clusterer.
*/
MarkerClusterer.prototype.clearMarkers = function() {
this.resetViewport(true);

// Set the markers a empty array.
this.markers_ = [];
};


/**
* Clears all existing clusters and recreates them.
* @param {boolean} opt_hide To also hide the marker.
*/
MarkerClusterer.prototype.resetViewport = function(opt_hide) {
// Remove all the clusters
for (var i = 0, cluster; cluster = this.clusters_[i]; i++) {
 cluster.remove();
}

// Reset the markers to not be added and to be invisible.
for (var i = 0, marker; marker = this.markers_[i]; i++) {
 marker.isAdded = false;
 if (opt_hide) {
   marker.setMap(null);
 }
}

this.clusters_ = [];
};

/**
*
*/
MarkerClusterer.prototype.repaint = function() {
var oldClusters = this.clusters_.slice();
this.clusters_.length = 0;
this.resetViewport();
this.redraw();

// Remove the old clusters.
// Do it in a timeout so the other clusters have been drawn first.
window.setTimeout(function() {
 for (var i = 0, cluster; cluster = oldClusters[i]; i++) {
   cluster.remove();
 }
}, 0);
};


/**
* Redraws the clusters.
*/
MarkerClusterer.prototype.redraw = function() {
this.createClusters_();
};


/**
* Calculates the distance between two latlng locations in km.
* @see http://www.movable-type.co.uk/scripts/latlong.html
*
* @param {google.maps.LatLng} p1 The first lat lng point.
* @param {google.maps.LatLng} p2 The second lat lng point.
* @return {number} The distance between the two points in km.
* @private
*/
MarkerClusterer.prototype.distanceBetweenPoints_ = function(p1, p2) {
if (!p1 || !p2) {
 return 0;
}

var R = 6371; // Radius of the Earth in km
var dLat = (p2.lat() - p1.lat()) * Math.PI / 180;
var dLon = (p2.lng() - p1.lng()) * Math.PI / 180;
var a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
 Math.cos(p1.lat() * Math.PI / 180) * Math.cos(p2.lat() * Math.PI / 180) *
 Math.sin(dLon / 2) * Math.sin(dLon / 2);
var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
var d = R * c;
return d;
};


/**
* Add a marker to a cluster, or creates a new cluster.
*
* @param {google.maps.Marker} marker The marker to add.
* @private
*/
MarkerClusterer.prototype.addToClosestCluster_ = function(marker) {
var distance = 40000; // Some large number
var clusterToAddTo = null;
var pos = marker.getPosition();
for (var i = 0, cluster; cluster = this.clusters_[i]; i++) {
 var center = cluster.getCenter();
 if (center) {
   var d = this.distanceBetweenPoints_(center, marker.getPosition());
   if (d < distance) {
     distance = d;
     clusterToAddTo = cluster;
   }
 }
}

if (clusterToAddTo && clusterToAddTo.isMarkerInClusterBounds(marker)) {
 clusterToAddTo.addMarker(marker);
} else {
 var cluster = new Cluster(this);
 cluster.addMarker(marker);
 this.clusters_.push(cluster);
}
};


/**
* Creates the clusters.
*
* @private
*/
MarkerClusterer.prototype.createClusters_ = function() {
if (!this.ready_) {
 return;
}

// Get our current map view bounds.
// Create a new bounds object so we don't affect the map.
var mapBounds = new google.maps.LatLngBounds(this.map_.getBounds().getSouthWest(),
   this.map_.getBounds().getNorthEast());
var bounds = this.getExtendedBounds(mapBounds);

for (var i = 0, marker; marker = this.markers_[i]; i++) {
 if (!marker.isAdded && this.isMarkerInBounds_(marker, bounds)) {
   this.addToClosestCluster_(marker);
 }
}
};


/**
* A cluster that contains markers.
*
* @param {MarkerClusterer} markerClusterer The markerclusterer that this
*     cluster is associated with.
* @constructor
* @ignore
*/
function Cluster(markerClusterer) {
this.markerClusterer_ = markerClusterer;
this.map_ = markerClusterer.getMap();
this.gridSize_ = markerClusterer.getGridSize();
this.minClusterSize_ = markerClusterer.getMinClusterSize();
this.averageCenter_ = markerClusterer.isAverageCenter();
this.center_ = null;
this.markers_ = [];
this.bounds_ = null;
this.clusterIcon_ = new ClusterIcon(this, markerClusterer.getStyles(),
   markerClusterer.getGridSize());
}

/**
* Determins if a marker is already added to the cluster.
*
* @param {google.maps.Marker} marker The marker to check.
* @return {boolean} True if the marker is already added.
*/
Cluster.prototype.isMarkerAlreadyAdded = function(marker) {
if (this.markers_.indexOf) {
 return this.markers_.indexOf(marker) != -1;
} else {
 for (var i = 0, m; m = this.markers_[i]; i++) {
   if (m == marker) {
     return true;
   }
 }
}
return false;
};


/**
* Add a marker the cluster.
*
* @param {google.maps.Marker} marker The marker to add.
* @return {boolean} True if the marker was added.
*/
Cluster.prototype.addMarker = function(marker) {
if (this.isMarkerAlreadyAdded(marker)) {
 return false;
}

if (!this.center_) {
 this.center_ = marker.getPosition();
 this.calculateBounds_();
} else {
 if (this.averageCenter_) {
   var l = this.markers_.length + 1;
   var lat = (this.center_.lat() * (l-1) + marker.getPosition().lat()) / l;
   var lng = (this.center_.lng() * (l-1) + marker.getPosition().lng()) / l;
   this.center_ = new google.maps.LatLng(lat, lng);
   this.calculateBounds_();
 }
}

marker.isAdded = true;
this.markers_.push(marker);

var len = this.markers_.length;
if (len < this.minClusterSize_ && marker.getMap() != this.map_) {
 // Min cluster size not reached so show the marker.
 marker.setMap(this.map_);
}

if (len == this.minClusterSize_) {
 // Hide the markers that were showing.
 for (var i = 0; i < len; i++) {
   this.markers_[i].setMap(null);
 }
}

if (len >= this.minClusterSize_) {
 marker.setMap(null);
}

this.updateIcon();
return true;
};


/**
* Returns the marker clusterer that the cluster is associated with.
*
* @return {MarkerClusterer} The associated marker clusterer.
*/
Cluster.prototype.getMarkerClusterer = function() {
return this.markerClusterer_;
};


/**
* Returns the bounds of the cluster.
*
* @return {google.maps.LatLngBounds} the cluster bounds.
*/
Cluster.prototype.getBounds = function() {
var bounds = new google.maps.LatLngBounds(this.center_, this.center_);
var markers = this.getMarkers();
for (var i = 0, marker; marker = markers[i]; i++) {
 bounds.extend(marker.getPosition());
}
return bounds;
};


/**
* Removes the cluster
*/
Cluster.prototype.remove = function() {
this.clusterIcon_.remove();
this.markers_.length = 0;
delete this.markers_;
};


/**
* Returns the center of the cluster.
*
* @return {number} The cluster center.
*/
Cluster.prototype.getSize = function() {
return this.markers_.length;
};


/**
* Returns the center of the cluster.
*
* @return {Array.<google.maps.Marker>} The cluster center.
*/
Cluster.prototype.getMarkers = function() {
return this.markers_;
};


/**
* Returns the center of the cluster.
*
* @return {google.maps.LatLng} The cluster center.
*/
Cluster.prototype.getCenter = function() {
return this.center_;
};


/**
* Calculated the extended bounds of the cluster with the grid.
*
* @private
*/
Cluster.prototype.calculateBounds_ = function() {
var bounds = new google.maps.LatLngBounds(this.center_, this.center_);
this.bounds_ = this.markerClusterer_.getExtendedBounds(bounds);
};


/**
* Determines if a marker lies in the clusters bounds.
*
* @param {google.maps.Marker} marker The marker to check.
* @return {boolean} True if the marker lies in the bounds.
*/
Cluster.prototype.isMarkerInClusterBounds = function(marker) {
return this.bounds_.contains(marker.getPosition());
};


/**
* Returns the map that the cluster is associated with.
*
* @return {google.maps.Map} The map.
*/
Cluster.prototype.getMap = function() {
return this.map_;
};


/**
* Updates the cluster icon
*/
Cluster.prototype.updateIcon = function() {
var zoom = this.map_.getZoom();
var mz = this.markerClusterer_.getMaxZoom();

if (mz && zoom > mz) {
 // The zoom is greater than our max zoom so show all the markers in cluster.
 for (var i = 0, marker; marker = this.markers_[i]; i++) {
   marker.setMap(this.map_);
 }
 return;
}

if (this.markers_.length < this.minClusterSize_) {
 // Min cluster size not yet reached.
 this.clusterIcon_.hide();
 return;
}

var numStyles = this.markerClusterer_.getStyles().length;
var sums = this.markerClusterer_.getCalculator()(this.markers_, numStyles);
this.clusterIcon_.setCenter(this.center_);
this.clusterIcon_.setSums(sums);
this.clusterIcon_.show();
};


/**
* A cluster icon
*
* @param {Cluster} cluster The cluster to be associated with.
* @param {Object} styles An object that has style properties:
*     'url': (string) The image url.
*     'height': (number) The image height.
*     'width': (number) The image width.
*     'anchor': (Array) The anchor position of the label text.
*     'textColor': (string) The text color.
*     'textSize': (number) The text size.
*     'backgroundPosition: (string) The background postition x, y.
* @param {number=} opt_padding Optional padding to apply to the cluster icon.
* @constructor
* @extends google.maps.OverlayView
* @ignore
*/
function ClusterIcon(cluster, styles, opt_padding) {
cluster.getMarkerClusterer().extend(ClusterIcon, google.maps.OverlayView);

this.styles_ = styles;
this.padding_ = opt_padding || 0;
this.cluster_ = cluster;
this.center_ = null;
this.map_ = cluster.getMap();
this.div_ = null;
this.sums_ = null;
this.visible_ = false;

this.setMap(this.map_);
}


/**
* Triggers the clusterclick event and zoom's if the option is set.
*
* @param {google.maps.MouseEvent} event The event to propagate
*/
ClusterIcon.prototype.triggerClusterClick = function(event) {
var markerClusterer = this.cluster_.getMarkerClusterer();

// Trigger the clusterclick event.
google.maps.event.trigger(markerClusterer, 'clusterclick', this.cluster_, event);

if (markerClusterer.isZoomOnClick()) {
 // Zoom into the cluster.
 this.map_.fitBounds(this.cluster_.getBounds());
 this.map_.setCenter(this.cluster_.getCenter());
}
};


/**
* Adding the cluster icon to the dom.
* @ignore
*/
ClusterIcon.prototype.onAdd = function() {
this.div_ = document.createElement('DIV');
if (this.visible_) {
 var pos = this.getPosFromLatLng_(this.center_);
 this.div_.style.cssText = this.createCss(pos);
 this.div_.innerHTML = this.sums_.text;
}

var panes = this.getPanes();
panes.overlayMouseTarget.appendChild(this.div_);

var that = this;
var isDragging = false;
google.maps.event.addDomListener(this.div_, 'click', function(event) {
 // Only perform click when not preceded by a drag
 if (!isDragging) {
   that.triggerClusterClick(event);
 }
});
google.maps.event.addDomListener(this.div_, 'mousedown', function() {
 isDragging = false;
});
google.maps.event.addDomListener(this.div_, 'mousemove', function() {
 isDragging = true;
});
};


/**
* Returns the position to place the div dending on the latlng.
*
* @param {google.maps.LatLng} latlng The position in latlng.
* @return {google.maps.Point} The position in pixels.
* @private
*/
ClusterIcon.prototype.getPosFromLatLng_ = function(latlng) {
var pos = this.getProjection().fromLatLngToDivPixel(latlng);

if (typeof this.iconAnchor_ === 'object' && this.iconAnchor_.length === 2) {
 pos.x -= this.iconAnchor_[0];
 pos.y -= this.iconAnchor_[1];
} else {
 pos.x -= parseInt(this.width_ / 2, 10);
 pos.y -= parseInt(this.height_ / 2, 10);
}
return pos;
};


/**
* Draw the icon.
* @ignore
*/
ClusterIcon.prototype.draw = function() {
if (this.visible_) {
 var pos = this.getPosFromLatLng_(this.center_);
 this.div_.style.top = pos.y + 'px';
 this.div_.style.left = pos.x + 'px';
}
};


/**
* Hide the icon.
*/
ClusterIcon.prototype.hide = function() {
if (this.div_) {
 this.div_.style.display = 'none';
}
this.visible_ = false;
};


/**
* Position and show the icon.
*/
ClusterIcon.prototype.show = function() {
if (this.div_) {
 var pos = this.getPosFromLatLng_(this.center_);
 this.div_.style.cssText = this.createCss(pos);
 this.div_.style.display = '';
}
this.visible_ = true;
};


/**
* Remove the icon from the map
*/
ClusterIcon.prototype.remove = function() {
this.setMap(null);
};


/**
* Implementation of the onRemove interface.
* @ignore
*/
ClusterIcon.prototype.onRemove = function() {
if (this.div_ && this.div_.parentNode) {
 this.hide();
 this.div_.parentNode.removeChild(this.div_);
 this.div_ = null;
}
};


/**
* Set the sums of the icon.
*
* @param {Object} sums The sums containing:
*   'text': (string) The text to display in the icon.
*   'index': (number) The style index of the icon.
*/
ClusterIcon.prototype.setSums = function(sums) {
this.sums_ = sums;
this.text_ = sums.text;
this.index_ = sums.index;
if (this.div_) {
 this.div_.innerHTML = sums.text;
}

this.useStyle();
};


/**
* Sets the icon to the the styles.
*/
ClusterIcon.prototype.useStyle = function() {
var index = Math.max(0, this.sums_.index - 1);
index = Math.min(this.styles_.length - 1, index);
var style = this.styles_[index];
this.url_ = style['url'];
this.height_ = style['height'];
this.width_ = style['width'];
this.textColor_ = style['textColor'];
this.anchor_ = style['anchor'];
this.textSize_ = style['textSize'];
this.backgroundPosition_ = style['backgroundPosition'];
this.iconAnchor_ = style['iconAnchor'];
};


/**
* Sets the center of the icon.
*
* @param {google.maps.LatLng} center The latlng to set as the center.
*/
ClusterIcon.prototype.setCenter = function(center) {
this.center_ = center;
};


/**
* Create the css text based on the position of the icon.
*
* @param {google.maps.Point} pos The position.
* @return {string} The css style text.
*/
ClusterIcon.prototype.createCss = function(pos) {
var style = [];
style.push('background-image:url(' + this.url_ + ');');
var backgroundPosition = this.backgroundPosition_ ? this.backgroundPosition_ : '0 0';
style.push('background-position:' + backgroundPosition + ';');

if (typeof this.anchor_ === 'object') {
 if (typeof this.anchor_[0] === 'number' && this.anchor_[0] > 0 &&
     this.anchor_[0] < this.height_) {
   style.push('height:' + (this.height_ - this.anchor_[0]) +
       'px; padding-top:' + this.anchor_[0] + 'px;');
 } else if (typeof this.anchor_[0] === 'number' && this.anchor_[0] < 0 &&
     -this.anchor_[0] < this.height_) {
   style.push('height:' + this.height_ + 'px; line-height:' + (this.height_ + this.anchor_[0]) +
       'px;');
 } else {
   style.push('height:' + this.height_ + 'px; line-height:' + this.height_ +
       'px;');
 }
 if (typeof this.anchor_[1] === 'number' && this.anchor_[1] > 0 &&
     this.anchor_[1] < this.width_) {
   style.push('width:' + (this.width_ - this.anchor_[1]) +
       'px; padding-left:' + this.anchor_[1] + 'px;');
 } else {
   style.push('width:' + this.width_ + 'px; text-align:center;');
 }
} else {
 style.push('height:' + this.height_ + 'px; line-height:' +
     this.height_ + 'px; width:' + this.width_ + 'px; text-align:center;');
}

var txtColor = this.textColor_ ? this.textColor_ : 'black';
var txtSize = this.textSize_ ? this.textSize_ : 11;

style.push('cursor:pointer; top:' + pos.y + 'px; left:' +
   pos.x + 'px; color:' + txtColor + '; position:absolute; font-size:' +
   txtSize + 'px; font-family:Arial,sans-serif; font-weight:bold');
return style.join('');
};


//Export Symbols for Closure
//If you are not going to compile with closure then you can remove the
//code below.
window['MarkerClusterer'] = MarkerClusterer;
MarkerClusterer.prototype['addMarker'] = MarkerClusterer.prototype.addMarker;
MarkerClusterer.prototype['addMarkers'] = MarkerClusterer.prototype.addMarkers;
MarkerClusterer.prototype['clearMarkers'] =
 MarkerClusterer.prototype.clearMarkers;
MarkerClusterer.prototype['fitMapToMarkers'] =
 MarkerClusterer.prototype.fitMapToMarkers;
MarkerClusterer.prototype['getCalculator'] =
 MarkerClusterer.prototype.getCalculator;
MarkerClusterer.prototype['getGridSize'] =
 MarkerClusterer.prototype.getGridSize;
MarkerClusterer.prototype['getExtendedBounds'] =
 MarkerClusterer.prototype.getExtendedBounds;
MarkerClusterer.prototype['getMap'] = MarkerClusterer.prototype.getMap;
MarkerClusterer.prototype['getMarkers'] = MarkerClusterer.prototype.getMarkers;
MarkerClusterer.prototype['getMaxZoom'] = MarkerClusterer.prototype.getMaxZoom;
MarkerClusterer.prototype['getStyles'] = MarkerClusterer.prototype.getStyles;
MarkerClusterer.prototype['getTotalClusters'] =
 MarkerClusterer.prototype.getTotalClusters;
MarkerClusterer.prototype['getTotalMarkers'] =
 MarkerClusterer.prototype.getTotalMarkers;
MarkerClusterer.prototype['redraw'] = MarkerClusterer.prototype.redraw;
MarkerClusterer.prototype['removeMarker'] =
 MarkerClusterer.prototype.removeMarker;
MarkerClusterer.prototype['removeMarkers'] =
 MarkerClusterer.prototype.removeMarkers;
MarkerClusterer.prototype['resetViewport'] =
 MarkerClusterer.prototype.resetViewport;
MarkerClusterer.prototype['repaint'] =
 MarkerClusterer.prototype.repaint;
MarkerClusterer.prototype['setCalculator'] =
 MarkerClusterer.prototype.setCalculator;
MarkerClusterer.prototype['setGridSize'] =
 MarkerClusterer.prototype.setGridSize;
MarkerClusterer.prototype['setMaxZoom'] =
 MarkerClusterer.prototype.setMaxZoom;
MarkerClusterer.prototype['onAdd'] = MarkerClusterer.prototype.onAdd;
MarkerClusterer.prototype['draw'] = MarkerClusterer.prototype.draw;

Cluster.prototype['getCenter'] = Cluster.prototype.getCenter;
Cluster.prototype['getSize'] = Cluster.prototype.getSize;
Cluster.prototype['getMarkers'] = Cluster.prototype.getMarkers;

ClusterIcon.prototype['onAdd'] = ClusterIcon.prototype.onAdd;
ClusterIcon.prototype['draw'] = ClusterIcon.prototype.draw;
ClusterIcon.prototype['onRemove'] = ClusterIcon.prototype.onRemove;


/**
 * jQuery gMap v3
 *
 * @url         http://www.smashinglabs.pl/gmap
 * @author      Sebastian Poreba <sebastian.poreba@gmail.com>
 * @fixes       hevada <contact@hevada.com>
 * @version     3.3.3
 * @date        27.12.2012
 */
(function(e) {
    var t = function() {
        this.markers = [];
        this.mainMarker = !1;
        this.icon = "https://www.google.com/mapfiles/marker.png"
    };
    t.prototype.dist = function(e) {
        return Math.sqrt(Math.pow(this.markers[0].latitude - e.latitude, 2) + Math.pow(this.markers[0].longitude - e.longitude, 2))
    };
    t.prototype.setIcon = function(e) {
        this.icon = e
    };
    t.prototype.addMarker = function(e) {
        this.markers[this.markers.length] = e
    };
    t.prototype.getMarker = function() {
        if (this.mainmarker) return this.mainmarker;
        var e, t;
        1 < this.markers.length ? (e = new n.MarkerImage("https://thydzik.com/thydzikGoogleMap/markerlink.php?text=" + this.markers.length + "&color=EF9D3F"), t = "cluster of " + this.markers.length + " markers") : (e = new n.MarkerImage(this.icon), t = this.markers[0].title);
        return this.mainmarker = new n.Marker({
            position: new n.LatLng(this.markers[0].latitude, this.markers[0].longitude),
            icon: e,
            title: t,
            map: null
        })
    };
    var n = google.maps,
        r = new n.Geocoder,
        i = 0,
        s = 0,
        o = {},
        o = {
            init: function(t) {
                var r, i = e.extend({}, e.fn.gMapResp.defaults, t);
                for (r in e.fn.gMapResp.defaults.icon) i.icon[r] || (i.icon[r] = e.fn.gMapResp.defaults.icon[r]);
                return this.each(function() {
                    var t = e(this);
                        r = o._getMapCenter.apply(t, [i]);
                    "fit" == i.zoom && (i.zoomFit = !0, i.zoom = o._autoZoom.apply(t, [i]));
                    var s = {
                        zoom: i.zoom,
                        center: r,
                        mapTypeControl: i.mapTypeControl,
                        mapTypeControlOptions: {},
                        zoomControl: i.zoomControl,
                        draggable: i.draggable,
                        zoomControlOptions: {},
                        scaleControl: i.scaleControl,
                        scaleControlOptions: {},
                        streetViewControl: i.streetViewControl,
                        streetViewControlOptions: {},
                        fullscreenControl: i.fullscreenControl,
                        mapTypeId: i.maptype,
                        scrollwheel: i.scrollwheel,
                        maxZoom: i.maxZoom,
                        minZoom: i.minZoom
                    };
                    i.controlsPositions.mapType && (s.mapTypeControlOptions.position = i.controlsPositions.mapType);
                    i.controlsPositions.zoom && (s.zoomControlOptions.position = i.controlsPositions.zoom);
                    i.controlsPositions.scale && (s.scaleControlOptions.position = i.controlsPositions.scale);
                    i.controlsPositions.streetView && (s.streetViewControlOptions.position = i.controlsPositions.streetView);
                    i.styles && (s.styles = i.styles);
                    s.mapTypeControlOptions.style = i.controlsStyle.mapType;
                    s.zoomControlOptions.style = i.controlsStyle.zoom;
                    s = new n.Map(this, s);
                    i.log && console.log("map center is:");
                    i.log && console.log(r);
                    t.data("$gmap", s);
                    t.data("gmap", {
                        opts: i,
                        gmap: s,
                        markers: [],
                        markerKeys: {},
                        infoWindow: null,
                        clusters: []
                    });
                    if (0 !== i.controls.length)
                        for (r = 0; r < i.controls.length; r += 1) s.controls[i.controls[r].pos].push(i.controls[r].div);
                    i.clustering.enabled ? (r = t.data("gmap"), r.markers = i.markers, o._renderCluster.apply(t, []), n.event.addListener(s, "bounds_changed", function() {
                        o._renderCluster.apply(t, [])
                    })) : 0 !== i.markers.length && o.addMarkers.apply(t, [i.markers]);
                    o._onComplete.apply(t, [])
                })
            },
            _delayedMode: false,
            _onComplete: function() {
                var data = this.data('gmap'),
                that = this;

                if (i !== 0) {
                    window.setTimeout(function() {
                        o._onComplete.apply(that, []); 
                    }, 100);
                    return;
                }
                if (o._delayedMode) {
                    var center = o._getMapCenter.apply(this, [data.opts, true]);
                    if (center !== undefined && [center] != [0, 0]) {
                        o._setMapCenter.apply(this, [center]);
                    }
                    if (data.opts.zoomFit) {
                        var zoom = o._autoZoom.apply(this, [data.opts, true]);
                        data.gmap.setZoom(zoom);
                    }
                }
                data.opts.onComplete();
            },
            _setMapCenter: function(center) {
                var data = this.data('gmap');
                if (data && data.opts.log) {
                    console.log('delayed setMapCenter called'); 
                }
                if (data && data.gmap !== undefined && i == 0) {
                    if (center !== undefined && [center] != [0, 0])
                        data.gmap.setCenter(center);
                } else {
                    var that = this;
                    window.setTimeout(function() {
                        if (center !== undefined && [center] != [0, 0])
                            o._setMapCenter.apply(that, [center]); 
                    }, 100);
                }
            },
            _boundaries: null,
            _getBoundaries: function(e) {
                var t = e.markers,
                    n, r = 1e3,
                    i = -1e3,
                    s = 1e3,
                    u = -1e3;
                if (t) {
                    for (n = 0; n < t.length; n += 1) t[n].latitude && t[n].longitude && (r > t[n].latitude && (r = t[n].latitude), i < t[n].longitude && (i = t[n].longitude), s > t[n].longitude && (s = t[n].longitude), u < t[n].latitude && (u = t[n].latitude), e.log && console.log(t[n].latitude, t[n].longitude, r, i, s, u));
                    o._boundaries = {
                        N: r,
                        E: i,
                        W: s,
                        S: u
                    }
                } - 1e3 == r && (o._boundaries = {
                    N: 0,
                    E: 0,
                    W: 0,
                    S: 0
                });
                return o._boundaries
            },
            _getBoundariesFromMarkers: function() {
                var e = this.data("gmap").markers,
                    t, n = 1e3,
                    r = -1e3,
                    i = 1e3,
                    s = -1e3;
                if (e) {
                    for (t = 0; t < e.length; t += 1) n > e[t].getPosition().lat() && (n = e[t].getPosition().lat()), r < e[t].getPosition().lng() && (r = e[t].getPosition().lng()), i > e[t].getPosition().lng() && (i = e[t].getPosition().lng()), s < e[t].getPosition().lat() && (s = e[t].getPosition().lat());
                    o._boundaries = {
                        N: n,
                        E: r,
                        W: i,
                        S: s
                    }
                } - 1e3 == n && (o._boundaries = {
                    N: 0,
                    E: 0,
                    W: 0,
                    S: 0
                });
                return o._boundaries
            },
            _getMapCenter: function(opts, fromMarkers) {

                // Create new object to geocode addresses
                var center,
                        that = this, // 'that' scope fix in geocoding
                        i,
                        selectedToCenter,
                        most; //hoisting

                if (opts.markers.length && (opts.latitude == 'fit' || opts.longitude == 'fit')) {
                    if (fromMarkers) {
                      opts.markers = o._convertMarkers(data.markers);
                    }

                    most = o._getBoundaries(opts);
                    center = new n.LatLng((most.N + most.S) / 2, (most.E + most.W) / 2);
                    if (opts.log) {
                      console.log(fromMarkers, most, center);
                    }
                    return center;
                  }

                  center = new n.LatLng(0, 0);

                  if (opts.latitude && opts.longitude) {
                            // lat & lng available, return
                            center = new n.LatLng(opts.latitude, opts.longitude);
                            return center;
                  } else {

                            // Check for a marker to center on (if no coordinates given)
                            if (opts.markers.length > 0) {
                                selectedToCenter = null;

                                for (i = 0; i < opts.markers.length; i += 1) {
                                    if (opts.markers[i].setCenter) {
                                        selectedToCenter = opts.markers[i];
                                        break;
                                    }
                                }

                                if (selectedToCenter === null) {
                                    for (i = 0; i < opts.markers.length; i += 1) {
                                        if (opts.markers[i].latitude && opts.markers[i].longitude) {
                                            selectedToCenter = opts.markers[i];
                                            break;
                                        } else if (opts.markers[i].address) {
                                            selectedToCenter = opts.markers[i];
                                            break;
                                        }
                                    }
                                }

                                // failed to find any reasonable marker (it's quite impossible BTW)
                                if (selectedToCenter === null) {
                                        return center;
                                }

                                // check if the marker has latitude, longitude
                                if (selectedToCenter.latitude && selectedToCenter.longitude) {
                                        return new n.LatLng(selectedToCenter.latitude, selectedToCenter.longitude);
                                }

                                // check if the marker has an address
                                if (selectedToCenter.address) {
                                            // Get the coordinates for given marker address and center
                                            r.geocode(
                                                {address: selectedToCenter.address},
                                                function(result, status) {
                                                        if (status === google.maps.GeocoderStatus.OK) {
                                                            o._setMapCenter.apply(that, [result[0].geometry.location]);
                                                        } else {
                                                            if (opts.log) {console.log('Geocode was not successful for the following reason: ' + status); }
                                                        }
                                                }
                                            );
                                }
                        }
                    }
                //return center;
            },
            _renderCluster: function() {
                var e = this.data("gmap"),
                    n = e.markers,
                    r = e.clusters,
                    i = e.opts,
                    s;
                for (s = 0; s < r.length; s += 1) r[s].getMarker().setMap(null);
                r.length = 0;
                if (s = e.gmap.getBounds()) {
                    var u = s.getNorthEast(),
                        a = s.getSouthWest(),
                        f = [],
                        l = (u.lat() - a.lat()) * i.clustering.clusterSize / 100;
                    for (s = 0; s < n.length; s += 1) n[s].latitude < u.lat() && n[s].latitude > a.lat() && n[s].longitude < u.lng() && n[s].longitude > a.lng() && (f[f.length] = n[s]);
                    i.log && console.log("number of markers " + f.length + "/" + n.length);
                    i.log && console.log("cluster radius: " + l);
                    for (s = 0; s < f.length; s += 1) {
                        u = -1;
                        for (n = 0; n < r.length && !(a = r[n].dist(f[s]), a < l && (u = n, i.clustering.fastClustering)); n += 1); - 1 === u ? (n = new t, n.addMarker(f[s]), r[r.length] = n) : r[u].addMarker(f[s])
                    }
                    i.log && console.log("Total clusters in viewport: " + r.length);
                    for (n = 0; n < r.length; n += 1) r[n].getMarker().setMap(e.gmap)
                } else {
                    var c = this;
                    window.setTimeout(function() {
                        o._renderCluster.apply(c)
                    }, 1e3)
                }
            },
            _processMarker: function(e, t, r, i) {
                var s = this.data("gmap"),
                    o = s.gmap,
                    u = s.opts,
                    a;
                void 0 === i && (i = new n.LatLng(e.latitude, e.longitude));
                if (!t) {
                    var f = {
                        image: u.icon.image,
                        iconSize: new n.Size(u.icon.iconsize[0], u.icon.iconsize[1]),
                        iconAnchor: new n.Point(u.icon.iconanchor[0], u.icon.iconanchor[1]),
                        infoWindowAnchor: new n.Size(u.icon.infowindowanchor[0], u.icon.infowindowanchor[1])
                    };
                    
                    t = new n.MarkerImage(f.image, f.iconSize, null, f.iconAnchor)
                }
                r || (new n.Size(u.icon.shadowsize[0], u.icon.shadowsize[1]), f && f.iconAnchor || new n.Point(u.icon.iconanchor[0], u.icon.iconanchor[1]));
                t = {
                    position: i,
                    icon: t,
                    title: e.title,
                    map: null,
                    draggable: !0 === e.draggable ? !0 : !1
                };
                u.clustering.enabled || (t.map = o);
                a = new n.Marker(t);
                a.setShadow(r);
                s.markers.push(a);
                e.key && (s.markerKeys[e.key] = a);
                var l;
                e.html && (r = {
                    content: "string" === typeof e.html ? u.html_prepend + e.html + u.html_append : e.html,
                    pixelOffset: e.infoWindowAnchor
                }, u.log && console.log("setup popup with data"), u.log && console.log(r), l = new n.InfoWindow(r), n.event.addListener(a, "click", function() {
                    u.log && console.log("opening popup " + e.html);
                    u.singleInfoWindow && s.infoWindow && s.infoWindow.close();
                    l.open(o, a);
                    o.setCenter(a.getPosition());
                    s.infoWindow = l
                }));
                e.html && e.popup && (u.log && console.log("opening popup " + e.html), l.open(o, a), s.infoWindow = l);
                e.onDragEnd && n.event.addListener(a, "dragend", function(t) {
                    u.log && console.log("drag end");
                    e.onDragEnd(t)
                })
            },
            _geocodeMarker: function(e, t, u) {
                var a = this;
                r.geocode({
                    address: e.address
                }, function(r, f) {
                    f === n.GeocoderStatus.OK ? (i -= 1, a.data("gmap").opts.log && console.log("Geocode was successful with point: ", r[0].geometry.location), o._processMarker.apply(a, [e, t, u, r[0].geometry.location])) : (f === n.GeocoderStatus.OVER_QUERY_LIMIT && (!a.data("gmap").opts.noAlerts && 0 === s && alert("Error: too many geocoded addresses! Switching to 1 marker/s mode."), s += 1e3, window.setTimeout(function() {
                        o._geocodeMarker.apply(a, [e, t, u])
                    }, s)), a.data("gmap").opts.log && console.log("Geocode was not successful for the following reason: " + f))
                })
            },
            _autoZoom: function(t, n) {
                var r = e(this).data("gmap"),
                    i = e.extend({}, r ? r.opts : {}, t),
                    s, u, r = 39135.758482;
                i.log && console.log("autozooming map");
                s = n ? o._getBoundariesFromMarkers.apply(this) : o._getBoundaries(i);
                i = 111e3 * (s.E - s.W) / this.width();
                u = 111e3 * (s.S - s.N) / this.height();
                for (s = 2; 20 > s && !(i > r || u > r); s += 1) r /= 2;
                return s - 1
            },
            addMarkers: function(e) {
                var t = this.data("gmap").opts;
                if (0 !== e.length) {
                    t.log && console.log("adding " + e.length + " markers");
                    for (t = 0; t < e.length; t += 1) o.addMarker.apply(this, [e[t]])
                }
            },
            addMarker: function(e) {
                var t = this.data("gmap").opts;
                t.log && console.log("putting marker at " + e.latitude + ", " + e.longitude + " with address " + e.address + " and html " + e.html);
                var r = t.icon.image,
                    s = new n.Size(t.icon.iconsize[0], t.icon.iconsize[1]),
                    u = new n.Point(t.icon.iconanchor[0], t.icon.iconanchor[1]),
                    a = new n.Size(t.icon.infowindowanchor[0], t.icon.infowindowanchor[1]),
                    f = t.icon.shadow,
                    l = new n.Size(t.icon.shadowsize[0], t.icon.shadowsize[1]),
                    c = new n.Point(t.icon.shadowanchor[0], t.icon.shadowanchor[1]);
                e.infoWindowAnchor = a;
                e.icon && (e.icon.image && (r = e.icon.image), e.icon.iconsize && (s = new n.Size(e.icon.iconsize[0], e.icon.iconsize[1])), e.icon.iconanchor && (u = new n.Point(e.icon.iconanchor[0], e.icon.iconanchor[1])), e.icon.infowindowanchor && new n.Size(e.icon.infowindowanchor[0], e.icon.infowindowanchor[1]), e.icon.shadow && (f = e.icon.shadow), e.icon.shadowsize && (l = new n.Size(e.icon.shadowsize[0], e.icon.shadowsize[1])), e.icon.shadowanchor && (c = new n.Point(e.icon.shadowanchor[0], e.icon.shadowanchor[1])));
                if (e.icon.iconsize === undefined) {
                    r = new n.MarkerImage(r);
                } else {
                    r = new n.MarkerImage(r, null, null, null, s);
                }
                
                f = new n.MarkerImage(f, l, null, c);
                e.address ? ("_address" === e.html && (e.html = e.address), "_address" == e.title && (e.title = e.address), t.log && console.log("geocoding marker: " + e.address), i += 1, o._delayedMode = true, o._geocodeMarker.apply(this, [e, r, f])) : ("_latlng" === e.html && (e.html = e.latitude + ", " + e.longitude), "_latlng" == e.title && (e.title = e.latitude + ", " + e.longitude), t = new n.LatLng(e.latitude, e.longitude), o._processMarker.apply(this, [e, r, f, t]))
            },
            removeAllMarkers: function() {
                var e = this.data("gmap").markers,
                    t;
                for (t = 0; t < e.length; t += 1) e[t].setMap(null), delete e[t];
                e.length = 0
            },
            getMarker: function(e) {
                return this.data("gmap").markerKeys[e]
            },
            fixAfterResize: function(e) {
                var t = this.data("gmap");
                if (t!== undefined) {
                    n.event.trigger(t.gmap, "resize");
                    //e && t.gmap.panTo(new google.maps.LatLng(0, 0));
                    var center = this.gMapResp("_getMapCenter", t.opts);
                    if (center !== undefined && [center] != [0, 0]) {
                        t.gmap.panTo(center);
                    }
                }
                
            },
            setZoom: function(e, t, n) {
                var r = this.data("gmap").gmap;
                "fit" === e && (e = o._autoZoom.apply(this, [t, n]));
                r.setZoom(parseInt(e))
            },
            changeSettings: function(e) {
                var t = this.data("gmap"),
                    n = [],
                    r;
                for (r = 0; r < t.markers.length; r += 1) n[r] = {
                    latitude: t.markers[r].getPosition().lat(),
                    longitude: t.markers[r].getPosition().lng()
                };
                e.markers = n;
                e.zoom && o.setZoom.apply(this, [e.zoom, e]);

                if (e.latitude || e.longitude) {
                    var center = o._getMapCenter.apply(this, [e]);
                    if (center !== undefined && [center] != [0, 0]) {
                        t.gmap.panTo(center);
                    }
                }
            },
            mapclick: function(e) {
                google.maps.event.addListener(this.data("gmap").gmap, "click", function(t) {
                    e(t.latLng)
                })
            },
            geocode: function(e, t, i) {
                r.geocode({
                    address: e
                }, function(e, r) {
                    r === n.GeocoderStatus.OK ? t(e[0].geometry.location) : i && i(e, r)
                })
            },
            getRoute: function(t) {
                var r = this.data("gmap"),
                    i = r.gmap,
                    s = new n.DirectionsRenderer,
                    o = new n.DirectionsService,
                    u = {
                        BYCAR: n.DirectionsTravelMode.DRIVING,
                        BYBICYCLE: n.DirectionsTravelMode.BICYCLING,
                        BYFOOT: n.DirectionsTravelMode.WALKING
                    },
                    a = {
                        MILES: n.DirectionsUnitSystem.IMPERIAL,
                        KM: n.DirectionsUnitSystem.METRIC
                    },
                    f = null,
                    l = null,
                    c = null;
                void 0 !== t.routeDisplay ? f = t.routeDisplay instanceof jQuery ? t.routeDisplay[0] : "string" == typeof t.routeDisplay ? e(t.routeDisplay)[0] : null : null !== r.opts.routeFinder.routeDisplay && (f = r.opts.routeFinder.routeDisplay instanceof jQuery ? r.opts.routeFinder.routeDisplay[0] : "string" == typeof r.opts.routeFinder.routeDisplay ? e(r.opts.routeFinder.routeDisplay)[0] : null);
                s.setMap(i);
                null !== f && s.setPanel(f);
                l = void 0 !== u[r.opts.routeFinder.travelMode] ? u[r.opts.routeFinder.travelMode] : u.BYCAR;
                c = void 0 !== a[r.opts.routeFinder.travelUnit] ? a[r.opts.routeFinder.travelUnit] : a.KM;
                o.route({
                    origin: t.from,
                    destination: t.to,
                    travelMode: l,
                    unitSystem: c
                }, function(t, i) {
                    i == n.DirectionsStatus.OK ? s.setDirections(t) : null !== f && e(f).html(r.opts.routeFinder.routeErrors[i])
                });
                return this
            }
        };
    e.fn.gMapResp = function(t) {
        if (o[t]) return o[t].apply(this, Array.prototype.slice.call(arguments, 1));
        if ("object" === typeof t || !t) return o.init.apply(this, arguments);
        e.error("Method " + t + " does not exist on jQuery.gmap")
    };
    e.fn.gMapResp.defaults = {
        log: !1,
        noAlerts: !0,
        address: "",
        latitude: null,
        longitude: null,
        zoom: 3,
        maxZoom: null,
        minZoom: null,
        markers: [],
        controls: {},
        scrollwheel: !0,
        maptype: google.maps.MapTypeId.ROADMAP,
        mapTypeControl: !0,
        zoomControl: !0,
        draggable: !0,
        scaleControl: !1,
        streetViewControl: !0,
        fullscreenControl: !0,
        controlsPositions: {
            mapType: null,
            zoom: null,
            scale: null,
            streetView: null
        },
        controlsStyle: {
            mapType: google.maps.MapTypeControlStyle.DEFAULT,
            zoom: google.maps.ZoomControlStyle.DEFAULT
        },
        singleInfoWindow: true,
        html_prepend: '<div class="gmap_marker">',
        html_append: "</div>",
        icon: {
            image: "https://www.google.com/mapfiles/marker.png",
            iconsize: [20, 34],
            iconanchor: [9, 34],
            infowindowanchor: [0, 0],
            shadow: "https://www.google.com/mapfiles/shadow50.png",
            shadowsize: [37, 34],
            shadowanchor: [9, 34]
        },
        onComplete: function() {},
        routeFinder: {
            travelMode: "BYCAR",
            travelUnit: "KM",
            routeDisplay: null,
            routeErrors: {
                INVALID_REQUEST: "The provided request is invalid.",
                NOT_FOUND: "One or more of the given addresses could not be found.",
                OVER_QUERY_LIMIT: "A temporary error occured. Please try again in a few minutes.",
                REQUEST_DENIED: "An error occured. Please contact us.",
                UNKNOWN_ERROR: "An unknown error occured. Please try again.",
                ZERO_RESULTS: "No route could be found within the given addresses."
            }
        },
        clustering: {
            enabled: !1,
            fastClustering: !1,
            clusterCount: 10,
            clusterSize: 40
        }
    }
})(jQuery)

/*!
 * Retina.js v1.3.0
 *
 * Copyright 2014 Imulus, LLC
 * Released under the MIT license
 *
 * Retina.js is an open source script that makes it easy to serve
 * high-resolution images to devices with retina displays.
 */
! function() {
    function a() {}

    function b(a) {
        return f.retinaImageSuffix + a
    }

    function c(a, c) {
        if (this.path = a || "", "undefined" != typeof c && null !== c) this.at_2x_path = c, this.perform_check = !1;
        else {
            if (void 0 !== document.createElement) {
                var d = document.createElement("a");
                d.href = this.path, d.pathname = d.pathname.replace(g, b), this.at_2x_path = d.href
            } else {
                var e = this.path.split("?");
                e[0] = e[0].replace(g, b), this.at_2x_path = e.join("?")
            }
            this.perform_check = !0
        }
    }

    function d(a) {
        this.el = a, this.path = new c(this.el.getAttribute("src"), this.el.getAttribute("data-at2x"));
        var b = this;
        this.path.check_2x_variant(function(a) {
            a && b.swap()
        })
    }
    var e = "undefined" == typeof exports ? window : exports,
        f = {
            retinaImageSuffix: "@2x",
            check_mime_type: !0,
            force_original_dimensions: !0
        };
    e.Retina = a, a.configure = function(a) {
        null === a && (a = {});
        for (var b in a) a.hasOwnProperty(b) && (f[b] = a[b])
    }, a.init = function(a) {
        null === a && (a = e);
        var b = a.onload || function() {};
        a.onload = function() {
            var a, c, e = document.getElementsByTagName("img"),
                f = [];
            for (a = 0; a < e.length; a += 1) c = e[a], c.getAttributeNode("data-no-retina") || f.push(new d(c));
            b()
        }
    }, a.isRetina = function() {
        var a = "(-webkit-min-device-pixel-ratio: 1.5), (min--moz-device-pixel-ratio: 1.5), (-o-min-device-pixel-ratio: 3/2), (min-resolution: 1.5dppx)";
        return e.devicePixelRatio > 1 ? !0 : e.matchMedia && e.matchMedia(a).matches ? !0 : !1
    };
    var g = /\.\w+$/;
    e.RetinaImagePath = c, c.confirmed_paths = [], c.prototype.is_external = function() {
        return !(!this.path.match(/^https?\:/i) || this.path.match("//" + document.domain))
    }, c.prototype.check_2x_variant = function(a) {
        var b, d = this;
        return this.is_external() ? a(!1) : this.perform_check || "undefined" == typeof this.at_2x_path || null === this.at_2x_path ? this.at_2x_path in c.confirmed_paths ? a(!0) : (b = new XMLHttpRequest, b.open("HEAD", this.at_2x_path), b.onreadystatechange = function() {
            if (4 !== b.readyState) return a(!1);
            if (b.status >= 200 && b.status <= 399) {
                if (f.check_mime_type) {
                    var e = b.getResponseHeader("Content-Type");
                    if (null === e || !e.match(/^image/i)) return a(!1)
                }
                return c.confirmed_paths.push(d.at_2x_path), a(!0)
            }
            return a(!1)
        }, b.send(), void 0) : a(!0)
    }, e.RetinaImage = d, d.prototype.swap = function(a) {
        function b() {
            c.el.complete ? (f.force_original_dimensions && (c.el.setAttribute("width", c.el.offsetWidth), c.el.setAttribute("height", c.el.offsetHeight)), c.el.setAttribute("src", a)) : setTimeout(b, 5)
        }
        "undefined" == typeof a && (a = this.path.at_2x_path);
        var c = this;
        b()
    }, a.isRetina() && a.init(e)
}();

