<?php

define( 'POWER_PACK_CACHE_DIR', POWER_PACK_CONTENT_DIR . '/page-cache' );
define( 'POWER_PACK_CACHE_DEBUG', FALSE );

// Experimental
define( 'POWER_PACK_ENABLE_GZIP', false );

if ( strncasecmp( PHP_OS, 'WIN', 3 ) == 0 ) {
	define( 'POWER_PACK_CACHE_WINDOWS', true );
}

define( 'POWER_PACK_CACHE_CSS_DIR', POWER_PACK_CONTENT_DIR . '/css' );
define( 'POWER_PACK_CACHE_JS_DIR', POWER_PACK_CONTENT_DIR . '/js' );

add_filter( 'admin_init', 'wptouch_power_pack_cache_add_directory' );
add_action( 'wptouch_cache_page', 'wptouch_power_pack_cache_do_cache' );
add_action( 'wptouch_addon_cache_cleanup_event', 'wptouch_power_pack_cache_do_scheduled_cleanup' );

add_filter( 'wptouch_body_classes', 'wptouch_power_pack_cache_add_slug_to_body' );

/* Page cache flushing events */
add_action( 'create_category', 'wptouch_power_pack_cache_flush' );
add_action( 'edit_category', 'wptouch_power_pack_cache_flush' );
add_action( 'delete_category', 'wptouch_power_pack_cache_flush' );
add_action( 'publish_page', 'wptouch_power_pack_cache_flush' );
add_action( 'publish_post', 'wptouch_power_pack_cache_flush' );
add_action( 'publish_future_post', 'wptouch_power_pack_cache_flush' );
add_action( 'edit_post', 'wptouch_power_pack_cache_flush' );
add_action( 'deleted_post', 'wptouch_power_pack_cache_flush' );
add_action( 'wptouch_admin_save_settings_completed', 'wptouch_power_pack_cache_flush' );
add_action( 'wptouch_version_update', 'wptouch_power_pack_cache_flush' );

add_action( 'wptouch_admin_ajax_infinity-cache-reset', 'wptouch_power_pack_cache_handle_ajax_reset' );
add_action( 'wptouch_update_settings_domain_' . ADDON_SETTING_DOMAIN, 'wptouch_power_pack_update_cron' );
add_action( 'wptouch_admin_render_setting', 'wptouch_power_pack_cache_render_setting' );

function wptouch_power_pack_debug( $message ) {
	require_once( dirname( __FILE__ ) . '/class-debug.php' );

	Power_Pack_Debug::get()->write( $message );
}

function wptouch_power_pack_cache_add_slug_to_body( $classes ) {
	if ( wptouch_power_pack_cache_is_enabled() ) {
		$classes[] = 'power-pack-cache-active';
	}
	return $classes;
}

function wptouch_power_pack_update_cron() {
	wp_clear_scheduled_hook( 'wptouch_addon_cache_cleanup_event' );

	wptouch_power_pack_cache_check_cron();
}

function wptouch_power_pack_cache_is_enabled() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	return $settings->cache_enable;
}

function wptouch_power_pack_cache_expiry_time() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	return $settings->cache_expiry_time;
}

function wptouch_power_pack_cache_browser_cache_enabled() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	return $settings->cache_enable_browser_cache;
}

function wptouch_power_pack_cache_is_cdn_enabled() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	return $settings->cache_optimize_enable_cdn;
}


function wptouch_power_pack_cache_can_use_gzip() {
	return ( version_compare( PHP_VERSION, '5.4.0') >= 0 && POWER_PACK_ENABLE_GZIP );
}

function wptouch_power_pack_cache_get_valid_time() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );

	return $settings->cache_last_flush_time;
}

function wptouch_power_pack_cache_should_cache_page() {
	// Check if cache has been disabled
	wptouch_power_pack_debug( 'Checking to see if CACHE is enabled' );

	if ( !wptouch_power_pack_cache_is_enabled() ) {
		wptouch_power_pack_debug( '...CACHE is disabled' );
		return false;
	}

	// Don't cache AJAX requests
	if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) == 'xmlhttprequest' ) {
		wptouch_power_pack_debug( '...Disabling CACHE due to AJAX request' );
		return false;
	}

	// Make sure our magic cookie is set
	if ( !isset( $_COOKIE[ WPTOUCH_CACHE_COOKIE ] ) ) {
		wptouch_power_pack_debug( '...Disabling due to lack of magic cookie' );
		return false;
	}

	// Don't cache comment forms that have cookies set
	$cookie_array = array_flip( $_COOKIE );
	if ( count( preg_grep( '/comment_author_/', $cookie_array ) ) > 0  || isset( $_COOKIE[ 'comment_author' ] ) || isset( $_COOKIE[ 'comment_author_email' ] ) || isset( $_COOKIE[ 'comment_author_url'] ) ) {
		wptouch_power_pack_debug( '...Disabling CACHE due to COMMENT cookies being set' );
		return false;
	}

	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	// Don't cache any page requests, either GET or POST
	if ( count( $_GET ) || count( $_POST ) ) {
		wptouch_power_pack_debug( '...Disabling CACHE due to GET or POST' );
		return false;
	}

	// See if we can cache desktop users
	if ( $_COOKIE[ WPTOUCH_CACHE_COOKIE ] == 'desktop' ) {
		if ( !$settings->cache_enable_desktop ) {
			wptouch_power_pack_debug( '...Disabling CACHE due to DESKTOP theme and disabled setting' );
			return false;
		}
	}

	// Check for password protected pages
	foreach( $_COOKIE as $cookie_name => $cookie_value ) {
  		if ( strncmp( $cookie_name, "wp-postpass_", 12 ) == 0 ) {
  			// Page has a password
  			wptouch_power_pack_debug( '...Disabling CACHE due to Password Protected post' );
  			return false;
  		}
  	}

	// Check for ignored pages
	if ( $settings->cache_ignored_urls ) {
		$urls = explode( "\n", trim( strtolower( $settings->cache_ignored_urls ) ) );
		if ( is_array( $urls ) && count( $urls ) ) {
			$page_uri = strtolower( $_SERVER[ 'REQUEST_URI' ] );
			foreach( $urls as $url ) {
				if ( $url != '' ) {
					if ( strpos( $page_uri, $url ) !== false ) {
						// Don't cache this page
						wptouch_power_pack_debug( '...Disabling CACHE due to matched URL fragment' );
						return false;
					}
				}
			}
		}
	}

	// Don't cache pages of logged in users
	if ( is_user_logged_in() ) {
		wptouch_power_pack_debug( '...Disabling CACHE due to logged in user' );
		return false;
	}

	// Don't cache admin pages
	if ( is_admin() ) {
		return false;
	}

	// Check for woo pages
	if ( class_exists( 'WooCommerce' ) ) {
		$page_ids = array(
			get_option( 'woocommerce_cart_page_id' ),
			get_option( 'woocommerce_checkout_page_id' ),
			get_option( 'woocommerce_pay_page_id' ),
			get_option( 'woocommerce_view_order_page_id' ),
			get_option( 'woocommerce_thanks_page_id' )
		);

		foreach( $page_ids as $page_id ) {
			if ( is_numeric( $page_id ) && is_page( $page_id ) ) {
				// Don't cache this WooCommerce page
				return false;
			}
		}
	}

	wptouch_power_pack_debug( '...CACHE is ENABLED' );
	return apply_filters( 'wptouch_addon_cache_current_page', true );
}

function wptouch_power_pack_get_cache_filename() {
	if ( in_array( $_COOKIE[ WPTOUCH_CACHE_COOKIE ], array( 'mobile', 'desktop', 'mobile-desktop' ) ) ) {
		$prefix = $_COOKIE[ WPTOUCH_CACHE_COOKIE ];
	}

	return POWER_PACK_CACHE_DIR . '/' .  $prefix . '/' . wptouch_power_pack_get_cache_key() . '.html';
}

function wptouch_power_pack_get_cache_key() {
	$key_string = $_SERVER[ 'HTTP_HOST' ] . $_SERVER[ 'REQUEST_URI'];

	// Add device class
	if ( $_COOKIE[ WPTOUCH_CACHE_COOKIE ] == 'mobile' ) {
		global $wptouch_pro;
		$key_string = $key_string . $wptouch_pro->active_device_class;
	}

	return md5( $key_string );
}

function wptouch_power_pack_cache_is_file_expired( $file_time ) {
	return ( $file_time < wptouch_power_pack_cache_get_valid_time() ) || ( ( time() - $file_time ) > wptouch_power_pack_cache_expiry_time() );
}

function wptouch_power_pack_cache_check_modified( $etag ) {
	if ( isset( $_SERVER[ 'HTTP_IF_NONE_MATCH' ] ) ) {
		$stored_etag = trim( $_SERVER[ 'HTTP_IF_NONE_MATCH' ] );
		if ( $stored_etag == $etag ) {
			header( 'HTTP/1.1 304 Not Modified' );
			die;
		}
	}
}

function wptouch_power_pack_cache_do_cache() {
	// Check for caching
	if ( wptouch_power_pack_cache_should_cache_page() ) {
		$cache_file = wptouch_power_pack_get_cache_filename();
		wptouch_power_pack_debug( 'CACHE file is ' . $cache_file );
		if ( file_exists( $cache_file ) ) {
			wptouch_power_pack_debug( '...found CACHE file; checking modification time' );
			$last_modify_time = filemtime( $cache_file );
			if ( !wptouch_power_pack_cache_is_file_expired( $last_modify_time ) ) {
				wptouch_power_pack_debug( '...file is still valid, serving' );
				// Serve cache file
				global $wptouch_pro;

				$cache_data = false;

				$etag = 0;
				if ( wptouch_power_pack_cache_can_use_gzip() && isset( $_SERVER[ 'HTTP_ACCEPT_ENCODING'] ) && strpos( $_SERVER[ 'HTTP_ACCEPT_ENCODING'], 'gzip' ) !== false && function_exists( 'gzcompress' ) && file_exists( $cache_file  . '.gz' ) ) {
					wptouch_power_pack_debug( '......Serving GZIP version of CACHE' );
					// Check for browsing cache
					if ( wptouch_power_pack_cache_browser_cache_enabled() ) {
						$etag = md5( filemtime( $cache_file  . '.gz' ) . wptouch_power_pack_cache_expiry_time() );
						wptouch_power_pack_cache_check_modified( $etag );
					}

					// Load file (with gzip encoding)
					$cache_data = $wptouch_pro->load_file( $cache_file  . '.gz' );

					// Fix Windows newlines in PHP
					if ( defined( 'POWER_PACK_CACHE_WINDOWS' ) ) {
						$cache_data = str_replace( "\r\n", "\n", $cache_data );
					}

					// Unserialize
					$cache_data = unserialize( $cache_data );

					header( 'Content-Encoding: gzip' );
					header( 'Content-Length: ' . strlen( $cache_data->body ) );
				} else {
					// Check for browsing cache
					if ( wptouch_power_pack_cache_browser_cache_enabled() ) {
						$etag = md5( filemtime( $cache_file ) . wptouch_power_pack_cache_expiry_time() );
						wptouch_power_pack_cache_check_modified( $etag );
					}

					// Load file
					$cache_data = $wptouch_pro->load_file( $cache_file );

					// Fix Windows newlines
					if ( defined( 'POWER_PACK_CACHE_WINDOWS' ) ) { $cache_data = str_replace( "\r\n", "\n", $cache_data ); }

					// Unserialize
					$cache_data = unserialize( $cache_data );
				}

				$skip_items = array( 'X-Power-Pack' );
				if ( wptouch_power_pack_cache_browser_cache_enabled() ) {
					$skip_items[] = 'Pragma: no-cache';
				}

				foreach( $cache_data->headers as $header ) {
					$skip = false;
					foreach( $skip_items as $item ) {
						if ( strpos( $header, $item ) !== false ) {
							$skip = true;
						}
					}

					if ( $skip ) {
						continue;
					}

					wptouch_power_pack_debug( '......Adding CACHE Header ' . $header );
					header( $header );
				}

				header( 'X-Power-Pack: Cache Hit' );

				if ( wptouch_power_pack_cache_browser_cache_enabled() ) {
					wptouch_power_pack_debug( '......Adding CACHE-CONTROL headers' );
					header( 'ETag: ' . $etag );
					header( 'Cache-Control: max-age=3600, must-revalidate' );
				}

				wptouch_power_pack_debug( '...Serving CACHE data' );
				echo $cache_data->body;
				die;
			}
		}

		global $infinity_cache_info;

		$infinity_cache_info = new stdClass;
		$infinity_cache_info->cookies = $_COOKIE;
		$infinity_cache_info->generation_time = time();

		wptouch_power_pack_debug( 'Starting to generate CACHE file' );

		ob_start( 'wptouch_power_pack_handle_cache_done' );
	} else {
		// Check for one of those settings and for showing mobile theme
		global $wptouch_pro;

		if ( ( !is_admin() && !is_user_logged_in() ) && $wptouch_pro->is_showing_mobile_theme_on_mobile_device() ) {
			ob_start( 'wptouch_power_pack_handle_cache_done' );
		}
	}
}

function wptouch_power_pack_cdn_get_urls() {
	$urls = array();
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	for ( $i = 1; $i <= POWER_PACK_MAX_CDN_URL; $i++ ) {
		$name = 'media_optimize_cdn_prefix_' . $i;
		if ( strlen( $settings->$name ) ) {
			$urls[] = $settings->$name;
		}
	}

	return $urls;
}

function wptouch_power_pack_cache_cdnize( $content ) {
	wptouch_power_pack_debug( '...Checking for CDN' );
	if ( !wptouch_power_pack_cache_is_cdn_enabled() ) {
		wptouch_power_pack_debug( '......CDN is disabled' );
		return $content;
	}

	$match_string = '#[\'"]([^\s\'"]*)\.(jpg|css|png|js|gif|ico)#iU';

	// $match_string = '#(href|src)=([\'"](.*)\.(jpg|css|png|js|gif|ico)[\'"\?])#iU';
	$result = preg_match_all( $match_string, $content, $matches );
	if ( $result ) {
		wptouch_power_pack_debug( '...found ' . count( $matches[0] ) . ' URLS' );

		$find = array();
		$replace = array();
		$cdn_urls = wptouch_power_pack_cdn_get_urls();

		$count = 0;
		foreach( $matches[0] as $find_url ) {
			$count++;

			$cdn_url = $cdn_urls[ $count % count( $cdn_urls ) ];

			if ( !preg_match( '#(http|https|\/\/)#i', $find_url, $new_matches ) ) {
				// Relative URL

				$actual_url = $matches[ 3 ][ $count-1 ];
				$new_url = str_replace( $actual_url, $cdn_url . $actual_url, $find_url );

				$find[] = $find_url;
				$replace[] = $new_url;

				wptouch_power_pack_debug( '......replacing ' . $find_url . ' WITH ' . $new_url );
			} else {
				$home_url = home_url();
				$new_url = str_replace( $home_url, $cdn_url, $find_url );

				$find[] = $find_url;
				$replace[] = $new_url;

				wptouch_power_pack_debug( '......replacing ' . $find_url . ' WITH ' . $new_url );
			}
		}

		return str_replace( $find, $replace, $content );
	} else {
		wptouch_power_pack_debug( '....NO URLs found :(' );
		return $content;
	}
}

function wptouch_power_pack_handle_cache_done( $buffer ) {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );

	$new_buffer = wptouch_power_pack_cache_cdnize( $buffer );

	if ( $settings->minification_css_enable ) {
		$css_optimizer = new Power_Pack_CSS_Optimizer;

		$new_buffer = $css_optimizer->process( $new_buffer );
	}

	if ( $settings->minification_js_enable ) {
		$js_optimizer = new Power_Pack_JS_Optimizer;

		$new_buffer = $js_optimizer->process( $new_buffer );
	}

	if ( wptouch_power_pack_cache_should_cache_page() ) {
		global $infinity_cache_info;

		$infinity_cache_info->headers = @headers_list();

		wptouch_power_pack_debug( '...attempting to save cache file' );
		$cache_file_name = wptouch_power_pack_get_cache_filename();
		$file_handle = fopen( $cache_file_name, 'w+t' );
		if ( $file_handle ) {
			$infinity_cache_info->body = $new_buffer;

			fwrite( $file_handle, serialize( $infinity_cache_info ) );
			fclose( $file_handle );

			wptouch_power_pack_debug( '.....saved as ' . $cache_file_name );
		} else {
			wptouch_power_pack_debug( '.....unable to save' );
		}

		if ( function_exists( 'gzcompress') && wptouch_power_pack_cache_can_use_gzip() ) {
			$file_handle = fopen( wptouch_power_pack_get_cache_filename() . '.gz', 'w+b' );
			if ( $file_handle ) {
				$infinity_cache_info->body = gzcompress( $new_buffer, 9, ZLIB_ENCODING_GZIP );

				fwrite( $file_handle, serialize( $infinity_cache_info ) );
				fclose( $file_handle );
			}
		}
	}

	return $new_buffer;
}

function wptouch_power_pack_cache_get_all_subdirectories() {
	return array(
		POWER_PACK_CACHE_DIR . '/desktop',
		POWER_PACK_CACHE_DIR . '/mobile',
		POWER_PACK_CACHE_DIR . '/mobile-desktop'
	);
}

function _wptouch_power_pack_cache_cleanup( $remove_all = false ) {
	require_once( WPTOUCH_DIR . '/core/file-operations.php' );
	$directories = wptouch_power_pack_cache_get_all_subdirectories();
	foreach( $directories as $dir ) {
		$files = wptouch_get_files_in_directory( $dir, '.html', true );
		if ( count( $files ) ) {
			foreach( $files as $file ) {
				if ( $remove_all ) {
					// Remove all files, can be used when cache should be invalidated
					unlink( $file );
				} else {
					// Remove only old files
					$last_modified_time = filemtime( $file );
					if ( wptouch_power_pack_cache_is_file_expired( $last_modified_time ) ) {
						unlink( $file );
					}
				}
			}
		}
	}
}

function wptouch_power_pack_cache_check_cron() {
	if ( !wp_next_scheduled( 'wptouch_addon_cache_cleanup_event' ) ) {
		$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );

		wp_schedule_event( time(), $settings->cache_garbage_collection_interval, 'wptouch_addon_cache_cleanup_event' );
	}
}

function wptouch_power_pack_cache_do_scheduled_cleanup() {
	_wptouch_power_pack_cache_cleanup();
}

function wptouch_power_pack_cache_flush() {
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	$settings->cache_last_flush_time = time();
	$settings->save();
}

function wptouch_power_pack_cache_add_directory() {
	if ( !file_exists( POWER_PACK_CACHE_DIR ) ) {
		wptouch_create_directory_if_not_exist( POWER_PACK_CACHE_DIR );
	}

	$sub_dirs = apply_filters( 'wptouch_addon_cache_subdirectories', wptouch_power_pack_cache_get_all_subdirectories() );

	foreach( $sub_dirs as $dir ) {
		if ( !file_exists( $dir ) ) {
			wptouch_create_directory_if_not_exist( $dir );
		}
	}

	wptouch_power_pack_cache_check_cron();
}

function wptouch_power_pack_cache_handle_ajax_reset() {
	if ( current_user_can( 'manage_options') ) {
		_wptouch_power_pack_cache_cleanup( true );
	}
}

function wptouch_power_pack_should_cache_desktop(){
	$settings = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	if ( $settings->cache_enable_desktop == true ) {
		return true;
	} else {
		return false;
	}
}

function wptouch_power_pack_cache_render_setting( $setting ) {
	$setting_info = wptouch_get_settings( ADDON_SETTING_DOMAIN );
	$this_setting = $setting->name;

	if ( false && isset( $setting_info->$this_setting ) && $setting_info->$this_setting ) {
		echo '<div class="cdn">' . sprintf( __( 'An example for this URL is: %s%s%s.', 'wptouch-pro' ), '<strong>', '</strong>', str_replace( home_url(), $setting_info->$this_setting, WPTOUCH_URL . '/readme.txt' ) ) . '</div>';
	}
}
