<?php
// TOOLSET TO CREATE AND MANAGE SUBSCRIPTIONS THROUGH (WOOCOMMERCE ..) ORDERS

class pcpp_subscriptions {
	public $order_error = false; // cached resource for errors reporting in order creation
	
    
	// cached resources to store trial period duration
	public $trial_duration = false; 
	public $trial_dur_type = 'days'; 
	
    
	
	/* KNOW IF THERE'S TRIAL PERIOD - used also to setup static properties 
	 * @return (bool)
	 */
	public function is_trial_enabled() {
		if($this->trial_duration === false) {
			$this->trial_duration = (int)get_option('pcpp_trial_duration');
			$this->trial_dur_type = get_option('pcpp_trial_type', 'days');
		}
		
		return ($this->trial_duration) ? true : false;
	}
	
	
    
	
	/*
	 * get available plans
	 * @param (bool) $strip_no_registr = whether to exclude plans not to be shown in registration forms
	 * @param (bool) $strip_no_renew = whether to exclude plans not to be shown in renewal form
	 * @plan_id (int) eventually returns only specific plan name
	 *
	 * @return (array/string) plans associative array (plan_id => name) or the plan name requested
	 */
	public function get_plans($strip_no_registr = true, $strip_no_renew = false, $plan_id = false) {
		$GLOBALS['pcpp_is_querying plans'] = true;
		
		// check for cache - only for stripped fields
		if(!$strip_no_registr && !$strip_no_renew && !$plan_id && isset($GLOBALS['pcpp_avail_plans'])) {
			unset($GLOBALS['pcpp_is_querying plans']);
			return $GLOBALS['pcpp_avail_plans'];
		}
		
		$plans = array();
		if(pcpp_static::avail_ecomm() === array()) { // no e-comm installed
			unset($GLOBALS['pcpp_is_querying plans']);
			return $plans;
		} 
		
		
		// get subject
		$chosen = get_option('pcpp_ecomm_to_use', pcpp_static::avail_ecomm(true));
	
		if($chosen == 'woocomm') {
			$args = array(
				'post_type' 		=> 'product',
				'post_status' 		=> 'publish',
				'posts_per_page' 	=> 100,
				'orderby'			=> 'menu_order',
				'order'				=> 'ASC',
                'meta_key'          => '_is_pvtcontent_plan',
                'meta_value'        => 'yes',
			);
			
			$query = new WP_Query($args);
			foreach($query->posts as $post) {
				$prod_id = pcpp_static::wpml_get_orig_id($post->ID);
				
				if($strip_no_registr && get_post_meta($prod_id, 'pcpp_not_in_forms', true)) {
					continue;
				}
				
				if($strip_no_renew && get_post_meta($prod_id, 'pcpp_not_in_renew', true)) {
					continue;
				}
				
                if(WC_Product_Factory::get_product_type($prod_id) == 'subscription' && !pcpp_woo_subscr_static::supports_woo_subscr()) {
                    continue;    
                }
                
				$plans[ $prod_id ] = $post->post_title;	
			}
		}
		
		
		/* PCPP FILTER - allows fetched plans array alteration - passes an associative array('plan-id' =>plan-name) and $strip_no_registr, $strip_no_renew */
		$plans = apply_filters('pcpp_fetched_plans', $plans, $strip_no_registr, $strip_no_renew);
		
		
		// set cache - only for stripped fields
		if(!$strip_no_registr && !$plan_id){
			$GLOBALS['pcpp_avail_plans'] = $plans;	
		}
		
		unset($GLOBALS['pcpp_is_querying plans']);
		return ($plan_id && isset($plans[$plan_id])) ? $plans[$plan_id] : $plans; 
	}
	
	
    
	
	/*
	 * GET PLAN'S DATA
	 * @param (int) $plan_id = plan id (custom taxonomy's post id) 
	 * @return (array|false) associative array containing plan's data or false if plan is not found
	 */ 
	public function get_plan_data($plan_id) {
		if(!defined('PCPP_ECOMMERCE')) {
            pcpp_ecomm_to_use();
        }
        
        $data = array(
            'id' => $plan_id
        );
		
		// get price
        $data['price'] = 0;	
        $data['currency_price'] = '\$0';
        
		if(PCPP_ECOMMERCE == 'woocomm') {
			$product = wc_get_product($plan_id);	
            
            if(!is_object($product)) {
                return false;    
            }
            
			$data['price'] = $product->get_price();
			$data['currency_price'] = $this->raw_to_pcpp_price( $product->get_price_html() );
		}
		
		// get duration
		$duration = (int)get_post_meta($plan_id, 'pcpp_plan_duration', true);
		$dur_type = get_post_meta($plan_id, 'pcpp_plan_dur_type', true);
		
		$data['duration'] 		= $duration;
		$data['duration_type'] 	= $dur_type;
		$data['duration_secs'] 	= $duration * pcpp_static::periods_to_sec($dur_type);
		$data['duration_txt'] 	= (!$data['duration_secs']) ? __('lifetime') : $duration.' '.pcpp_static::sing_plur_switch($dur_type, $duration);
		
		// get title and description 
        $data['title'] = get_post_field('post_title', pcpp_static::wpml_get_trans_id($plan_id));
		$data['txt'] = get_post_field('post_content', pcpp_static::wpml_get_trans_id($plan_id));

        // is a woo subscription? (recurring payment)
        $data['recurr_pay'] = ($product->is_type('subscription')) ? true : false; 
        
		return $data;
	}
	
    
    
    
    
    /* Given the raw price code returned by the ecommerce system, returns the pcpp format */
    public function raw_to_pcpp_price($code) {
        
        if(PCPP_ECOMMERCE == 'woocomm') {
            if(strpos($code, '<del') !== false) {
                preg_match('/<del(.*?)<\/del>/s', $code, $matches);   
                
                if(is_array($matches) && count($matches)) {
                    $code = str_replace($matches[0], '', $code);    
                }
            }
            elseif(strpos($code, 'subscription-details') !== false) {
                preg_match('/<span class="subscription-details">(.*?)<\/span>/s', $code, $matches);   
                
                if(is_array($matches) && count($matches)) {
                    $code = str_replace($matches[0], '', $code);    
                }
            }
            
            $code = strip_tags($code);
        }
        
        return $code;    
    }
    
    
    
	
    
	
	/* CREATE ORDER 
	 * @param (int) $user_id - privatecontent user ID to associate
	 * @param (int|array) $plan_id - plan to associate to the order (if empty will be queried). Might use an IDs array if want to add also duration extensions
	 * @param (array) $user_data - associative array containing data useful for the order creation (if empty - a query will be performed)
     * @param (array) $fees - additional fees (positive or negative) to apply to order (eg. duration extension or plan's change discount) - array(array('FEE NAME', $amount), ...)
	 * @param (string) $coupon - discount coupon to eventually use during order creation
     *
	 * @return (obj|false) order object if successfully created - otherwise false
	 */
	public function create_order($user_id, $plan_id = false, $user_data = array(), $fees = array(), $coupon = false) {
		global $pc_users, $pc_meta, $pc_wp_user;
		$extensions = array();
        
        // plan + extension - differentiate
        if(is_array($plan_id)) {
            $extensions = $plan_id;
            $plan_id = $extensions[0];
            unset($extensions[0]);
        }
        
		// whether to query also associated plan ID
		elseif(empty($plan_id)) {
			$plan_id = (isset($user_data['pcpp_plan'])) ? $user_data['pcpp_plan'] : $pc_meta->get_meta($user_id, 'pcpp_plan'); 	
		}
		
		// whether to get user's data
		if(empty($user_data) || !isset($user_data['wp_user_id'])) {
			$user_data = $pc_users->get_user($user_id, $args = array('to_get' => array('name', 'surname', 'username', 'email', 'tel', 'wp_user_id', 'psw')));	
		}
		else {
			if(!isset($user_data['psw'])) {
				$user_data['psw'] = $pc_users->get_user_field($user_id, 'psw');	
			}
		}
        
        
        
        // PCPP-ACTION - allows extra operations right before a new order is created
        do_action('pcpp_pre_order_creation', $user_id, $user_data['wp_user_id'], $plan_id, $fees, $coupon);
        
        
        
		// woocommerce order
		if(PCPP_ECOMMERCE == 'woocomm') {
            $is_woo_subscr = pcpp_woo_subscr_static::plan_is_subscr($plan_id);
            
            // recurring subscription - be sure user and system is okay
            if($is_woo_subscr && pcpp_woo_subscr_static::supports_woo_subscr() && empty($user_data['wp_user_id'])) {
                $this->order_error = __('Recurring payments require a WP synced user', PCPP_ML);
				return false;    
            }
            
            
            // create
			$args = array(
				'customer_id' => (empty($user_data['wp_user_id'])) ? false : $user_data['wp_user_id'], // if WP-synced - attach user
			);
            
			$order = wc_create_order($args);
            $order_id = trim(str_replace('#', '', $order->get_order_number()));
            
            $order->add_product( wc_get_product($plan_id), 1); // 1 == quantity

            
            // update user's billing and shipping address 
			if(!empty($user_data['wp_user_id'])) {
				$wc_user_meta = array(
                    'billing_first_name'=> $user_data['name'],
                    'billing_last_name' => $user_data['surname'],
                    'billing_email'     => $user_data['email'],
					'billing_company'	=> $this->billing_cust_field($user_id, 'pcpp_bill_company_f'),
					'billing_address_1'	=> $this->billing_cust_field($user_id, 'pcpp_bill_address_f'),
                    'billing_address_2'	=> $this->billing_cust_field($user_id, 'pcpp_bill_address2_f'),
					'billing_city'     	=> $this->billing_cust_field($user_id, 'pcpp_bill_city_f'),
					'billing_postcode'	=> $this->billing_cust_field($user_id, 'pcpp_bill_postcode_f'),
					'billing_country' 	=> $this->billing_cust_field($user_id, 'pcpp_bill_country_f'),
					'billing_state'		=> $this->billing_cust_field($user_id, 'pcpp_bill_state_f'),
					'billing_phone' 	=> $user_data['tel'],
					
                    'shipping_first_name'   => $user_data['name'],
                    'shipping_last_name'    => $user_data['surname'],
					'shipping_company'		=> $this->billing_cust_field($user_id, 'pcpp_bill_company_f'),
					'shipping_address_1'	=> $this->billing_cust_field($user_id, 'pcpp_bill_address_f'),
                    'shipping_address_2'	=> $this->billing_cust_field($user_id, 'pcpp_bill_address2_f'),
					'shipping_city'     	=> $this->billing_cust_field($user_id, 'pcpp_bill_city_f'),
					'shipping_postcode'		=> $this->billing_cust_field($user_id, 'pcpp_bill_postcode_f'),
					'shipping_country' 		=> $this->billing_cust_field($user_id, 'pcpp_bill_country_f'),
					'shipping_state'		=> $this->billing_cust_field($user_id, 'pcpp_bill_state_f'),
					'shipping_phone' 		=> $user_data['tel'],
				);
				
				foreach($wc_user_meta as $meta_key => $meta_value) {
					update_user_meta($user_data['wp_user_id'], $meta_key, $meta_value );
				}
			}
            

			// set billing address
            $billing = array(
                'first_name' => $user_data['name'],
                'last_name'  => $user_data['surname'],
                'company'    => $this->billing_cust_field($user_id, 'pcpp_bill_company_f'),
                'email'      => $user_data['email'],
                'phone'      => $user_data['tel'],
                'address_1'  => $this->billing_cust_field($user_id, 'pcpp_bill_address_f'),
                'address_2'  => $this->billing_cust_field($user_id, 'pcpp_bill_address2_f'),
                'city'       => $this->billing_cust_field($user_id, 'pcpp_bill_city_f'),
                'postcode'   => $this->billing_cust_field($user_id, 'pcpp_bill_postcode_f'),
                'country'    => $this->billing_cust_field($user_id, 'pcpp_bill_country_f'),
                'state'      => $this->billing_cust_field($user_id, 'pcpp_bill_state_f'),
            );
            $GLOBALS['pcpp_woo_billing_data'] = $billing;
            $order->set_address($billing, 'billing');
			
            
            // PCPP-ACTION - allow extra actions after having set woo billing fields 
            do_action('pcpp_after_woo_billing_setup', $order, $order_id, $user_id, $user_data['wp_user_id']);
            
            
            
            // plan extensions?
            if(count($extensions)) {
                foreach($extensions as $ext_db_id) {
                    $order->add_product( wc_get_product($ext_db_id), 1, array(
                        'name' => pcpp_dur_ext_static::get_ext_title($ext_db_id, $plan_id)
                    ));     
                }
            }
            
            
            // additional fees - not for Woo Subscription
            if(!empty($fees) && !pcpp_woo_subscr_static::plan_is_subscr($plan_id)) {
                foreach($fees as $fee) {
                    
                    $item_fee = new WC_Order_Item_Fee();
                    
                    $item_fee->set_name($fee[0]);
                    $item_fee->set_amount( (float)$fee[1] );
                    $item_fee->set_total( (float)$fee[1] );
                    
                    $order->add_item($item_fee);
                }
            }
            
            
            // PCPP-FILTER - allow extra control over WOO order object before calculating totals
            $order = apply_filters('pcpp_pre_woo_order_totals', $order, $user_id, $plan_id);
            
            
            $order->calculate_totals();
            
            
			// if there's a coupon
			if($order->get_total() > 0 && !empty($coupon)) {
				$coupon_inst = new WC_Coupon($coupon);
				$coup_amount = $coupon_inst->get_amount();
				$discount_amount = 0;
                
                // apply coupon to plan
                $product = new WC_Product($plan_id);
                $product_price = $product->get_price();
                
                if(
                    !empty($coup_amount) && 
                    $coupon_inst->is_valid_for_product($product) && 
                    $product_price >= $coupon_inst->get_minimum_amount() && 
                    (!$coupon_inst->get_maximum_amount() || $product_price <= $coupon_inst->get_maximum_amount())
                ) {
                    $order->apply_coupon($coupon);
                }
			}
            
            
			// check for errors
			if(is_wp_error($order)) {
				$this->order_error = $order->get_error_message();
				return false;
			} 
            
            
            
            // woo subscription? create it!
            if($is_woo_subscr) {
                $woo_subscr = pcpp_woo_subscr_static::create_subscr($order_id, $plan_id, $user_data['wp_user_id'], $user_id);   
                
                if(is_wp_error($woo_subscr)) {
                    $this->order_error = $woo_subscr->get_error_message();
                    
                    wp_delete_post($order_id, true); // delete newly created order
                    return false;
                }
                else {
                    $pc_meta->update_meta($user_id, 'pcpp_recurr_subscr', $woo_subscr->get_id());
                    update_post_meta($woo_subscr->get_id(), 'is_pvtcontent_subscr', 1);    
                }
            }
            else {
                $pc_meta->delete_meta($user_id, 'pcpp_recurr_subscr');    
            }

            
			// since July 2018 WooComm changes user psw - then have to force it
			if(!empty($user_data['wp_user_id'])) {
				global $wpdb;
				
				$pc_wp_user->sync_psw_to_wp($user_data['psw'], (int)$user_data['wp_user_id']);	
				$wpdb->update(PC_USERS_TABLE, array('psw' => $user_data['psw']),  array('id' => (int)$user_id));
			}
			

			// calculate total
			$order->calculate_totals();

            // free order - mark as completed
			if($order->get_total() <= 0) {
				$order->update_status('completed', '', false);
			}
            
            
			// add meta to mark it as pvtontent order
            update_post_meta($order_id, 'is_pvtcontent_order', 1);
            update_post_meta($order_id, 'pvtcontent_user_id', $user_id);
			
            
            // PCPP-ACTION - allow order object manipulation - passes order object and its ID + user ID and WP user ID
            do_action('pcpp_extra_order_actions', $order, $order_id, $user_id, $user_data['wp_user_id']);
            
			return $order;
		}
		
		return false;	
	}
	
    
    
    
    /* given the order object, returns the order ID basing on the available ecommerce platform */
    public function order_obj_to_id($obj) {
        $id = false;
        if(!is_object($obj)) {
            return $id;    
        }
        
        if(PCPP_ECOMMERCE == 'woocomm') {
            $id = $obj->get_id();    
        }
        
        return $id;
    }
    
    
    
    
    /* given the order object, returns the order total amount basing on the available ecommerce platform */
    public function order_obj_to_amount($obj) {
        $amount = false;
        if(!is_object($obj)) {
            return $amount;    
        }
        
        if(PCPP_ECOMMERCE == 'woocomm') {
            $amount = $obj->get_total();    
        }
        
        return (float)$amount;
    }
    
    
    
	
	// return user meta to fill specific billing data
	public function billing_cust_field($user_id, $field) {
		// woocomm user orders list - abort
		if(isset($_GET['post_type']) && $_GET['post_type'] = 'shop_order' && isset($_GET['_customer_user'])) {
            return false;
        }
		
		if(isset($GLOBALS['pcpp_billing_custom_fields']) && isset($GLOBALS['pcpp_billing_custom_fields'][$user_id])) {
			return (isset($GLOBALS['pcpp_billing_custom_fields'][$user_id][$field])) ? $GLOBALS['pcpp_billing_custom_fields'][$user_id][$field] : '';
		}
		
		global $pc_users;
		$fields = array(
			'pcpp_bill_company_f' 	=> get_option('pcpp_bill_company_f'),
			'pcpp_bill_address_f' 	=> get_option('pcpp_bill_address_f'),	
			'pcpp_bill_city_f' 		=> get_option('pcpp_bill_city_f'),	
			'pcpp_bill_state_f'		=> get_option('pcpp_bill_state_f'),
			'pcpp_bill_country_f'	=> get_option('pcpp_bill_country_f'),
			'pcpp_bill_postcode_f'	=> get_option('pcpp_bill_postcode_f'),		
		);
		foreach($fields as $index => $val) {
			if(empty($val)) {
                unset($fields[$index]);
            }	
		}
		
		// no fields associated 
		if(empty($fields)) {
			$GLOBALS['pcpp_billing_custom_fields'] = array();
			return '';	
		}
		
		$u_data = $pc_users->get_user($user_id, array('to_get' => array_values($fields) ));	
		$GLOBALS['pcpp_billing_custom_fields'] = array();
		$GLOBALS['pcpp_billing_custom_fields'][$user_id] = array();

		foreach($fields as $pcpp_key => $db_key) {
			$GLOBALS['pcpp_billing_custom_fields'][$user_id][$pcpp_key] = (empty($db_key)) ? '' : $u_data[$db_key];	
		}
		
		return (isset($GLOBALS['pcpp_billing_custom_fields'][$user_id][$field])) ? $GLOBALS['pcpp_billing_custom_fields'][$user_id][$field] : ''; 
	}

    

    
	// given the order ID - returns checkout URL
	public function checkout_url($order_id) {
		if(PCPP_ECOMMERCE == 'woocomm') {
			$checkout_url = untrailingslashit(wc_get_checkout_url());
			$link_char3 = (strpos($checkout_url, '?') === false) ? '?' : '&';
			$link_char2 = ($link_char3 == '&') ? '=' : '/';
			$link_char1 = ($link_char3 == '&') ? '&' : '/';
			
			$url = $checkout_url . $link_char1 .'order-pay'. $link_char2 . $order_id . $link_char3 .'pay_for_order=true&key='. get_post_meta($order_id, '_order_key', true);
		}
		
		return $url;
	}
	

    
    

	/* GET ORDER object */
	public function get_order($order_id) {
		if(PCPP_ECOMMERCE == 'woocomm') {
			$order = wc_get_order($order_id);	
		}
		
		return $order;
	}
	
	
    
    
	
	/* GET ORDER STATUS - given order ID */
	public function get_order_status($order_id) {
		$status = ''; // no status by default
		
		if(PCPP_ECOMMERCE == 'woocomm') { 
            $order = wc_get_order($order_id); 
            $wc_status = $order->get_status();
            
			switch($wc_status) {
				case 'pending'   : 
                case 'on-hold'   :
					$status = 'unpaid'; break;	
					
				case 'cancelled' : 
				case 'refunded'  : 
				case 'failed'    : 
                case 'expired'   : 
                case 'pending-cancel' : 
					$status = 'expired'; break;	
					
				case 'completed' : 
                case 'active'    :  
                case 'processing':     
					$status = 'active'; break;			
			}
		}
				
		//trial/unpaid/expired/active
		return $status;
	}
	
	
    
	
	/* 
	 * CANCEL ORDER - given order ID cancel it if isn't paid or already cancelled and eventually add a note
	 */
	public function cancel_order($order_id, $note = false) {
		if(!in_array($this->get_order_status($order_id), array('active', 'expired'))) {
			if(PCPP_ECOMMERCE == 'woocomm') {
				$order = $this->get_order($order_id);
				$order->update_status('cancelled', $note.' -', true);	
			}
		}
		
		return true;	
	}
	
    
    
    
    
    /* 
     * GET ORDER PRODUCTS - given order id 
     * @return (array) associative array with involved products and eventual duration extension ('plan_id' => ... , 'dur_ext_id' => ...)  
     */
	public function get_order_products($order_id) {
		$products = array();
        
        if(PCPP_ECOMMERCE == 'woocomm') {
			$order = wc_get_order((int)$order_id);
            if(!$order) {
                trigger_error('PCPP - WooCommerce - Order not found');
                return false;    
            }
            
			$items = $order->get_items();
        
            foreach($items as $item_data) {
                if(!$item_data->get_variation_id()) {
                    $products['plan_id'] = $item_data->get_product_id();    
                } else {
                    $products['dur_ext_id'] = $item_data->get_variation_id();          
                }
            }
		}
		
		return $products;
	}
    
    
    
    
	/* GET ORDER'S PLAN - given order id, returns related pcpp plan id */
	public function get_order_plan_id($order_id) {
		$products = $this->get_order_products($order_id);
		return (isset($products['plan_id'])) ? $products['plan_id'] : false;
	}
	
	
    
    
	/* GET PLAN EXPIRATION TIME - RELATED TO ORDER DATE OR USER ACTUAL EXPIRATION (if extending the plan)
	 * @param (int) $order_id - order ID
	 * @param (string) $starting_date = when to start period (order = order payment/submission date | now = current date). It is OVERRIDED if order is a duration extension
     *
	 * @return (mixed) expiration date in time format or "unlimited" or empty string if is a recurring payment
	 */
	public function order_plan_expiration($order_id, $starting_date = 'order') {
		global $pc_meta;
        
        if(PCPP_ECOMMERCE == 'woocomm') {
			$order = wc_get_order($order_id);
            if(!$order) {
                trigger_error('PCPP - WooCommerce - Order not found');
                return false;    
            }
            
			// get order's plan duration
            $products = $this->get_order_products($order_id);
            if(!isset($products['plan_id'])) {
                return false;    
            }
            
			$plan_data = $this->get_plan_data($products['plan_id']);
			
			// if unlimited
            if(empty($plan_data['duration'])) {
				$exp = 'unlimited';	
			} 
            else {	
                $is_dur_ext = $this->is_plan_expir_ext_order($order_id); // returns false or 'Y-m-d H:i:s' 
                
                if($is_dur_ext) {
                    $rel_user_id = get_post_meta($order_id, 'pvtcontent_user_id', true); 
                    $user_exp = $pc_meta->get_meta($rel_user_id, 'pcpp_subscr_end');
                    
                    $start_from = (is_numeric($user_exp)) ? date('Y-m-d H:i:s', $user_exp) : false;
                }
                elseif($starting_date == 'order') {
                    
                    $start_from = (in_array($order->get_status(), array('processing', 'completed'))) ? 
                        date('Y-m-d H:i:s', $order->get_date_paid()->getOffsetTimestamp()) : 
                        get_post_time('Y-m-d H:i:s', $gmt = false, $order_id);
                }
                else { // now
                    $start_from = false;    
                }

                // compile
                $date = new DateTime($start_from);
                $date->modify('+'. (int)$plan_data['duration'] .' '. $plan_data['duration_type']);

                if(isset($products['dur_ext_id'])) {
                    $ext_dur_data = pcpp_dur_ext_static::get_ext_data($products['dur_ext_id'], $products['plan_id'], $search_by = 'db_id');
                    $date->modify('+'. (int)$ext_dur_data['duration'] .' '. $ext_dur_data['dur_type']);        
                }

                $exp = $date->format('U');
			}
		}
		
		return $exp; 	
	}
	
	
    
    
    /* RETURNS THE DISCOUNT AMOUNT TO APPLY FOR A PLAN'S CHANGE
     * @param (int) $order_id - order ID
     * @param (int) $active_plan - plan ID associated to user
	 * @param (int) $subscr_end = user subscription's end date (timestamp)
     *
	 * @return (float) amount
	 */
    public function plan_change_discount($order_id, $active_plan, $subscr_end) {
		if($subscr_end == 'unlimited') {
            return 0;    
        }
        
        $remaining_secs = (int)$subscr_end - current_time('timestamp'); 
        if($remaining_secs < 60 * 60) { // at least 1 hour!
            return 0;
        }
        
        // get active plan duration
        $plan_data = $this->get_plan_data($active_plan);
        if(!$plan_data || empty($plan_data['duration'])) { // unlimited plan
            return 0;    
        }
        
        
        if(PCPP_ECOMMERCE == 'woocomm') {
            $order = wc_get_order($order_id);
            if(!$order) {
                trigger_error('PCPP - WooCommerce - Order not found');
                return false;    
            }
            
            $amount_spent = (float)$order->get_total();
            if(empty($amount_spent)) {
                return 0;    
            }
            
            // calculate basing on the real amount of time purchased
            $date_paid = $order->get_date_paid();
            
            // know how much paid per second
            $amount_per_sec = $amount_spent / ((int)$subscr_end - $date_paid->date('U'));
            $to_discount = $remaining_secs * $amount_per_sec;
            
            if($to_discount > $amount_spent) {
                $to_discount = $amount_spent;    
            }
        }
        
        
        // PCPP FILTER - allows plan change discount management - passes order ID, plan ID, active suscription end date
        $to_discount = apply_filters('pcpp_plan_change_discount', round($to_discount, 2), $order_id, $active_plan, $subscr_end);
        return $to_discount;
    }
    
    
    

    /* KNOW WHETHER AN EXISTING ORDER IS FREE */
	public function order_is_free($order_id, $order_obj = false) {
		if(PCPP_ECOMMERCE == 'woocomm') {
            if(!$order_obj) {
                $order_obj = wc_get_order((int)$order_id);
                if(!$order_obj) {
                    trigger_error('PCPP - WooCommerce - Order not found');
                    return false;    
                }
            }

            return ($order_obj->get_total() <= 0) ? true : false;		
        }
        
        return false;
	}
    
    

    
	/* KNOW WHETHER AN ORDER BELONGS TO PVTCONTENT */
	public function is_pvtcontent_order($order_id) {
		return (get_post_meta($order_id, 'is_pvtcontent_order', true)) ? true : false;		
	}
    
    
    
    
    /* KNOW WHETHER AN ORDER IS PURELY A DURATION EXTENSION */
	public function is_plan_expir_ext_order($order_id) {
		return (get_post_meta($order_id, 'pcpp_is_expir_ext', true)) ? get_post_meta($order_id, 'pcpp_is_expir_ext', true) : false;		
	}
    
    
    
    
    
    ////////////////////////////////////////////////////////
    
    
    
    
    
    /* KNOW WHETHER THERE'S A SUPPORTED INVOICING SYSTEM INSTALLED */
    public function invoicing_system_exists() {
        if(PCPP_ECOMMERCE == 'woocomm') {
            
            // https://wordpress.org/plugins/woocommerce-pdf-invoices-packing-slips/
            if(is_plugin_active('woocommerce-pdf-invoices-packing-slips/woocommerce-pdf-invoices-packingslips.php')) { 
                return true;    
            }
        }
        
        return false;
    }
    
    
    
    
    /* GIVEN ORDER ID - RETURNS HTML LINK FOR INVOICE (must be used only in frontend) */
    public function get_invoice_html_link($order_id) {
        $code = '';
        
        if(PCPP_ECOMMERCE == 'woocomm') {
            
            // https://wordpress.org/plugins/woocommerce-pdf-invoices-packing-slips/
            if(is_plugin_active('woocommerce-pdf-invoices-packing-slips/woocommerce-pdf-invoices-packingslips.php')) { 
                if(!wcpdf_get_invoice((int)$order_id)) {
                    return '';    
                }
                
                $code = strip_tags(do_shortcode('[wcpdf_download_invoice link_text="%%TXT%%" order_id="'. (int)$order_id .'"]'), '<a>');
            }
        }
        
        $txt = '<i class="far fa-file-pdf" title="'. esc_attr__('download invoice', PCPP_ML) .'"></i>';
        return str_replace('%%TXT%%', $txt, $code);
    }
    
}

$GLOBALS['pcpp_subscr'] = new pcpp_subscriptions();