<?php
/**
 * WooCommerce Mixpanel
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Mixpanel to newer
 * versions in the future. If you wish to customize WooCommerce Mixpanel for your
 * needs please refer to http://docs.woocommerce.com/document/mixpanel/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginFramework\v5_5_0 as Framework;

/**
 * WooCommerce Mixpanel main plugin class.
 *
 * @since 1.2
 */
class WC_Mixpanel extends Framework\SV_WC_Plugin {


	/** plugin version number */
	const VERSION = '1.17.0';

	/** @var WC_Mixpanel single instance of this plugin */
	protected static $instance;

	/** plugin id */
	const PLUGIN_ID = 'mixpanel';

	/** @var \WC_Mixpanel_Subscriptions_Integration instance */
	protected $subscriptions_integration;

	/**
	 * Initializes the plugin
	 *
	 * @since 1.2
	 */
	public function __construct() {

		parent::__construct(
			self::PLUGIN_ID, self::VERSION, array(
				'text_domain' => 'woocommerce-mixpanel',
			)
		);

		// load integration
		$this->includes();
	}


	/**
	 * Initializes the lifecycle handler.
	 *
	 * @since 1.13.0
	 */
	protected function init_lifecycle_handler() {

		require_once( $this->get_plugin_path() . '/includes/Lifecycle.php' );

		$this->lifecycle_handler = new \SkyVerge\WooCommerce\Mixpanel\Lifecycle( $this );
	}


	/**
	 * Include required files
	 *
	 * @since 1.2
	 */
	public function includes() {

		require_once( $this->get_plugin_path() . '/includes/class-wc-mixpanel-integration.php' );

		if ( $this->is_plugin_active( 'woocommerce-subscriptions.php' ) ) {
			$this->subscriptions_integration = $this->load_class( '/includes/class-wc-mixpanel-subscriptions-integration.php', 'WC_Mixpanel_Subscriptions_Integration' );
		}

		add_filter( 'woocommerce_integrations', array( $this, 'load_integration' ) );
	}


	/**
	 * Adds Mixpanel to the list of integrations WooCommerce loads.
	 *
	 * @since 1.2
	 *
	 * @param string[] $integrations
	 * @return string[]
	 */
	public function load_integration( $integrations ) {

		$integrations[] = 'WC_Mixpanel_Integration';

		return $integrations;
	}


	/**
	 * Gets the integration handler instance.
	 *
	 * @since 1.6.0
	 *
	 * @return \WC_Mixpanel_Integration
	 */
	public function get_integration() {

		$integrations = WC()->integrations->get_integrations();

		return $integrations['mixpanel'];
	}


	/**
	 * Gets the Subscriptions integration handler instance.
	 *
	 * @since 1.7.1
	 *
	 * @return \WC_Mixpanel_Subscriptions_Integration
	 */
	public function get_subscriptions_integration_instance() {

		return $this->subscriptions_integration;
	}


	/**
	 * Gets the main Mixpanel instance.
	 *
	 * Ensures that only one instance is/can be loaded.
	 *
	 * @since 1.5.0
	 *
	 * @return WC_Mixpanel
	 */
	public static function instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}


	/**
	 * Gets the plugin name, localized.
	 *
	 * @since 1.3
	 *
	 * @return string the plugin name
	 */
	public function get_plugin_name() {

		return __( 'WooCommerce Mixpanel', 'woocommerce-mixpanel' );
	}


	/**
	 * Gets __FILE__.
	 *
	 * @since 1.3
	 *
	 * @return string the full path and filename of the plugin file
	 */
	protected function get_file() {

		return __FILE__;
	}


	/**
	 * Gets the plugin documentation URL
	 *
	 * @since 1.3.0
	 *
	 * @return string
	 */
	public function get_documentation_url() {

		return 'https://docs.woocommerce.com/document/mixpanel/';
	}


	/**
	 * Gets the plugin support URL.
	 *
	 * @since 1.6.0
	 *
	 * @return string
	 */
	public function get_support_url() {

		return 'https://woocommerce.com/my-account/marketplace-ticket-form/';
	}


	/**
	 * Gets the plugin sales page URL.
	 *
	 * @since 1.13.1
	 *
	 * @return string
	 */
	public function get_sales_page_url() {

		return 'https://woocommerce.com/products/mixpanel/';
	}


	/**
	 * Gets the URL to the settings page.
	 *
	 * @since 1.3
	 *
	 * @param string $_ unused
	 * @return string
	 */
	public function get_settings_url( $_ = '' ) {

		return admin_url( 'admin.php?page=wc-settings&tab=integration&section=mixpanel' );
	}


	/**
	 * Determines if the current page is the settings page.
	 *
	 * @since 1.3
	 *
	 * @return bool
	 */
	public function is_plugin_settings() {

		return isset( $_GET['page'], $_GET['tab'], $_GET['section'] )
			&& 'wc-settings' === $_GET['page']
			&& 'integration' === $_GET[ 'tab' ]
			&& 'mixpanel'    === $_GET[ 'section' ];
	}


}


/**
 * Returns the One True Instance of Mixpanel.
 *
 * @since 1.13.0
 *
 * @return \WC_Mixpanel
 */
function wc_mixpanel() {

	return WC_Mixpanel::instance();
}
