<?php
/**
 * WooCommerce Mixpanel
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Mixpanel to newer
 * versions in the future. If you wish to customize WooCommerce Mixpanel for your
 * needs please refer to http://docs.woocommerce.com/document/mixpanel/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Mixpanel;

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginFramework\v5_5_0 as Framework;

/**
 * Plugin lifecycle handler.
 *
 * @since 1.13.0
 *
 * @method \WC_Mixpanel get_plugin()
 */
class Lifecycle extends Framework\Plugin\Lifecycle {


	/** @var string settings option key */
	private $settings_option_key = 'woocommerce_mixpanel_settings';


	/**
	 * Lifecycle constructor.
	 *
	 * @since 1.13.1
	 *
	 * @param \WC_Mixpanel $plugin
	 */
	public function __construct( $plugin ) {

		parent::__construct( $plugin );

		$this->upgrade_versions = [
			'1.6.1',
			'1.8.0',
		];
	}


	/**
	 * Gets the plugin's raw settings.
	 *
	 * @since 1.13.1
	 *
	 * @return array associative array
	 */
	private function get_plugin_settings() {

		$settings = get_option( $this->settings_option_key, [] );

		return is_array( $settings ) ? $settings : [];
	}


	/**
	 * Updates the plugin raw settings.
	 *
	 * @since 1.13.1
	 *
	 * @param array $settings associative array
	 */
	private function update_plugin_settings( array $settings ) {

		update_option( $this->settings_option_key, $settings );
	}


	/**
	 * Updates to v1.6.1
	 *
	 * @since 1.13.1
	 */
	protected function upgrade_to_1_6_1() {

		$settings = $this->get_plugin_settings();

		// ensure total_initial_payment_property_name is not set to "subscription name"
		if ( isset( $settings['total_initial_payment_property_name'] ) && 'subscription name' === $settings['total_initial_payment_property_name'] ) {
			$settings['total_initial_payment_property_name'] = 'total initial payment';
		}

		// add product price property name
		$settings['product_price_property_name'] = 'product price';

		if ( $this->get_plugin()->is_plugin_active( 'woocommerce-subscriptions.php' ) ) {
			$settings['subscription_id_property_name']               = 'subscription id';
			$settings['subscription_price_property_name']            = 'subscription price';
			$settings['subscription_end_of_prepaid_term_event_name'] = 'subscription prepaid term ended';
		}

		$this->update_plugin_settings( $settings );
	}


	/**
	 * Updates to v1.8.0
	 *
	 * @since 1.13.0-dev.1
	 */
	protected function upgrade_to_1_8_0() {

		$settings = $this->get_plugin_settings();

		$settings['completed_payment_event_name'] = 'Completed Payment';

		$this->update_plugin_settings( $settings );
	}


}
