<?php

class WC_Autoship_Payflow_Gateway extends WC_Payment_Gateway_CC {
	
	protected $_sandbox_url = 'https://pilot-payflowpro.paypal.com';
	protected $_url = 'https://payflowpro.paypal.com';
	protected $_token;
	protected $_token_expiration = 0;
	protected $_title;
	protected $_partner;
	protected $_vendor;
	protected $_user;
	protected $_pwd;
	protected $_authorize_only;
	protected $_sandbox_mode;
	
	public function __construct() {
		// WooCommerce fields
		$this->id = 'wc_autoship_payflow';
		$this->icon = '';
		$this->has_fields = true;
		$this->method_title = __( "WC Autoship Payflow Payments", 'wc-autoship' );
		$this->method_description = __( 
			"PayPal Payflow Pro payment gateway for WooCommerce and WC Autoship",
			'wc-autoship'
		);
		$this->description = $this->method_description;
		// WooCommerce settings
		$this->init_form_fields();
		$this->init_settings();
		// Assign settings
		$this->title = $this->get_option( 'title' );
		$this->_partner = $this->get_option( 'partner' );
		$this->_vendor = $this->get_option( 'vendor' );
		$this->_user = $this->get_option( 'user' );
		$this->_pwd = $this->get_option( 'pwd' );
		$this->_authorize_only = $this->get_option( 'authorize_only' );
		$this->_sandbox_mode = $this->get_option( 'sandbox_mode' );
		// Supports
		$this->supports = array(
			'refunds',
			'tokenization'
		);
		// Payment gateway hooks
		add_action( 
			'woocommerce_update_options_payment_gateways_' . $this->id, 
			array( $this, 'process_admin_options' )
		);
	}
	
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled' => array(
				'title' => __( 'Enable/Disable', 'wc-autoship' ),
				'type' => 'checkbox',
				'label' => __( 'Enable ' . $this->method_title, 'wc-autoship' ),
				'default' => 'yes'
			),
			'title' => array(
				'title' => __( 'Title', 'wc-autoship' ),
				'type' => 'text',
				'description' => __( 
					'This controls the title which the user sees during checkout.', 'wc-autoship'
				),
				'default' => __( 'Credit Card', 'wc-autoship' ),
				'desc_tip' => true,
			),
			'license_key' => array(
				'title' => __( 'License Key', 'wc-autoship' ),
				'description' => __( 'Enter your software license key issued after purchase.', 'wc-autoship' ),
				'desc_tip' => true,
				'type' => 'text'
			),
			'partner' => array(
				'title' => __( 'Partner', 'wc-autoship' ),
				'description' => __( 'Payflow parter (ex: paypal)', 'wc-autoship' ),
				'desc_tip' => true,
				'type' => 'text'
			),
			'vendor' => array(
				'title' => __( 'Merchant Login', 'wc-autoship' ),
				'description' => __( 'Payflow merchant or vendor name', 'wc-autoship' ),
				'desc_tip' => true,
				'type' => 'text'
			),
			'user' => array(
				'title' => __( 'User', 'wc-autoship' ),
				'description' => __( 'Payflow user', 'wc-autoship' ),
				'desc_tip' => true,
				'type' => 'text'
			),
			'pwd' => array(
				'title' => __( 'Password', 'wc-autoship' ),
				'description' => __( 'Payflow password', 'wc-autoship' ),
				'desc_tip' => true,
				'type' => 'password'
			),
			'authorize_only' => array(
				'title' => __( 'Payment Mode', 'wc-autoship' ),
				'label' => __( 'Authorize Only', 'wc-autoship' ),
				'description' => __( 'Only authorize payments', 'wc-autoship' ),
				'desc_tip' => __( 
					'Use this option to capture payments later.',
					'wc-autoship' 
				),
				'type' => 'checkbox'
			),
			'sandbox_mode' => array(
				'title' => __( 'Sandbox Mode', 'wc-autoship' ),
				'description' => __( 'Enable sandbox mode', 'wc-autoship' ),
				'desc_tip' => __( 
					'Select this option to send transactions to the test gateway.',
					'wc-autoship'
				),
				'type' => 'checkbox',
				'default' => 'yes'
			)
		);
	}
	
	public function payment_fields() {
		$customer_id = get_current_user_id();
		$payment_tokens = WC_Payment_Tokens::get_customer_tokens( $customer_id, $this->id );
		include WC_AUTOSHIP_PAYFLOW_PAYMENTS_PLUGIN_DIR . '/templates/payment-fields.php';
	}
	
	public function process_payment( $order_id ) {
		// Get our order
		$order = wc_get_order( $order_id );
		// Customer ID
		$customer_id = $order->get_user_id();
		// Token
		$token = null;
		if ( empty( $_POST['wc-wc_autoship_payflow-payment-token'] ) || $_POST['wc-wc_autoship_payflow-payment-token'] == 'new' ) {
			$payflow_token = $_POST['wc_autoship_payflow_token'];
			$token = $this->add_customer_payment_method( $customer_id, $_POST );
		} else {
			// Get the selected token
			$token_id = $_POST['wc-wc_autoship_payflow-payment-token'];
			$token = WC_Payment_Tokens::get( $token_id );
		}
		// Create sale data
		$data = array(
			'TRXTYPE' => 'S',
			'TENDER' => 'C',
			'ORIGID' => $token->get_token(),
			'CURRENCY' => get_woocommerce_currency(),
			'AMT' => $order->get_total(),
			'COMMENT1' => __( 'WC Order #' . $order->get_order_number(), 'wc-autoship-payflow-payments' )
		);
		$data = array_merge( $data, $this->get_customer_data( $customer_id ) );
		$result = $this->send_request( $data );
		if ( '0' === $result['RESULT'] ) {
			$order->payment_complete( $result['PNREF'] );
			$order->add_payment_token( $token );
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order )
			);
		}
		throw new Exception( $result['RESPMSG'] );
	}

	public function add_payment_method() {
		// Customer ID
		$customer_id = get_current_user_id();
		try {
			// Add payment method
			$this->add_customer_payment_method( $customer_id, $_POST );
			// Return result
			return array(
				'result'   => 'success',
				'redirect' => wc_get_endpoint_url( 'payment-methods' )
			);
		} catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
			return;
		}
	}
	
	public function add_customer_payment_method( $customer_id, $form_data ) {
		// Create sale data
		$exp_data = explode( '/', $_POST['wc_autoship_payflow_exp'] );
		$data = array(
			'TRXTYPE' => 'A',
			'TENDER' => 'C',
			'ACCT' => preg_replace( '/[^\d]/', '', $form_data['wc_autoship_payflow_number'] ),
			'EXPDATE' => sprintf( '%s%s', $exp_data[0], substr( $exp_data[1], -2 ) ),
			'CVV2' => $form_data['wc_autoship_payflow_cvv'],
			'CURRENCY' => get_woocommerce_currency(),
			'AMT' => '1.00',
			'COMMENT1' => __( 'WC Autoship Auth for customer #' . $customer_id, 'wc-autoship-payflow-payments' )
		);
		$data = array_merge( $data, $this->get_customer_data( $customer_id ) );
		$result = $this->send_request( $data );
		if ( '0' === $result['RESULT'] ) {
			// Void the 1.00 authorization
			$void_data = array(
				'TRXTYPE' => 'V',
				// PNREF of authorization
				'ORIGID' => $result['PNREF']
			);
			$void_response_data = $this->send_request( $void_data );
			if ( '0' === $void_response_data['RESULT'] ) {
				// Void was successful
				$token = new WC_Payment_Token_CC();
				$token->set_user_id( $customer_id );
				$token->set_token( $void_response_data['PNREF'] );
				$token->set_gateway_id( $this->id );
				$token->set_card_type( __( 'Credit Card', 'wc-autoship-payflow-payments' ) );
				$token->set_last4( substr( $form_data['wc_autoship_payflow_number'], -4 ) );
				$token->set_expiry_month( $exp_data[0] );
				$token->set_expiry_year( $exp_data[1] );
				$saved = $token->save();
				if ( ! $saved ) {
					throw new Exception( __( 'Error saving WC payment token!', 'wc-autoship-payflow-payments' ) );
				}
				return $token;
			}
			throw new Exception( $void_response_data['RESPMSG'] );

		}
		throw new Exception( $result['RESPMSG'] );
	}
	
	/**
	 * Process a refund for an order
	 * @see WC_Payment_Gateway::process_refund()
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		if ( $amount == null ) {
			return false;
		}
		
		// Do refund
		$data = array(
			'TRXTYPE' => 'C',
			'ORIGID' => get_post_meta( $order_id, '_transaction_id', true ),
			'AMT' => $amount
		);
		$response_data = $this->send_request( $data );
		if ( $response_data['RESPMSG'] == 'Approved' ) {
			// Refund was successful
			return true;
		}
		throw new Exception( $response_data['RESPMSG'] );
	}
	
	public function validate_fields() {
		return true;
	}

	public function get_customer_data( $customer_id ) {
		$data = array();
		$data['BILLTOFIRSTNAME'] = get_user_meta( $customer_id, 'billing_first_name', true );
		$data['BILLTOLASTNAME'] = get_user_meta( $customer_id, 'billing_last_name', true );
		$data['BILLTOSTREET'] = get_user_meta( $customer_id, 'billing_address_1', true );
		$data['BILLTOSTREET2'] = get_user_meta( $customer_id, 'billing_address_2', true );
		$data['BILLTOCITY'] = get_user_meta( $customer_id, 'billing_city', true );
		$data['BILLTOSTATE'] = get_user_meta( $customer_id, 'billing_state', true );
		$data['BILLTOZIP'] = get_user_meta( $customer_id, 'billing_postcode', true );
		$data['BILLTOCOUNTRY'] = get_user_meta( $customer_id, 'billing_country', true );
		$data['SHIPTOFIRSTNAME'] = get_user_meta( $customer_id, 'shipping_first_name', true );
		$data['SHIPTOLASTNAME'] = get_user_meta( $customer_id, 'shipping_last_name', true );
		$data['SHIPTOSTREET'] = get_user_meta( $customer_id, 'shipping_address_1', true );
		$data['SHIPTOCITY'] = get_user_meta( $customer_id, 'shipping_city', true );
		$data['SHIPTOSTATE'] = get_user_meta( $customer_id, 'shipping_state', true );
		$data['SHIPTOZIP'] = get_user_meta( $customer_id, 'shipping_postcode', true );
		$data['SHIPTOCOUNTRY'] = get_user_meta( $customer_id, 'shipping_country', true );
		$user = new WP_User( $customer_id );
		$data['EMAIL'] = $user->user_email;
		return $data;
	}
	
	public function get_url() {
		if ( $this->_sandbox_mode == 'yes' ) {
			return $this->_sandbox_url;
		}
		return $this->_url;
	}
	
	public function get_auth() {
		$auth = array(
			'USER' => $this->_user,
			'VENDOR' => $this->_vendor,
			'PARTNER' => $this->_partner,
			'PWD' => $this->_pwd,
			'BUTTONSOURCE' => 'Patterns_SI_Custom'
		);
		return $auth;
	}
	
	/**
	 * Send a request
	 * @param string $data Data to be sent in the body payload not including auth params
	 * @return array Response query string data
	 */
	public function send_request( $data ) {
		$url = $this->get_url();
		$ch = curl_init( $url );
		if ( ! $ch ) {
			throw new Exception( 'Could not open connection' );
		}
		curl_setopt( $ch, CURLOPT_POST, 1 );
		curl_setopt( $ch, CURLOPT_SSL_VERIFYHOST, 2 );
		curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, 0 );
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
		curl_setopt( $ch, CURLOPT_HEADER, 0 );
		
		$auth = $this->get_auth();
		$data = array_merge( $auth, $data );
		$payload = $this->encode_url_data( $data );
		curl_setopt( $ch, CURLOPT_POSTFIELDS, $payload );
		$response = curl_exec( $ch );
		curl_close( $ch );
		$parsed_response = array();
		parse_str( $response, $parsed_response );
		
		return $parsed_response;
	}
	
	public function process_admin_options() {
		parent::process_admin_options();
		
		if ( ! isset( $_POST['woocommerce_wc_autoship_payflow_license_key'] ) ) {
			return false;
		}
		$license = $_POST['woocommerce_wc_autoship_payflow_license_key'];

		$api_params = array( 
			'edd_action' => 'activate_license', 
			'license' => $license, 
			'item_name' => urlencode( 'WC Autoship Payflow Payments' ),
			'url' => home_url()
		);
		
		$response = wp_remote_get( add_query_arg( $api_params, 'https://wooautoship.com' ), array( 
			'timeout' => 15, 'sslverify' => false 
		) );

		if ( is_wp_error( $response ) ) {
			WC_Admin_Settings::add_error( __( 'Error validating your license key!', 'wc-autoship' ) );
			return true;
		}

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		
		if ( $license_data->success ) {
			WC_Admin_Settings::add_message( __( 'Your license key is valid!', 'wc-autoship' ) );
		} elseif ( ! $license_data->success ) {
			if ( $license_data->error == 'expired' ) {
				WC_Admin_Settings::add_error( __( 'Your license key expired on ', 'wc-autoship' )
					. date( get_option( 'date_format' ), strtotime( $license_data->expires ) )
				);
			} else {
				WC_Admin_Settings::add_error( __( 'Your license key is invalid!', 'wc-autoship' ) );
			}
		}
		
		return true;
	}
	
	public function encode_url_value( $value ) {
		$encoded = str_replace(
			array(
				'&', '='
			),
			array(
				'%26', '%3D'
			), 
			$value
		);
		return $encoded;
	}
	
	public function encode_url_data( $data ) {
		$params = array();
		foreach ( $data as $name => $value ) {
			$params[] = $this->encode_url_value( $name ) . '=' . $this->encode_url_value( $value );
		}
		return implode( '&', $params );
	}
}

?>