<?php

namespace WcPsigate\Admin;

use WcPsigate\Helpers\Factories;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class that represents admin notices.
 *
 * @since 1.5.0
 */
class Admin_Notices {
	
	/**
	 * All allowed notices
	 * @var array
	 */
	public $allowed_notices = array(
		'valid_ssl',
		'phpver',
		'wcver',
		'curl',
		'not_enabled_am_credentials_subs',
		'missing_am_credentials',
		'not_enabled_am_credentials_preorder',
	);
	/**
	 * Collection of all active notices
	 * @var array
	 */
	public $active_notices = array();
	protected $prefix;
	protected $id;
	
	/**
	 * Admin_Notices constructor.
	 */
	public function __construct() {
		$this->gateway_id = 'psigate';
		$this->id         = 'psigate';
		$this->prefix     = 'wc_psigate';
	}
	
	/**
	 * Hooks. Should be loaded only once
	 */
	public function hooks() {
		add_action( 'admin_notices', array( $this, 'maybe_admin_notices' ) );
		add_action( 'wp_loaded', array( $this, 'hide_notices' ) );
	}
	
	/**
	 * Adds an allowed notice slug, so we can retrieve and update it
	 *
	 * @since 1.5.0
	 *
	 * @param $slug
	 */
	public function add_allowed_notice( $slug ) {
		$this->allowed_notices[] = $slug;
	}
	
	/**
	 * Adds a notice to the display list
	 *
	 * @since 1.5.0
	 *
	 * @param      $slug
	 * @param      $type
	 * @param      $message
	 * @param bool $dismissible
	 */
	public function add_notice( $slug, $type, $message, $dismissible = false ) {
		
		$map_class = array(
			'error'   => 'error',
			'notice'  => 'notice notice-warning',
			'warning' => 'notice notice-error',
		);
		
		$this->active_notices[ $slug ] = array(
			'class'       => $map_class[ $type ],
			'message'     => $message,
			'dismissible' => $dismissible,
		);
	}
	
	/**
	 * Updates the display status of a notice
	 *
	 * @since 1.5.0
	 *
	 * @param $slug
	 * @param $value
	 */
	public function update_notice( $slug, $value ) {
		update_option( $this->prefix . '_show_notice_' . $slug, $value );
	}
	
	/**
	 * Loads the notices
	 * @since 1.5.0
	 */
	public function maybe_admin_notices() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		
		$this->perform_checks();
		
		foreach ( (array) $this->active_notices as $notice_key => $notice ) {
			echo '<div class="' . esc_attr( $notice['class'] ) . '" style="position:relative;">';
			
			if ( $notice['dismissible'] ) {
				?>
				<a href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'wc-' . $this->id . '-hide-notice', $notice_key ), $this->prefix . '_hide_notice_nonce', $this->prefix . '_notice_nonce' ) ); ?>" class="woocommerce-message-close notice-dismiss" style="position:absolute;right:1px;padding:9px;text-decoration:none;"></a>
				<?php
			}
			
			echo '<p>';
			echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ) ) );
			echo '</p></div>';
		}
	}
	
	public function get_notices_values() {
		$values = array();
		foreach ( $this->allowed_notices as $name ) {
			$values[ $name ] = 'no' == get_option( $this->prefix . '_show_notice_' . $name ) ? false : true;
		}
		
		return $values;
	}
	
	/**
	 * Performs the plugin checks
	 * @since 1.5.0
	 */
	public function perform_checks() {
		$notices_values = $this->get_notices_values();
		
		/**
		 * @var \WC_Gateway_PsiGate $gateway
		 */
		$gateway = Factories::get_gateway( $this->gateway_id );
		
		if ( false == $gateway ) {
			return;
		}
		
		// Bail, if the gateway is not enabled
		if ( 'yes' !== $gateway->enabled ) {
			return;
		}
		
		$testmode = 'yes' == $gateway->testmode;
		
		if ( $notices_values['phpver'] ) {
			if ( version_compare( phpversion(), \WC_PsiGate::MIN_PHP_VERSION, '<' ) ) {
				$message = __( 'WooCommerce PsiGate - The minimum PHP version required for this plugin is %1$s. You are running %2$s.', \WC_PsiGate::TEXT_DOMAIN );
				
				$this->add_notice( 'phpver', 'error', sprintf( $message, \WC_PsiGate::MIN_PHP_VERSION, phpversion() ), true );
				
				return;
			}
		}
		
		if ( $notices_values['wcver'] ) {
			$wc_version = defined( 'WC_VERSION' ) ? WC_VERSION : '1.0.0';
			if ( version_compare( $wc_version, \WC_PsiGate::MIN_WC_VERSION, '<' ) ) {
				$message = __( 'WooCommerce PsiGate - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', \WC_PsiGate::TEXT_DOMAIN );
				
				$this->add_notice( 'wcver', 'notice', sprintf( $message, \WC_PsiGate::MIN_WC_VERSION, $wc_version ), true );
				
				return;
			}
		}
		
		if ( $notices_values['curl'] ) {
			if ( ! function_exists( 'curl_init' ) ) {
				$this->add_notice( 'curl', 'notice', __( 'WooCommerce PsiGate - cURL is not installed.', \WC_PsiGate::TEXT_DOMAIN ), true );
			}
		}
		
		if ( $notices_values['valid_ssl'] ) {
			if ( ! wc_checkout_is_https() && ! $testmode ) {
				$this->add_notice( 'valid_ssl', 'notice', sprintf( __( 'WooCommerce PsiGate - live payments are enabled, but an SSL certificate is not detected. Your checkout may not be secure! Please ensure your server has a valid <a href="%1$s" target="_blank">SSL certificate</a>', \WC_PsiGate::TEXT_DOMAIN ), 'https://en.wikipedia.org/wiki/Transport_Layer_Security' ), true );
			}
		}
		
		if ( $notices_values['missing_am_credentials'] ) {
			$is_saving_customers = 'yes' == $gateway->save_customers;
			$am_password         = $gateway->acc_password;
			$am_user             = $gateway->acc_user_id;
			
			if ( $is_saving_customers && ( '' == $am_password || '' == $am_user ) ) {
				$this->add_notice( 'missing_am_credentials', 'warning', sprintf( __(
					'WooCommerce PsiGate - You are allowing customers to pay with saved cards, but the Account Manager API is not configured in the plugin settings. %sClick here%s to set the settings', \WC_PsiGate::TEXT_DOMAIN
				), '<a href="' . $this->get_setting_link() . '#woocommerce_psigate_am_api_settings">', '</a>' ), true );
			}
		}
		
		if ( $notices_values['not_enabled_am_credentials_subs'] && \WC_PsiGate::is_subscriptions_active() ) {
			$am_password = $gateway->acc_password;
			$am_user     = $gateway->acc_user_id;
			
			if ( '' == $am_password || '' == $am_user ) {
				$this->add_notice( 'not_enabled_am_credentials_subs', 'warning', sprintf( __(
					'WooCommerce PsiGate - You have WC Subscriptions plugin active, but the Account Manager API is not configured in the plugin settings. Your recurring payments will not work. %sClick here%s to set the settings', \WC_PsiGate::TEXT_DOMAIN
				), '<a href="' . $this->get_setting_link() . '#woocommerce_psigate_am_api_settings">', '</a>' ), true );
			}
		}
		
		if ( $notices_values['not_enabled_am_credentials_preorder'] && \WC_PsiGate::is_pre_orders_active() ) {
			$am_password = $gateway->acc_password;
			$am_user     = $gateway->acc_user_id;
			
			if ( '' == $am_password || '' == $am_user ) {
				$this->add_notice( 'not_enabled_am_credentials_preorder', 'warning', sprintf( __(
					'WooCommerce PsiGate - You have WC Pre-Orders plugin active, but the Account Manager API is not configured in the plugin settings. Your release payments will not work. %sClick here%s to set the settings', \WC_PsiGate::TEXT_DOMAIN
				), '<a href="' . $this->get_setting_link() . '#woocommerce_psigate_am_api_settings">', '</a>' ), true );
			}
		}
		
		// Allow 3rd party to add notices
		do_action( $this->prefix . '_admin_notices_checks', $notices_values, $this );
	}
	
	/**
	 * Hides any admin notices.
	 *
	 * @since 1.5.0
	 */
	public function hide_notices() {
		if ( isset( $_GET[ 'wc-' . $this->id . '-hide-notice' ] ) && isset( $_GET[ $this->prefix . '_notice_nonce' ] ) ) {
			if ( ! wp_verify_nonce( $_GET[ $this->prefix . '_notice_nonce' ], $this->prefix . '_hide_notice_nonce' ) ) {
				wp_die( __( 'Action failed. Please refresh the page and retry.', \WC_PsiGate::TEXT_DOMAIN ) );
			}
			
			if ( ! current_user_can( 'manage_woocommerce' ) ) {
				wp_die( __( 'Cheatin&#8217; huh?', \WC_PsiGate::TEXT_DOMAIN ) );
			}
			
			$notice = wc_clean( $_GET[ 'wc-' . $this->id . '-hide-notice' ] );
			
			if ( in_array( $notice, $this->allowed_notices ) ) {
				$this->update_notice( $notice, 'no' );
			}
		}
	}
	
	/**
	 * Get setting link.
	 *
	 * @since 1.5.0
	 *
	 * @return string Setting link
	 */
	public function get_setting_link() {
		$use_id_as_section = function_exists( 'WC' ) ? version_compare( WC()->version, '2.6', '>=' ) : false;
		
		// TODO: needs to revisit the integration part as get_gateway_class will return hosted or direct class integration
		$section_slug = $use_id_as_section ? $this->id : \WC_PsiGate::get_gateway_class();
		
		return admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . $section_slug );
	}
}