<?php

namespace WcPsigate\Admin;

use WcPsigate\Compatibility;
use WcPsigate\Helpers\Factories;
use WcPsigate\Psigate_Order;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since  1.5.0
 * @author VanboDevelops | Ivan Andreev
 *
 *        Copyright: (c) 2019 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Capture {
	
	public function __construct() {
		$this->gateway_id = 'psigate';
	}
	
	/**
	 * Loads the capture action for the plugin
	 *
	 * This is loaded here and not in the gateway
	 * because we need it to load a little bit earlier for the action to be added to order edit screen
	 *
	 * @since 3.2.0
	 */
	public function hooks() {
		/**
		 * @var \WC_Payment_Gateway|\WC_Gateway_PsiGate $gateway
		 */
		$gateway = Factories::get_gateway( $this->gateway_id );
		
		// This is a custom filter that will run the capture_payment process and return its response
		add_filter( 'wc_psigate_capture_payment_for_order', array( $gateway, 'capture_payment' ), 10, 2 );
		
		add_filter( 'woocommerce_order_actions', array( $gateway, 'add_order_capture_action' ) );
		add_action( 'woocommerce_order_action_psigate_capture_payment', array(
			$gateway,
			'capture_payment'
		) );
		
		add_action( 'woocommerce_admin_order_totals_after_refunded', array(
			$this,
			'order_meta_box_add_capture_field'
		) );
		add_action( 'woocommerce_order_item_add_action_buttons', array(
			$this,
			'order_meta_box_add_capture_payment_buttons'
		) );
	}
	
	/**
	 * Adds Capture buttons to the admin order edit screen
	 *
	 * @since 3.2.0
	 *
	 * @param \WC_Order $order
	 *
	 * @return bool
	 */
	public function order_meta_box_add_capture_payment_buttons( $order ) {
		$method = Compatibility::get_prop( $order, 'payment_method' );
		if ( $this->gateway_id != $method ) {
			return false;
		}
		
		$ps_order               = new Psigate_Order( $order );
		$is_captured            = $ps_order->get_is_payment_captured();
		$allowed_order_statuses = self::get_capture_allowed_order_statuses();
		
		if ( $ps_order->is_subscription()
		     || $is_captured
		     || ! in_array( $order->get_status(), $allowed_order_statuses ) ) {
			return false;
		}
		
		?>
		<button type="button" class="button wc-psigate-capture-payment-init">
			<?php _e( 'Capture', \WC_PsiGate::TEXT_DOMAIN ); ?>
		</button>
		<span class="wc-psigate-capture-payment-wrapper" style="display:none">
				<?php echo wc_help_tip( __( 'Enter the amount you want to capture above and press "Capture Payment" button.', \WC_PsiGate::TEXT_DOMAIN ) ); ?>
			<button type="button" class="button button-primary wc-psigate-capture-payment">
				<?php _e( 'Capture Payment', \WC_PsiGate::TEXT_DOMAIN ); ?>
			</button>
			</span>
		<button type="button" class="button wc-psigate-capture-cancel" style="display:none">
			<?php _e( 'Cancel Capture', \WC_PsiGate::TEXT_DOMAIN ); ?>
		</button>
		<?php
	}
	
	/**
	 * Adds the capture amount field to the admin order edit screen
	 *
	 * @since 3.2.0
	 *
	 * @param $order_id
	 *
	 * @return bool
	 */
	public function order_meta_box_add_capture_field( $order_id ) {
		$order = wc_get_order( $order_id );
		
		$method = Compatibility::get_prop( $order, 'payment_method' );
		if ( $this->gateway_id != $method ) {
			return false;
		}
		
		$ps_order               = new Psigate_Order( $order );
		$is_captured            = $ps_order->get_is_payment_captured();
		$allowed_order_statuses = self::get_capture_allowed_order_statuses();
		
		if ( $ps_order->is_subscription()
		     || $is_captured
		     || ! in_array( $order->get_status(), $allowed_order_statuses ) ) {
			return false;
		}
		
		$authorized_amount = $ps_order->get_order_amount_authorized();
		if ( empty( $authorized_amount ) ) {
			$authorized_amount = $order->get_total();
		}
		
		$amount_captured = $ps_order->get_order_amount_captured();
		$amount_allowed  = wc_format_decimal( $authorized_amount - $amount_captured );
		
		?>
		<tr class="wc-psigate-capture-amount-wrapper" style="display: none;">
			<td class="label capture-total">
				<?php echo wc_help_tip( __( 'You can capture no more than the initially authorized amount.', \WC_PsiGate::TEXT_DOMAIN ) ); ?>
				<?php _e( 'Capture', \WC_PsiGate::TEXT_DOMAIN ); ?>:
			</td>
			<?php // There are changes in the cells, so we needed to two versions ?>
			<?php echo Compatibility::is_wc_2_6() ? '<td width="1%"></td>' : ''; ?>
			<td class="total capture-total">
				<input type="text"
				       class="wc-psigate-capture-amount wc_input_price"
				       name="wc-psigate-capture-amount"
				       value="<?php echo esc_attr( $amount_allowed ); ?>"
				/>
			</td>
			<?php echo Compatibility::is_wc_2_6() ? '' : '<td width="1%"></td>'; ?>
		</tr>
		<tr class="wc-psigate-capture-allowed-amount-wrapper" style="display: none;">
			<td class="label capture-total">
				<small><?php echo esc_html( __( 'Total amount allowed to capture', \WC_PsiGate::TEXT_DOMAIN ) ); ?>:</small>
			</td>
			<?php // There are changes in the cells, so we needed to two versions ?>
			<?php echo Compatibility::is_wc_2_6() ? '<td width="1%"></td>' : ''; ?>
			<td class="total capture-total">
				<?php echo esc_html( $amount_allowed ); ?>
			</td>
			<?php echo Compatibility::is_wc_2_6() ? '' : '<td width="1%"></td>'; ?>
		</tr>
		<?php
	}
	
	/**
	 * Returns the allowed order statuses to perform capture of a transaction.
	 * We naturally assume that the status of an order should be a paid order status, not completed and not failed payment.
	 *
	 * @since 3.2.0
	 *
	 * @return mixed
	 */
	public static function get_capture_allowed_order_statuses() {
		return apply_filters( 'wc_psigate_capture_allowed_order_statuses', array(
			'processing',
			'on-hold',
		) );
	}
}