<?php

namespace WcPsigate\Api\Html_Api;

use WcPsigate\Compatibility;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Description
 *
 * @since  1.5.0
 * @author VanboDevelops | Ivan Andreev
 *
 *        Copyright: (c) 2019 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Html_Response extends Abstract_Commons {
	
	public function hooks() {
		add_action( 'woocommerce_api_wc_gateway_psigate',
			array(
				$this,
				'check_html_api_payment_response'
			)
		);
	}
	
	/**
	 * Processes the HTML API Payment response
	 *
	 * @since 1.5.0
	 */
	public function check_html_api_payment_response() {
		$this->process_response();
		
		die( 'Validation Failed' );
	}
	
	/**
	 * Process Payment Response form the HTML API
	 */
	public function process_response() {
		$posted = stripslashes_deep( $_GET );
		
		\WC_PsiGate::add_debug_log( 'Payment response received. Response is: ' . print_r( $posted, true ) );
		
		if ( $this->is_payment_response_valid() ) {
			
			$order_id     = (int) $this->get_order_id_from_response( \WC_PsiGate::get_field( 'OrderID', $_GET, '' ) );
			$order        = wc_get_order( $order_id );
			$result       = \WC_PsiGate::get_field( 'Approval', $_GET, '' );
			$redirect_url = $this->get_gateway()->get_return_url( $order );
			
			if ( 'completed' == $order->get_status() || 'processing' == $order->get_status() ) {
				$this->redirect_and_end_execution( $redirect_url, $order );
			}
			
			if ( 'Successful' == $result ) {
				$transaction_id = \WC_PsiGate::get_field( 'TransRefNumber', $_GET, '' );
				
				// Update order
				$order->add_order_note(
					sprintf(
						__(
							'Payment Completed.'
							. ' Transaction Reference Number: %s.', \WC_PsiGate::TEXT_DOMAIN
						),
						$transaction_id
					)
				);
				
				// Save the transaction order ID used. This is the final transaction ID
				update_post_meta( Compatibility::get_order_id( $order ), '_psigate_html_api_transaction_order_id', wc_clean( \WC_PsiGate::get_field( 'OrderID', $_GET, '' ) ) );
				
				// Debug log
				\WC_PsiGate::add_debug_log( 'Payment completed.' );
				
				Compatibility::empty_cart();
				
				$order->payment_complete( $transaction_id );
			} else {
				// Debug log
				\WC_PsiGate::add_debug_log( 'Payment failed.' );
				$error_message = \WC_PsiGate::get_field( 'ErrMsg', $_GET, '' );
				$error_code    = \WC_PsiGate::get_field( 'ReturnCode', $_GET, '' );
				
				// Update order
				$order->add_order_note(
					sprintf(
						__(
							'Payment Failed.
Error Message: %s
Error Code: %s.', \WC_PsiGate::TEXT_DOMAIN
						),
						$error_message, $error_code
					)
				);
				
				$order->update_status( 'failed' );
				
				// Add error to show the customer and the cancel URL
				wc_add_notice(
					sprintf( __(
						'Your Payment Failed.
						Error Message: %s
						Please try again or use another payment option.', \WC_PsiGate::TEXT_DOMAIN
					), $error_message ), 'error'
				);
				
				$redirect_url = apply_filters( 'wc_psigate_html_api_response_failed_payment_redirect_url', $order->get_checkout_payment_url(), $order );
			}
			
			$this->redirect_and_end_execution( $redirect_url, $order );
		}
	}
	
	/**
	 * Checks, if the payment response is valid
	 *
	 * @return bool
	 */
	public function is_payment_response_valid() {
		$order_id = (int) $this->get_order_id_from_response( \WC_PsiGate::get_field( 'OrderID', $_GET, '' ) );
		$order    = wc_get_order( $order_id );
		
		if ( false === $order ) {
			return false;
		}
		
		$compare = $this->get_html_api_security_hash( $order );
		$array   = array(
			\WC_PsiGate::get_field( 'PaymentType', $_GET, '' ),
			Compatibility::get_order_id( $order ),
			$this->get_gateway()->format_amount( \WC_PsiGate::get_field( 'Amount', $_GET, '' ) )
		);
		
		\WC_PsiGate::add_debug_log( 'Response Validation: ' . print_r( $array, true ) );
		
		$generated = $this->generate_security_hash_string( $array );
		
		return $compare == $generated;
	}
	
	/**
	 * Recreates the order ID sent to Psigate and extracts the real order ID from it
	 *
	 * @param $response_id
	 *
	 * @return mixed
	 */
	public function get_order_id_from_response( $response_id ) {
		$explode = explode( $this->get_gateway()->suffix_connector(), $response_id );
		
		// Take into account the SON and SONP plugin order numbering
		$order_id = $this->get_gateway()->get_sonp_order_id( $explode[0] );
		
		// The first element is the order ID
		return (int) $order_id;
	}
	
	public function redirect_and_end_execution( $url, $order ) {
		$url = apply_filters( 'wc_psigate_html_api_response_redirect_url', $url, $order );
		
		wp_safe_redirect( $url );
		exit;
	}
}