<?php
/**
 * Class PsiGate_API
 * Handle PsiGate API requests and responses
 *
 * @since   1.1
 * @version 1.5.0
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 *
 *        Copyright: (c) 2012 - 2019 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace WcPsigate\Api;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Request {
	
	/**
	 * Process API request and return an XML object of the response
	 *
	 * @since 1.1
	 *
	 * @param string $xml The XML string of the request
	 *
	 * @throws \Exception
	 * @return Response
	 */
	public function process_request( $xml, $url ) {
		
		$params = array(
			'method'      => 'POST',
			'sslverify'   => apply_filters( 'psigate_sslverify', true ),
			'timeout'     => 120,
			'redirection' => 0,
			'body'        => $xml,
		);
		
		$response = $this->send( $params, $url );
		
		return $this->parse_response( $response );
	}
	
	/**
	 * Send the POST request and return the response
	 *
	 * @since 1.1
	 *
	 * @param array $params
	 *
	 * @return array
	 * @throws \Exception
	 */
	private function send( $params, $url ) {
		
		// Send the request and get the response
		$response = wp_remote_post( $url, $params );
		
		// If Error return the code and message
		if ( is_wp_error( $response ) ) {
			throw new \Exception( sprintf( __( 'There was an error while processing your request. Error Code: %s. Error Message: %s.', \WC_PsiGate::TEXT_DOMAIN ), $response->get_error_code(), $response->get_error_message() ) );
		}
		
		return $response;
	}
	
	/**
	 * Load the response xml into <b>SimpleXMLElement</b> object.
	 *
	 * @param array $response
	 *
	 * @return Response
	 * @throws \Exception
	 */
	public function parse_response( $response ) {
		$xml = simplexml_load_string( wp_remote_retrieve_body( $response ) );
		
		if ( false === $xml ) {
			// Debug log
			\WC_PsiGate::add_debug_log( 'Response Body: ' . print_r( wp_remote_retrieve_body( $response ), true ) );
			
			throw new \Exception( __( 'There was an error while processing the payment response. Please try again or contact the administrator.', \WC_PsiGate::TEXT_DOMAIN ) );
		}
		
		return new Response( $xml );
	}
	
	/**
	 * Generate xml from and associative array
	 *
	 * @since 1.1
	 *
	 * @param array  $params
	 * @param string $start_tag Starting Tag
	 *
	 * @return string The formatted xml string
	 */
	function array_to_xml( array $params, $start_tag = 'Order' ) {
		
		// Initialize the simplexml object
		$xml = simplexml_load_string( "<?xml version='1.0' encoding='utf-8'?><{$start_tag} />" );
		
		$this->add_xml_elements( $params, $xml );
		
		return $xml->asXML();
	}
	
	/**
	 * Loop through array elements and create xml nodes from them
	 *
	 * @since 1.1
	 *
	 * @param array  $params
	 * @param object $xml SimpleXMLElement object
	 * @param string $parent
	 */
	function add_xml_elements( $params, &$xml, $parent = null ) {
		foreach ( $params as $key => $value ) {
			if ( is_array( $value ) ) {
				if ( is_numeric( $key ) ) {
					if ( null != $parent ) {
						$key_node = $xml->addChild( $parent );
					} else {
						$key_node = $xml;
					}
				} else {
					if ( is_numeric( key( $value ) ) ) {
						$parent   = $key;
						$key_node = $xml;
					} else {
						$key_node = $xml->addChild( $key );
					}
				}
				$this->add_xml_elements( $value, $key_node, $parent );
			} else {
				$xml->addChild( $key, htmlentities( $value ) );
			}
		}
	}
}