<?php

use WcPsigate\Compatibility;

/**
 * WC_Gateway_PsiGate_Addons class.
 *
 * WC_Gateway_PsiGate_Addons manages the subscription and pre-orders support for the PsiGate functionality.
 *
 * @since   1.1
 * @extends WC_Gateway_PsiGate
 * @package PsiGate Gateway
 * @author  VanboDevelops | Ivan Andreev
 */
class WC_Gateway_PsiGate_Addons extends WC_Gateway_PsiGate {
	
	function __construct() {
		
		// Init the parent
		parent::__construct();
		
		if ( 2 === WC_PsiGate::get_subscriptions_version() ) {
			// Scheduled payment
			add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, array(
				$this,
				'scheduled_subscription_payment_request'
			), 10, 2 );
			
			// Meta data renewal remove
			add_action( 'wcs_resubscribe_order_created', array( $this, 'remove_renewal_order_meta' ), 10 );
			
			// Update change payment method
			add_action( 'woocommerce_subscription_failing_payment_method_updated_' . $this->id, array(
				$this,
				'changed_failing_payment_method'
			), 10, 2 );
			
			// Display card used details
			add_filter( 'woocommerce_my_subscriptions_payment_method', array(
				$this,
				'maybe_render_subscription_payment_method'
			), 10, 2 );
			
			// Handle display of the Admin facing payment method change
			add_filter( 'woocommerce_subscription_payment_meta', array(
				$this,
				'add_subscription_payment_meta'
			), 10, 2 );
			// Handle validation of the Admin facing payment method change
			add_filter( 'woocommerce_subscription_validate_payment_meta', array(
				$this,
				'validate_subscription_payment_meta'
			), 10, 2 );
		}
		
		// Add support for Pre-Orders
		if ( WC_PsiGate::is_pre_orders_active() ) {
			add_action( 'wc_pre_orders_process_pre_order_completion_payment_' . $this->id, array(
				$this,
				'process_pre_order_release_payment'
			) );
		}
	}
	
	/**
	 * Process the payment
	 */
	public function process_payment( $order_id ) {
		$order         = wc_get_order( $order_id );
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		if ( ! $this->check_pass ) {
			$this->validate_payment_information();
		}
		
		if ( $psigate_order->is_pre_order_with_tokenization() ) {
			// Pre-Order that requires tokenization will need special handling
			return $this->process_pre_orders_payment( $order );
		} else if ( $psigate_order->contains_subscription() ) {
			// TODO: Add change payment method handling
			// Subscription Payment
			return $this->process_subscription_payment( $order );
		} else {
			// Normal Products Payment
			return parent::process_payment( $order_id );
		}
	}
	
	/**
	 * Processes a subscriptions payment
	 *
	 * @param \WC_Order $order
	 *
	 * @return array
	 */
	public function process_subscription_payment( \WC_Order $order ) {
		try {
			$psigate_order = new \WcPsigate\Psigate_Order( $order );
			
			$initial_payment    = $order->get_total();
			$transaction_number = '';
			
			// If there is an initial payment
			if ( 0 < $initial_payment ) {
				if ( $this->is_using_new_card() ) {
					$response           = $this->process_credit_card_payment( $order, 'subscription' );
					$transaction_number = (string) $response->get_value( 'TransRefNumber' );
					
					try {
						$this->create_am_account_from_payment( $order, $response );
					}
					catch ( Exception $e ) {
						$order->add_order_note( sprintf( __( 'The payment was processed, but the customer token creation failed. The subscription future payments may fail.<br>Error message: %s', WC_PsiGate::TEXT_DOMAIN ), $e->getMessage() ) );
					}
				} else {
					// Customer is using a saved card, so add the account_id and serial_no to the order
					$card = $this->get_saved_card( WC_PsiGate::get_post( 'psigate-used-cc' ) );
					
					$psigate_order->save_account_id( $card['account_id'] );
					$psigate_order->save_serial_number( $card['serial_no'] );
					
					$transaction_number = $this->process_acc_manager_payment( $order, 'subscription' );
				}
			}
			
			// If subscriptions was successfully paid and customer profile save to the order
			// Payment complete
			$order->payment_complete( $transaction_number );
			
			// Remove cart
			Compatibility::empty_cart();
			
			$account_id    = $psigate_order->get_account_id();
			$serial_number = $psigate_order->get_serial_number();
			$psigate_order->save_meta_data_to_subscription( $account_id, $serial_number );
			
			// Return thank you page redirect
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order )
			);
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Process subscription payment for each period
	 *
	 * @param float     $amount_to_charge float The amount to charge.
	 * @param \WC_Order $renewal_order    The WC_Order object of the order which the subscription was purchased in.
	 *
	 * @access public
	 * @return void
	 */
	public function scheduled_subscription_payment_request( $amount_to_charge, \WC_Order $renewal_order ) {
		try {
			WC_PsiGate::add_debug_log( 'Scheduled payment: ' . print_r( Compatibility::get_order_id( $renewal_order ), true ) );
			
			$transaction_number = $this->process_acc_manager_payment( $renewal_order, 'subscription', $amount_to_charge );
			
			// Complete the renewal order
			$renewal_order->payment_complete( $transaction_number );
		}
		catch ( Exception $e ) {
			
			$renewal_order->update_status( 'failed', $e->getMessage() );
			
			// Debug log
			WC_PsiGate::add_debug_log( $e->getMessage() );
		}
	}
	
	/**
	 * Don't transfer psigate customer/token meta when creating a parent renewal order.
	 *
	 * @access public
	 *
	 * @param \WC_Order $resubscribe_order
	 */
	public function remove_renewal_order_meta( $resubscribe_order ) {
		$psigate_subscription = new \WcPsigate\Psigate_Order( $resubscribe_order );
		
		$psigate_subscription->delete_account_id();
		$psigate_subscription->delete_serial_number();
	}
	
	/**
	 * Update the accountID and the Serial No on the original order, when a renewal order is placed, after it failed previously
	 *
	 * @param \WC_Order $subscription
	 * @param \WC_Order $renewal_order
	 */
	public function changed_failing_payment_method( \WC_Order $subscription, \WC_Order $renewal_order ) {
		$original_psigate_order = new \WcPsigate\Psigate_Order( $subscription );
		$renewal_psigate_order  = new \WcPsigate\Psigate_Order( $renewal_order );
		
		$original_psigate_order->save_account_id( $renewal_psigate_order->get_account_id() );
		$original_psigate_order->save_serial_number( $renewal_psigate_order->get_serial_number() );
	}
	
	/**
	 * Show payment method on My Subscriptions section of My Account page
	 *
	 * @param string   $payment_method_to_display
	 * @param WC_Order $subscription
	 *
	 * @return string
	 */
	public function maybe_render_subscription_payment_method( $payment_method_to_display, WC_Order $subscription ) {
		// If it is not PsiGate don't bother
		if ( $this->id !== Compatibility::get_prop($subscription, 'payment_method' )) {
			return $payment_method_to_display;
		}
		
		$psigate_order = new \WcPsigate\Psigate_Order( $subscription );
		$card          = $this->get_card_from_account_id( $psigate_order->get_account_id() );
		
		if ( is_array( $card ) ) {
			$payment_method_to_display = sprintf( __( 'Via %s ending in %s', WC_PsiGate::TEXT_DOMAIN ), ucfirst( $card['card_type'] ), $card['last4'] );
		}
		
		return $payment_method_to_display;
	}
	
	/**
	 * Add payment method change fields
	 *
	 * @since 1.3
	 *
	 * @param $payment_meta
	 * @param $subscription
	 *
	 * @return mixed
	 */
	public function add_subscription_payment_meta( $payment_meta, $subscription ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $subscription );
		
		$payment_meta[ $this->id ] = array(
			'post_meta' => array(
				'_psigate_account_id' => array(
					'value' => $psigate_order->get_account_id(),
					'label' => 'PsiGate Customer Account ID',
				),
				'_psigate_serial_no'  => array(
					'value' => $psigate_order->get_serial_number(),
					'label' => 'PsiGate Saved Card Serial Number',
				),
			),
		);
		
		return $payment_meta;
	}
	
	/**
	 * Validate Payment method change
	 *
	 * @since 1.3
	 *
	 * @param $payment_method_id
	 * @param $payment_meta
	 *
	 * @throws Exception
	 */
	public function validate_subscription_payment_meta( $payment_method_id, $payment_meta ) {
		if ( $this->id === $payment_method_id ) {
			if ( ! isset( $payment_meta['post_meta']['_psigate_account_id']['value'] )
			     || empty( $payment_meta['post_meta']['_psigate_account_id']['value'] )
			) {
				throw new Exception( 'A "_psigate_account_id"(PsiGate Customer Account ID) value is required.' );
			}
			
			if ( ! isset( $payment_meta['post_meta']['_psigate_serial_no']['value'] )
			     || empty( $payment_meta['post_meta']['_psigate_serial_no']['value'] )
			) {
				throw new Exception( 'A "_psigate_serial_no"(PsiGate Saved Card Serial Number) value is required.' );
			}
		}
	}
	
	/**
	 * Process Pre-Order payment request.
	 * Only Register the customer payment method, pre-order payment is delayed one.
	 *
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	private function process_pre_orders_payment( WC_Order $order ) {
		
		try {
			$psisate_order = new \WcPsigate\Psigate_Order( $order );
			
			if ( $this->is_using_new_card() ) {
				// Create and save new customer profile
				$this->create_am_customer_account( $order );
			} else {
				// Customer is using a saved card, so add the account_id and serial_no to the order
				$card = $this->get_saved_card( WC_PsiGate::get_post( 'psigate-used-cc' ) );
				
				$psisate_order->save_account_id( $card['account_id'] );
				$psisate_order->save_serial_number( $card['serial_no'] );
			}
			
			// Now that we have the info need for future payment, mark the order pre-ordered
			WC_Pre_Orders_Order::mark_order_as_pre_ordered( $order );
			
			// empty cart
			Compatibility::empty_cart();
			
			// redirect to thank you page
			return array(
				'result'   => 'success',
				'redirect' => $this->get_return_url( $order ),
			);
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Charge the payment on order release
	 *
	 * @param WC_Order $order
	 */
	public function process_pre_order_release_payment( WC_Order $order ) {
		try {
			$transaction_number = $this->process_acc_manager_payment( $order, 'pre-order' );
			
			// Payment complete
			$order->payment_complete( $transaction_number );
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		return $psigate_order->contains_subscription();
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		return $psigate_order->contains_pre_order();
	}
}