<?php
/*
 * Main PsiGate Payment class
 *
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 *
 *	Copyright: (c) 2012 - 2014 VanboDevelops
 *	License: GNU General Public License v3.0
 *	License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

use WcPsigate\Compatibility;
use WcPsigate\Psigate_Order;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WC_Gateway_PsiGate extends WC_Payment_Gateway {
	
	public $testmode;
	public $testmode_auth;
	public $StoreID;
	public $Passphrase;
	public $accepted_cards = array();
	public $authorize;
	public $liveurl;
	public $debug;
	public $save_customers;
	public $save_to_account_text;
	public $check_pass;
	public $testurl;
	public $acc_cid;
	public $acc_user_id;
	public $acc_password;
	public $live_manager_url;
	public $test_manager_url;
	public $html_test_url;
	public $interac_enable;
	public $html_store_key;
	public $html_live_url;
	public $html_page_language;
	public $html_description;
	public $html_pass_phrase;
	public $psi;
	public $card_options;
	public $cc_number;
	public $cc_cvc;
	public $cc_exp_year;
	public $cc_exp_month;
	public $what_payment_customer_used;
	public $send_xml_api_order_details;
	public $xml;
	public $html_api_request;
	public $html_api_response;
	
	public function __construct() {
		$this->id                 = 'psigate';
		$this->icon               = apply_filters( 'woocommerce_psigate_icon', plugin_dir_url( __FILE__ ) . '/assets/images/psi_gate_card_logos.png' );
		$this->has_fields         = true;
		$this->method_title       = __( 'PsiGate', WC_PsiGate::TEXT_DOMAIN );
		$this->method_description = __( 'PsiGate Gateway enables you to accept credit/debit card payments straight from your checkout page. It works by generating an order request and securely sending it to PsiGate for card authorization and processing. PsiGate gateway supports Subscriptions and Pre-Orders with the PsiGate Account Manager API.', WC_PsiGate::TEXT_DOMAIN );
		
		$this->card_options = apply_filters(
			'psigate_card_options', array(
				'visa'       => __( 'Visa', WC_PsiGate::TEXT_DOMAIN ),
				'mastercard' => __( 'Master Card', WC_PsiGate::TEXT_DOMAIN ),
				'amex'       => __( 'American Express', WC_PsiGate::TEXT_DOMAIN ),
				'discover'   => __( 'Discover', WC_PsiGate::TEXT_DOMAIN ),
				'jcb'        => __( 'JCB', WC_PsiGate::TEXT_DOMAIN )
			)
		);
		
		// Load the form fields.
		$this->init_form_fields();
		
		// Load the settings.
		$this->init_settings();
		
		// Define user set variables
		$this->title                      = $this->get_option( 'title', '' );
		$this->description                = $this->get_option( 'description', '' );
		$this->testmode                   = $this->get_option( 'testmode', 'yes' );
		$this->testmode_auth              = $this->get_option( 'testmode_auth', 'A' );
		$this->enabled                    = $this->get_option( 'enabled', 'no' );
		$this->StoreID                    = $this->get_option( 'StoreID' );
		$this->Passphrase                 = $this->get_option( 'PassPhrase' );
		$this->accepted_cards             = $this->get_option( 'accepted_cards', array() );
		$this->authorize                  = $this->get_option( 'authorize', '0' );
		$this->liveurl                    = $this->get_option( 'liveurl' );
		$this->debug                      = $this->get_option( 'debug', 'no' );
		$this->save_customers             = $this->get_option( 'save_customers', 'no' );
		$this->save_to_account_text       = $this->get_option( 'save_to_account_text', 'Save to account' );
		$this->send_xml_api_order_details = 'yes' == $this->get_option( 'send_xml_api_order_details', 'no' );
		$this->check_pass                 = false;
		$this->testurl                    = apply_filters( 'wc_psigate_xml_test_url', 'https://realtimestaging.psigate.com/xml' );
		
		// Addons settings
		$this->acc_cid          = $this->get_option( 'acc_cid' );
		$this->acc_user_id      = $this->get_option( 'acc_user_id' );
		$this->acc_password     = $this->get_option( 'acc_password' );
		$this->live_manager_url = $this->get_option( 'live_manager_url' );
		$this->test_manager_url = apply_filters( 'wc_psigate_manager_test_url', 'https://accountsstaging.psigate.com/xml' );
		
		// Interac settings
		$this->html_test_url      = apply_filters( 'wc_psigate_html_test_url', 'https://stagingcheckout.psigate.com/HTMLPost/HTMLMessenger' );
		$this->interac_enable     = $this->get_option( 'interac_enable' );
		$this->html_store_key     = $this->get_option( 'html_store_key' );
		$this->html_live_url      = $this->get_option( 'html_live_url' );
		$this->html_page_language = $this->get_option( 'html_page_language' );// EN_CA - Canadian English Default FR_CA - Canadian French
		$this->html_description   = $this->get_option( 'html_description' );
		$this->html_pass_phrase   = $this->get_option( 'html_pass_phrase' );
		
		// API class
		$this->psi = new \WcPsigate\Api\Request();
		
		// All gateway supported operations
		$this->supports = array(
			'subscriptions',
			'products',
			'subscription_cancellation',
			'subscription_reactivation',
			'subscription_suspension',
			'subscription_amount_changes',
			'subscription_payment_method_change', // Subs 1.n compatibility
			'subscription_payment_method_change_customer',
			'subscription_payment_method_change_admin',
			'subscription_date_changes',
			'multiple_subscriptions',
			'refunds',
			'pre-orders',
		);
		
		if ( $this->has_html_api_features() ) {
			add_action( 'woocommerce_receipt_psigate', array( $this, 'receipt_page' ) );
			
			// Load request class
			$this->html_api_request = new \WcPsigate\Api\Html_Api\Html_Request( $this );
			
			// Load response class
			$this->html_api_response = new \WcPsigate\Api\Html_Api\Html_Response( $this );
			$this->html_api_response->hooks();
		}
		
		// Save options
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array(
			$this,
			'process_admin_options'
		) );
	}
	
	/**
	 *
	 */
	public function admin_options() {
		wp_enqueue_script( 'psigate-admin' );
		parent::admin_options();
	}
	
	/**
	 * Initialise Gateway Settings Form Fields
	 **/
	function init_form_fields() {
		$this->form_fields = array(
			'enabled' => array(
				'title'   => __( 'Enable/Disable', WC_PsiGate::TEXT_DOMAIN ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable PsiGate', WC_PsiGate::TEXT_DOMAIN ),
				'default' => 'no'
			),
			
			'title' => array(
				'title'       => __( 'Method Title', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => __( 'Credit/Debit Card', WC_PsiGate::TEXT_DOMAIN )
			),
			
			'description' => array(
				'title'       => __( 'Description', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'textarea',
				'description' => __( 'This controls the description which the user sees during checkout. Will show above the credit card fields.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => __( "Please make sure you enter your correct billing information.", WC_PsiGate::TEXT_DOMAIN )
			),
			
			'general_settings_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
				'class'       => '',
			),
			
			'xml_api_settings' => array(
				'title'       => __( 'XML API Settings', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => '',
				'desc_tip'    => true,
				'class'       => '',
			),
			
			'StoreID' => array(
				'title'       => __( 'Store ID', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your StoreID received from PsiGate.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'PassPhrase' => array(
				'title'       => __( 'Pass Phrase', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'password',
				'description' => __( 'Enter your Passphrase received from PsiGate. The field will always appear empty.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'liveurl' => array(
				'title'       => __( 'Live URL', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your live URL, received from PsiGate. Production Mode Only.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'xml_api_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
				'class'       => '',
			),
			
			'transaction_general_settings' => array(
				'title'       => __( 'General Transaction Settings', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => '',
				'desc_tip'    => true,
				'class'       => '',
			),
			
			'accepted_cards' => array(
				'title'       => __( 'Accepted Cards', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'multiselect',
				'description' => __( "Choose the cards you can and want to accept payments from.", WC_PsiGate::TEXT_DOMAIN ),
				'options'     => $this->card_options,
				'class'       => 'chosen_select',
				'css'         => 'min-width: 350px;',
				'default'     => array( 'VISA', 'MC' ),
			),
			
			'authorize' => array(
				'title'       => __( 'Payment Action Type', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'select',
				'description' => __( "Choose how to handle the payment process.<br/> <b>'Sale'</b> will authorize and capture the payment right away. <br><b>'Authorize'</b> will only authorize the payment and you will have up to 5 days to manually capture the payment via your PsiGate admin panel.", WC_PsiGate::TEXT_DOMAIN ),
				'options'     => array( '0' => 'Sale', '1' => 'Authorize' ),
				'class'       => 'chosen_select',
				'css'         => 'min-width: 350px;',
				'default'     => ''
			),
			
			'send_xml_api_order_details' => array(
				'type'        => 'checkbox',
				'label'       => __( "Send itemized order details with the requests", WC_PsiGate::TEXT_DOMAIN ),
				'description' => __( "Itemized order details will be sent with the request.<br> <b>IMPORTANT:</b> If enabled, the transaction total will be calculated by PsiGate based on the itemized transaction details. There is a possibility that the order total and the calculated transaction total are not an exact match. Please test orders with tax, discount and any additional fees that your store can have, to make sure that each transaction total matches the order total.", WC_PsiGate::TEXT_DOMAIN ),
				'default'     => 'no'
			),
			
			'save_customers' => array(
				'type'        => 'checkbox',
				'class'       => 'psigate_save_customers',
				'title'       => __( 'Save Customers', WC_PsiGate::TEXT_DOMAIN ),
				'label'       => __( "Allow customers to pay with saved cards", WC_PsiGate::TEXT_DOMAIN ),
				'description' => __( "Enable if you want your customer to be able to pay with saved cards.<br/> Only the last 4 digits of the Card, expiration date and their generated customer ID will be saved in the Store database. Customer will be able to delete or edit their saved information from their 'My Account' Page.", WC_PsiGate::TEXT_DOMAIN ),
				'default'     => 'yes'
			),
			
			'save_to_account_text' => array(
				'type'        => 'text',
				'title'       => __( 'Save To Account checkbox text', WC_PsiGate::TEXT_DOMAIN ),
				'class'       => 'show_if_save_customers',
				'label'       => __( "Display save to account checkbox", WC_PsiGate::TEXT_DOMAIN ),
				'description' => __( "The text message the customer will see next to the 'Save to account' checkbox.", WC_PsiGate::TEXT_DOMAIN ),
				'default'     => 'Save to account'
			),
			
			'transaction_general_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
				'class'       => '',
			),
			
			'am_api_settings' => array(
				'desc_tip'    => true,
				'title'       => __( 'Account Manager API Settings', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => sprintf( __( 'Enter below your credentials to use with PsiGate Account Manager API. Leave empty, if you are not using the API.', WC_PsiGate::TEXT_DOMAIN ) ),
			),
			
			'acc_cid' => array(
				'title'       => __( 'Account Manager CID', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'The CID specifies your Merchant Account. Your CID will be given to you by PsiGate.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'acc_user_id' => array(
				'title'       => __( 'Account Manager User ID', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your User ID received from PsiGate.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'acc_password' => array(
				'title'       => __( 'Account Manager Password', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'password',
				'description' => __( 'Enter your Password received from PsiGate.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'live_manager_url' => array(
				'title'       => __( 'Live Account Manager URL', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your live URL used for the Account Manager API. You should receive this URL from PsiGate. Production Mode Only.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => ''
			),
			
			'am_api_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
				'class'       => '',
			),
			
			'html_api_settings' => array(
				'title'       => __( 'Real-time HTML API Settings', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => sprintf( __( 'Enter below your credentials to use with PsiGate Real-time HTML API. The API is required to accept interac online payments.', WC_PsiGate::TEXT_DOMAIN ) ),
			),
			
			'interac_enable' => array(
				'title'       => __( 'Interac Online Enabled?', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'select',
				'description' => __( "Enable the Interac Online and start accepting bank transfers.", WC_PsiGate::TEXT_DOMAIN ),
				'options'     => array( '0' => 'No', '1' => 'Yes' ),
				'class'       => 'chosen_select',
				'css'         => 'min-width: 350px;',
				'default'     => '0'
			),
			
			'html_store_key' => array(
				'title'       => __( 'Store Key', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your Store Key(MerchantID) received from PsiGate.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => 'psigatecapturescard001010',
				'class'       => 'show_if_interac_enable',
			),
			
			'html_pass_phrase' => array(
				'title'       => __( 'Security Password', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter a security password. The password is self chosen and it will give you an additional level of security.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => sha1( uniqid( 'psigate' ) ),
				'class'       => 'show_if_interac_enable',
			),
			
			'html_live_url' => array(
				'title'       => __( 'HTML API Live URL', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'text',
				'description' => __( 'Enter your live URL used for the Real-time HTML API. You should receive this URL from PsiGate. Production Mode Only.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => '',
				'class'       => 'show_if_interac_enable',
			),
			
			'html_page_language' => array(
				'title'       => __( 'Psigate HTML Page Language', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'select',
				'description' => __( "Enable the Interac Online and start accepting bank transfers.", WC_PsiGate::TEXT_DOMAIN ),
				'options'     => array( 'EN_CA' => 'English', 'FR_CA' => 'French' ),
				'class'       => 'chosen_select show_if_interac_enable',
				'css'         => 'min-width: 350px;',
				'default'     => 'EN_CA',
			),
			
			'html_description' => array(
				'title'       => __( 'Interac Online Description', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'textarea',
				'description' => __( 'Displayed to the customer, when Interac Online payment option is selected on checkout.', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => "",
				'class'       => 'show_if_interac_enable',
			),
			
			'html_api_end' => array(
				'title'       => '<hr/>',
				'type'        => 'title',
				'description' => '',
				'class'       => '',
			),
			
			'test_mode_settings' => array(
				'title'       => __( 'Test/Debug Settings', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'title',
				'description' => sprintf( __( 'Settings to enable test transactions and debug logging.', WC_PsiGate::TEXT_DOMAIN ) ),
			),
			
			'testmode' => array(
				'title'   => __( 'Test Mode', WC_PsiGate::TEXT_DOMAIN ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable PsiGate Test Mode', WC_PsiGate::TEXT_DOMAIN ),
				'default' => 'yes'
			),
			
			'testmode_auth' => array(
				'title'       => __( 'Test Mode Authorization Response', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'select',
				'description' => __( 'Choose what type of response you want to be returned when processing a Test Mode transaction.', WC_PsiGate::TEXT_DOMAIN ),
				'options'     => array(
					'A' => 'Approved',
					'D' => 'Declined',
					'R' => 'Random',
					'F' => 'Fraud'
				),
				'class'       => 'chosen_select show_if_testmode',
				'css'         => 'min-width: 350px;',
				'default'     => ''
			),
			
			'debug' => array(
				'title'       => __( 'Debug Log', WC_PsiGate::TEXT_DOMAIN ),
				'type'        => 'checkbox',
				'label'       => __( 'Recommended: Test Mode only', WC_PsiGate::TEXT_DOMAIN ),
				'default'     => 'no',
				'description' => sprintf( __( 'Debug log will provide you with most of the data and events generated by the payment process. Logged inside %s.' ), '<code>' . wc_get_log_file_path( 'PsiGate' ) . '</code>' ),
			),
		);
	} // End init_form_fields()
	
	/**
	 * Filter the gateway icon according to the accepted cards option
	 *
	 * @return string The Card images in a html format
	 */
	function get_icon() {
		$icon = '';
		if ( $this->accepted_cards ) {
			foreach ( $this->accepted_cards as $card ) {
				if ( file_exists( \WC_PsiGate::plugin_path() . '/assets/images/' . strtolower( $card ) . '.png' ) ) {
					$icon .= '<img src="' . esc_url( \WC_HTTPS::force_https_url( \WC_PsiGate::plugin_url() . '/assets/images/' . strtolower( $card ) . '.png' ) ) . '" alt="' . esc_attr( strtolower( $card ) ) . '" />';
				}
			}
		} else {
			$icon = '<img src="' . esc_url( \WC_HTTPS::force_https_url( $this->icon ) ) . '" alt="' . esc_attr( $this->title ) . '" />';
		}
		
		if ( $this->interac_enable ) {
			$icon .= '<img src="' . esc_url( \WC_HTTPS::force_https_url( \WC_PsiGate::plugin_url() . '/assets/images/interac-online.jpg' ) ) . '" alt="' . esc_attr( 'interac-online' ) . '" />';
		}
		
		return $icon;
	}
	
	/**
	 * There are no payment fields for psigate, but we want to show the description if set.
	 **/
	function payment_fields() {
		
		if ( $this->description ) {
			echo wpautop( wptexturize( $this->description ) );
		}
		
		$saved_cards = array();
		if ( is_user_logged_in() ) {
			$saved_cards = get_user_meta( get_current_user_id(), '_psigate_saved_cards', false );
		}
		
		// Load the Cards/Interac Onlince forms switch
		if ( $this->has_html_api_features() ) {
			if ( ! $this->cart_contains_subscription()
			     && ! $this->cart_contains_pre_order()
			) {
				wc_get_template(
					'checkout/psigate-switch-forms.php',
					array(
						'gateway' => $this,
					),
					'',
					WC_PsiGate::plugin_path() . '/templates/'
				);
				
				wc_get_template(
					'checkout/psigate-interac-online-instructions.php',
					array(
						'gateway' => $this,
					),
					'',
					WC_PsiGate::plugin_path() . '/templates/'
				);
			}
		} else {
			?>
			<input style="display: none !important;" type="radio" id="<?php echo esc_attr( $this->id ); ?>_cc_choice" checked="checked" value="cc" name="<?php echo esc_attr( $this->id ); ?>_payment_type_choice">
			<?php
		}
		
		$display_save_to_account = true;
		if ( 'yes' != $this->save_customers
		     || $this->cart_contains_subscription()
		     || $this->cart_contains_pre_order() ) {
			$display_save_to_account = false;
		}
		
		// Load the credit card form template
		wc_get_template(
			'checkout/psigate-payment-form.php',
			array(
				'save_customers'          => $this->save_customers,
				'saved_cards'             => $saved_cards,
				'gateway'                 => $this->id,
				'display_save_to_account' => $display_save_to_account,
				'save_to_account_text'    => $this->save_to_account_text,
			),
			'',
			WC_PsiGate::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Validate payment fields
	 **/
	function validate_fields() {
		
		// Validate card only, if we submitted one
		if ( 'cc' == WC_PsiGate::get_field( 'psigate_payment_type_choice', $_POST, 'cc' ) ) {
			$this->validate_payment_information();
		}
		
		if ( ! wc_notice_count( 'error' ) ) {
			$this->check_pass = true;
			
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Validate the payment info
	 */
	function validate_payment_information() {
		if ( $this->is_using_new_card() ) {
			// Set the credit card details
			$this->set_credit_card_details();
			
			// Check and validate the posted card fields
			$this->validate_card_fields(
				$this->cc_number,
				$this->cc_cvc,
				$this->cc_exp_year,
				$this->cc_exp_month
			);
			
			$this->what_payment_customer_used = 'new_cc';
		} else {
			$this->what_payment_customer_used = 'saved_cc';
		}
	}
	
	/**
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_using_new_card() {
		if ( 'cc' == WC_PsiGate::get_field( 'psigate_payment_type_choice', $_POST, 'cc' )
		     && 'new' == WC_PsiGate::get_field( 'psigate-used-cc', $_POST, 'new' ) ) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Process the payment and return the result
	 *
	 * @param int $order_id
	 *
	 * @return array
	 **/
	function process_payment( $order_id ) {
		
		if ( ! $this->check_pass ) {
			$this->validate_payment_information();
		}
		
		if ( ! wc_notice_count( 'error' ) ) {
			try {
				$order = wc_get_order( $order_id );
				
				// Process a card payment
				if ( 'cc' == WC_PsiGate::get_field( 'psigate_payment_type_choice', $_POST ) ) {
					return $this->run_process_card_payment( $order );
				} else {
					// Redirect the user to process an interac online payment
					return array(
						'result'   => 'success',
						'redirect' => $order->get_checkout_payment_url( true )
					);
				}
			}
			catch ( Exception $e ) {
				// Debug log
				WC_PsiGate::add_debug_log( 'Error: ' . $e->getMessage() );
				
				wc_add_notice( $e->getMessage(), 'error' );
			}
		}
	}
	
	/**
	 * Determines if we need to save the payment card to an AM API account
	 *
	 * @since 1.5.0
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function should_create_am_account( $order ) {
		if ( 'yes' == $this->save_customers
		     && $this->is_using_new_card()
		     && ( is_user_logged_in() && '' != $order->get_user_id() )
		     && '1' == WC_PsiGate::get_field( $this->id . '-save-card', $_POST, '0' )
		) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * @param WC_Order $order
	 *
	 * @return array
	 * @throws Exception
	 */
	public function run_process_card_payment( $order ) {
		if ( ! $this->is_using_new_card() ) {
			// Process an account payment on the saved card
			$transaction_number = $this->process_acc_manager_payment( $order );
		} else {
			
			$response           = $this->process_credit_card_payment( $order );
			$transaction_number = $response->get_value( 'TransRefNumber' );
			
			try {
				if ( $this->should_create_am_account( $order ) ) {
					$this->create_am_account_from_payment( $order, $response );
				}
			}
			catch ( Exception $e ) {
				// The payment is still successful, but we could not create a Token. Add an order note, so the merchant knows about it.
				$order->add_order_note( sprintf( __( 'The payment was processed, but the customer card was not saved for future use. Error message: %s', WC_PsiGate::TEXT_DOMAIN ), $e->getMessage() ) );
			}
		}
		
		// If we reached this stage, then the payment went through with positive result
		$order->payment_complete( $transaction_number );
		
		Compatibility::empty_cart();
		
		return array(
			'result'   => 'success',
			'redirect' => $this->get_return_url( $order )
		);
	}
	
	/**
	 * Process a payment against the XML API
	 *
	 * $payment_type = 'subscription' - Value is only used for first subscription payments
	 *
	 * @param WC_Order $order
	 * @param string   $payment_type The type of payment we are processing. Values: normal, subscription
	 *
	 * @return \WcPsigate\Api\Response
	 * @throws Exception
	 */
	public function process_credit_card_payment( WC_Order $order, $payment_type = 'normal' ) {
		// Set the credit card details in this object
		$this->set_credit_card_details();
		
		// Get the correct api url
		$url = $this->get_xml_api_url();
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Generating payment request for order #' . $this->get_order_number( $order ) );
		WC_PsiGate::add_debug_log( 'Payment type: ' . $payment_type );
		
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		$psi_order_id = $this->generate_gateway_order_id( $order );
		
		$parameters = array(
			'OrderID'       => $psi_order_id,
			'ShippingTotal' => number_format( Compatibility::get_total_shipping( $order ), 2, '.', '' ),
			'Tax1'          => number_format( $order->get_total_tax(), 2, '.', '' ),
			'Subtotal'      => number_format( ( $order->get_total() - Compatibility::get_total_shipping( $order ) - $order->get_total_tax() ), 2, '.', '' ),
			
			// Card info
			'PaymentType'   => 'CC',
			'CardIDCode'    => '1',
			'CardAction'    => $this->authorize,
			'StoreID'       => $this->StoreID,
			
			'Bname'       => Compatibility::get_order_billing_first_name( $order ) . ' ' . Compatibility::get_order_billing_last_name( $order ),
			'Bcompany'    => Compatibility::get_order_billing_company( $order ),
			'Baddress1'   => Compatibility::get_order_billing_address_1( $order ),
			'Baddress2'   => Compatibility::get_order_billing_address_2( $order ),
			'Bcity'       => Compatibility::get_order_billing_city( $order ),
			'Bprovince'   => Compatibility::get_order_billing_state( $order ),
			'Bpostalcode' => Compatibility::get_order_billing_postcode( $order ),
			'Bcountry'    => Compatibility::get_order_billing_country( $order ),
			
			'Sname'       => Compatibility::get_order_shipping_first_name( $order ) . ' ' . Compatibility::get_order_shipping_last_name( $order ),
			'Scompany'    => Compatibility::get_order_shipping_company( $order ),
			'Saddress1'   => Compatibility::get_order_shipping_address_1( $order ),
			'Saddress2'   => Compatibility::get_order_shipping_address_2( $order ),
			'Scity'       => Compatibility::get_order_shipping_city( $order ),
			'Sprovince'   => Compatibility::get_order_shipping_state( $order ),
			'Spostalcode' => Compatibility::get_order_shipping_postcode( $order ),
			'Scountry'    => Compatibility::get_order_shipping_country( $order ),
			'Phone'       => Compatibility::get_order_billing_phone( $order ),
			'Email'       => Compatibility::get_order_billing_email( $order ),
			
			'Comments'   => Compatibility::get_order_customer_note( $order ),
			'CustomerIP' => Compatibility::get_order_customer_ip( $order ),
		);
		
		if ( 'yes' == $this->testmode ) {
			$parameters['TestResult'] = $this->testmode_auth;
		}
		
		if ( $this->send_xml_api_order_details ) {
			$parameters += $this->get_request_items_xml_api( $this->get_request_items( $order, $order->get_total() ) );
		}
		
		// Add the recurring parameters. Required by Visa and Mastercard to comply with the Stored Credential Transaction Framework
		if ( 'subscription' == $payment_type ) {
			$parameters['Recurring'] = 'Y';
			$parameters['Iteration'] = '1';
		}
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Payment request: ' . print_r( $parameters, true ) );
		
		$parameters['Passphrase']   = $this->Passphrase;
		$parameters['CardNumber']   = $this->cc_number;
		$parameters['CardExpMonth'] = $this->cc_exp_month;
		$parameters['CardExpYear']  = $this->cc_exp_year;
		$parameters['CardIDNumber'] = $this->cc_cvc;
		
		$parameters = apply_filters( 'wc_psigate_process_credit_card_payment_parameters', $parameters, $order );
		
		// Build xml from the parameters array
		$xml = $this->psi->array_to_xml( $parameters, 'Order' );
		
		// Perform the request
		$result = $this->psi->process_request( $xml, $url );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Credit Card payment response parameters: ' . print_r( $result->get_response(), true ) );
		
		if ( 'APPROVED' == $result->get_value( 'Approved' ) ) { //Payment APPROVED
			//Update order
			$order->add_order_note(
				sprintf(
					__(
						'Credit Card Payment Completed.'
						. ' Status: %s.'
						. ' Transaction Reference Number: %s. ', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'Approved' ),
					$result->get_value( 'TransRefNumber' )
				)
			);
			
			$psigate_order->save_payment_order_id( $result->get_value( 'OrderID' ) );
			
			$total_charged = $result->get_value( 'FullTotal' );
			
			// If we are running authorizations only, we will need to mark the order captured or not and how much we authorized
			$psigate_order->save_is_payment_captured( ! $this->processing_authorizations_only() );
			$psigate_order->save_order_amount_authorized( $total_charged );
			$psigate_order->save_order_amount_captured( $this->processing_authorizations_only() ? 0 : $total_charged );
			
			return $result;
		} elseif ( 'DECLINED' == $result->get_value( 'Approved' ) ) { //Payment DECLINED
			//Add note for a failed transaction
			$order->add_order_note(
				sprintf(
					__(
						'Credit Card Payment Declined.'
						. ' Status: %s.'
						. ' Error Code: %s.'
						. ' Error Message: %s.'
						. ' Transaction Reference Number: %s. ', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'Approved' ), $result->get_value( 'ReturnCode' ),
					$result->get_value( 'ErrMsg' ), $result->get_value( 'TransRefNumber' )
				)
			);
			
			// Change status Failed
			$order->update_status( 'failed' );
			
			//Add customer visible error.
			throw new Exception( __( 'Your payment was declined. Please try again or choose an alternative method.', WC_PsiGate::TEXT_DOMAIN ) );
		} else {
			//Error occurred
			$order->add_order_note(
				sprintf(
					__(
						'Error occurred while processing credit card payment.'
						. ' Error Code: %s.'
						. ' Error Message: %s.', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'ReturnCode' ), $result->get_value( 'ErrMsg' )
				)
			);
			
			// Change status Failed
			$order->update_status( 'failed' );
			
			//Add customer visible error.
			throw new Exception(
				sprintf(
					__(
						'An error occurred while processing your payment transaction.'
						. ' Please try again. If problem persists contact the store admin with the error message.'
						. ' Error message: %s', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'ErrMsg' )
				)
			);
		}
	}
	
	/**
	 * @param WC_Order                $order
	 * @param \WcPsigate\Api\Response $payment_response
	 *
	 * @throws Exception
	 */
	public function create_am_account_from_payment( $order, $payment_response ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		// Set customer details
		$this->set_credit_card_details();
		
		// Get the correct api url
		$url = $this->get_account_manager_api_url();
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Creating an AM Account from payment: ' . Compatibility::get_order_id( $order ) );
		
		$parameters = array(
			'Action'    => 'AMA20',
			'Condition' => array(
				'StoreID' => $this->StoreID,
				'OrderID' => $payment_response->get_value( 'OrderID' )
			)
		);
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Create account from payment request: ' . print_r( $parameters, true ) );
		
		// Add the API credentials
		$this->add_acc_manager_credentials( $parameters );
		
		$parameters = apply_filters( 'wc_psigate_create_am_account_from_payment_parameters', $parameters, $order, $payment_response );
		
		$xml = $this->psi->array_to_xml( $parameters, 'Request' );
		
		// Perform the request
		$result = $this->psi->process_request( $xml, $url );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Create account from payment response: ' . print_r( $result, true ) );
		
		// Three codes, even though we should only need "RPA-0150", but "RPA-0150" is not documented in the guide,
		// so let's give it a bit of a doubt
		if ( 'RPA-0150' == $result->get_value( 'ReturnCode' )
		     || 'RPA-0000' == $result->get_value( 'ReturnCode' )
		     || 'RPA-0010' == $result->get_value( 'ReturnCode' )
		) {
			
			// Add an order note.
			$order->add_order_note( __( 'Payment token successfully created.', WC_PsiGate::TEXT_DOMAIN ) );
			
			// Profile was created, save the info to the customer user data
			// Check if the customer is logged in.
			if ( is_user_logged_in() && 'yes' == $this->save_customers ) {
				add_user_meta(
					$order->get_user_id(), '_psigate_saved_cards', array(
						'card_type'  => $this->format_card_type( $result->Account->CardInfo->get_value( 'CardType' ) ),
						'serial_no'  => $result->Account->CardInfo->get_value( 'SerialNo' ),
						'account_id' => $result->Account->CardInfo->get_value( 'AccountID' ),
						'last4'      => substr( $result->Account->CardInfo->get_value( 'CardNumber' ), - 4, 4 ),
						'exp_year'   => $result->Account->CardInfo->get_value( 'CardExpYear' ),
						'exp_month'  => $result->Account->CardInfo->get_value( 'CardExpMonth' ),
					)
				);
			}
			
			$psigate_order->save_account_id( $result->Account->CardInfo->get_value( 'AccountID' ) );
			$psigate_order->save_serial_number( $result->Account->CardInfo->get_value( 'SerialNo' ) );
		} else {
			// We will give the account error code and message as they seem to be more descriptive
			throw new Exception(
				sprintf(
					__(
						'Registration of your new payment account failed.'
						. ' Code: %s.'
						. ' Message: %s', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'ReturnCode' ),
					$result->get_value( 'ReturnMessage' )
				)
			);
		}
	}
	
	/**
	 * Process a payment against the Account Manager API
	 *
	 * @since 1.1
	 *
	 * @param \WC_Order $order
	 * @param string    $payment_type
	 * @param double    $amount_to_charge
	 *
	 * @return string
	 * @throws Exception
	 */
	public function process_acc_manager_payment( \WC_Order $order, $payment_type = 'normal', $amount_to_charge = null ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		// Get the card we want to use
		$account_id = $psigate_order->get_account_id();
		$serial_no  = $psigate_order->get_serial_number();
		
		// If the account id and serial number are not attached to the order check for POSTed card
		if ( '' == $account_id || '' == $serial_no ) {
			$used_card = $this->get_saved_card( WC_PsiGate::get_post( 'psigate-used-cc' ) );
			
			// Get the card we want to use
			$account_id = $used_card['account_id'];
			$serial_no  = $used_card['serial_no'];
		}
		
		// Get the correct api url
		$url = $this->get_account_manager_api_url();
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Processing account manager payment.' );
		
		$parameters = array(
			'Action' => 'RBC99',
			'Charge' => array(
				'StoreID'   => $this->StoreID,
				'AccountID' => $account_id,
				'SerialNo'  => $serial_no,
			),
		);
		
		// If the merchant choose to process authorizations only
		if ( $this->processing_authorizations_only() ) {
			$parameters['Charge']['ProcessType'] = 'P';
		}
		
		// Add the CVC to the charge when we have it
		if ( '' != WC_PsiGate::get_post( 'psigate-card-cvc' ) ) {
			$parameters['Charge']['CardIDNumber'] = WC_PsiGate::get_post( 'psigate-card-cvc' );
		}
		
		// Add the recurring flag if we are paying for a subscription
		if ( 'subscription' == $payment_type ) {
			$parameters['Charge']['Recurring'] = 'Y';
		}
		
		$parameters['Charge'] += $this->get_request_items_am_api( $this->get_request_subscription_items( $order, $amount_to_charge ) );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'AMP Request parameters: ' . print_r( $parameters, true ) );
		
		// Add the API credentials
		$this->add_acc_manager_credentials( $parameters );
		
		$parameters = apply_filters( 'wc_psigate_process_acc_manager_payment_parameters', $parameters, $order );
		
		$xml = $this->psi->array_to_xml( $parameters, 'Request' );
		
		// Perform the request
		$result = $this->psi->process_request( $xml, $url );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'AMP Charge Response: ' . print_r( $result, true ) );
		
		if ( 'PSI-0000' == $result->get_value( 'ReturnCode' ) && 'APPROVED' == $result->Result->get_value( 'Approved' ) ) {
			
			// Payment APPROVED
			
			//Update order
			$order->add_order_note(
				sprintf(
					__(
						'Account Credit Card Payment Completed.'
						. ' Status: %s.'
						. ' Payment OrderID: %s.'
						. ' Transaction Reference Number: %s. ', WC_PsiGate::TEXT_DOMAIN
					),
					$result->Result->get_value( 'Approved' ),
					$result->Result->get_value( 'OrderID' ),
					$result->Result->get_value( 'TransRefNumber' )
				)
			);
			
			$psigate_order->save_payment_order_id( $result->Result->get_value( 'OrderID' ) );
			
			$total_charged = $result->Result->get_value( 'FullTotal' );
			
			// Account Manager API does not do authorizations only, so we will capture the payment right away
			$psigate_order->save_is_payment_captured( ! $this->processing_authorizations_only() );
			$psigate_order->save_order_amount_authorized( $total_charged );
			$psigate_order->save_order_amount_captured( $this->processing_authorizations_only() ? 0 : $total_charged );
			
			return $result->Result->get_value( 'TransRefNumber' );
		} elseif ( 'DECLINED' == $result->Result->get_value( 'Approved' ) ) {
			
			// Payment DECLINED
			
			//Add note for a failed transaction
			$order->add_order_note(
				sprintf(
					__(
						'Account Credit Card Payment Declined.'
						. ' Status: %s.'
						. ' Error Code: %s.'
						. ' Error Message: %s.'
						. ' Transaction Reference Number: %s.', WC_PsiGate::TEXT_DOMAIN
					),
					$result->Result->get_value( 'Approved' ), $result->Result->get_value( 'ReturnCode' ),
					$result->Result->get_value( 'ErrMsg' ), $result->Result->get_value( 'TransRefNumber' )
				)
			);
			
			// Change status Failed
			$order->update_status( 'failed' );
			
			//Add customer visible error.
			throw new Exception( __( 'Your payment was declined. Please try again or choose an alternative method.', WC_PsiGate::TEXT_DOMAIN ) );
		} else {
			
			// Payment ERROR
			
			$order->add_order_note(
				sprintf(
					__(
						'Error occurred while processing credit card payment.'
						. ' Error Code: %s.'
						. ' Error Message: %s.', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'ReturnCode' ), $result->get_value( 'ReturnMessage' )
				)
			);
			
			// Change status Failed
			$order->update_status( 'failed' );
			
			//Add customer visible error.
			throw new Exception(
				sprintf(
					__(
						'An error occurred while processing your account payment transaction.<br/>'
						. ' Please try again. If problem persists contact the store admin with the error message.<br/>'
						. ' Error Code: %s'
						. ' Error Message: %s', WC_PsiGate::TEXT_DOMAIN
					), $result->get_value( 'ReturnCode' ), $result->get_value( 'ReturnMessage' )
				)
			);
		}
	}
	
	/**
	 * Register customer account with Account Manager API and save credit card to it
	 *
	 * @since 1.1
	 *
	 * @param WC_Order $order
	 *
	 * @throws Exception
	 */
	public function create_am_customer_account( WC_Order $order ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		// Set customer details
		$this->set_credit_card_details();
		
		// Get the correct api url
		$url = $this->get_account_manager_api_url();
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Registering payment account for customerID: ' . $order->get_user_id() );
		
		$parameters = array(
			'Action'  => 'AMA01',
			'Account' => array(
				'Name'       => Compatibility::get_order_billing_first_name( $order ) . ' ' . Compatibility::get_order_billing_last_name( $order ),
				'Company'    => Compatibility::get_order_billing_company( $order ),
				'Address1'   => Compatibility::get_order_billing_address_1( $order ),
				'Address2'   => Compatibility::get_order_billing_address_2( $order ),
				'Province'   => Compatibility::get_order_billing_state( $order ),
				'Postalcode' => Compatibility::get_order_billing_postcode( $order ),
				'City'       => Compatibility::get_order_billing_city( $order ),
				'Country'    => Compatibility::get_order_billing_country( $order ),
				'Phone'      => Compatibility::get_order_billing_phone( $order ),
				'Email'      => Compatibility::get_order_billing_email( $order ),
			),
		);
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Request parameters: ' . print_r( $parameters, true ) );
		
		$parameters['Account']['CardInfo']['CardHolder']   = Compatibility::get_order_billing_first_name( $order ) . ' ' . Compatibility::get_order_billing_last_name( $order );
		$parameters['Account']['CardInfo']['CardNumber']   = $this->cc_number;
		$parameters['Account']['CardInfo']['CardExpMonth'] = $this->cc_exp_month;
		$parameters['Account']['CardInfo']['CardExpYear']  = $this->cc_exp_year;
		
		// Add the API credentials
		$this->add_acc_manager_credentials( $parameters );
		
		$parameters = apply_filters( 'wc_psigate_register_customer_account_parameters', $parameters, $order );
		
		$xml = $this->psi->array_to_xml( $parameters, 'Request' );
		
		// Perform the request
		$result = $this->psi->process_request( $xml, $url );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Register new account response: ' . print_r( $result->get_response(), true ) );
		
		// Account is registered successfully
		if ( 'RPA-0000' == $result->get_value( 'ReturnCode' ) && 'RPA-0010' == $result->Account->get_value( 'ReturnCode' ) ) {
			
			// Add an order note.
			$order->add_order_note( __( 'Registered customer payment account.', WC_PsiGate::TEXT_DOMAIN ) );
			
			// Profile was created, save the info to the customer user data
			// Check if the customer is logged in.
			if ( is_user_logged_in() ) {
				add_user_meta(
					get_current_user_id(), '_psigate_saved_cards', array(
						'card_type'  => $this->format_card_type( $result->Account->CardInfo->get_value( 'CardType' ) ),
						'serial_no'  => $result->Account->CardInfo->get_value( 'SerialNo' ),
						'account_id' => $result->Account->CardInfo->get_value( 'AccountID' ),
						'last4'      => substr( $result->Account->CardInfo->get_value( 'CardNumber' ), - 4, 4 ),
						'exp_year'   => $result->Account->CardInfo->get_value( 'CardExpYear' ),
						'exp_month'  => $result->Account->CardInfo->get_value( 'CardExpMonth' ),
					)
				);
			}
			
			$psigate_order->save_account_id( $result->Account->CardInfo->get_value( 'AccountID' ) );
			$psigate_order->save_serial_number( $result->Account->CardInfo->get_value( 'SerialNo' ) );
		} else {
			// We will give the account error code and message as they seem to be more descriptive
			throw new Exception(
				sprintf(
					__(
						'Registering your new payment account failed.'
						. ' Code: %s.'
						. ' Message: %s', WC_PsiGate::TEXT_DOMAIN
					),
					$result->get_value( 'ReturnCode' ),
					$result->get_value( 'ReturnMessage' )
				)
			);
		}
	}
	
	/**
	 * Process automatic refunds
	 *
	 * @since 1.2
	 *
	 * @param int    $order_id
	 * @param null   $amount
	 * @param string $reason
	 *
	 * @return bool|WP_Error
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		
		try {
			$order = wc_get_order( $order_id );
			
			if ( ! $order ) {
				// Refund was not successful
				throw new Exception( __( "Invalid Order ID provided.", WC_PsiGate::TEXT_DOMAIN ) );
			}
			
			$psigate_order = new \WcPsigate\Psigate_Order( $order );
			
			// Add a retry count suffix to the orderID.
			$refund_order_id = $psigate_order->get_payment_order_id();
			
			if ( ! $refund_order_id ) {
				// Refund was not successful
				throw new Exception( __( "Can't process refund on the order. Missing required PsiGate Order ID.", WC_PsiGate::TEXT_DOMAIN ) );
			}
			
			$parameters = array(
				'OrderID'     => $refund_order_id,
				'Subtotal'    => wc_format_decimal( $amount, 2 ),
				
				// Card info
				'PaymentType' => 'CC',
				'CardAction'  => '3',
			);
			
			// Debug log
			WC_PsiGate::add_debug_log( 'Refund request: ' . print_r( $parameters, true ) );
			
			$parameters['StoreID']    = $this->StoreID;
			$parameters['Passphrase'] = $this->Passphrase;
			
			// Get the correct api url
			$url = $this->get_xml_api_url();
			
			$parameters = apply_filters( 'wc_psigate_process_refund_parameters', $parameters, $order );
			
			// Build xml from the parameters array
			$xml = $this->psi->array_to_xml( $parameters, 'Order' );
			
			// Perform the request
			$result = $this->psi->process_request( $xml, $url );
			
			// Debug log
			WC_PsiGate::add_debug_log( 'Refund response: ' . print_r( $result->get_response(), true ) );
			
			if ( 'APPROVED' == $result->get_value( 'Approved' ) ) { //Payment APPROVED
				// Debug log
				WC_PsiGate::add_debug_log( 'Refund completed.' );
				
				// Add order note
				$order->add_order_note(
					sprintf(
						__(
							'Refunded %s. Refund ID: %s. %s',
							WC_PsiGate::TEXT_DOMAIN
						),
						$amount,
						$result->get_value( 'TransRefNumber' ),
						( '' != $reason ) ? sprintf(
							__(
								'Credit Note: %s.'
							), $reason
						) : ''
					)
				);
				
				return true;
			} else {
				$error = $this->get_transaction_error_details( $result->get_value( 'ErrMsg' ) );
				
				$error_message = '';
				if ( is_array( $error ) ) {
					$error_message = sprintf( __( 'Error Code: %s Message: %s.', WC_PsiGate::TEXT_DOMAIN ), $error['code'], $error['message'] );
				} elseif ( $error ) {
					$error_message = sprintf( __( 'Error Message: %s.', WC_PsiGate::TEXT_DOMAIN ), $error );
				}
				
				$message = sprintf( __( 'Refund declined. %s', WC_PsiGate::TEXT_DOMAIN ), $error_message );
				
				// Debug log
				WC_PsiGate::add_debug_log( 'Refund failed.' );
				
				// Add order note
				$order->add_order_note( $message );
				
				throw new Exception( $message );
			}
		}
		catch ( Exception $ex ) {
			return new WP_Error( 'psigate-error', $ex->getMessage() );
		}
	}
	
	/**
	 * Adds capture payment action to the order actions
	 *
	 * @since 3.2.0
	 *
	 * @param $actions
	 *
	 * @return mixed
	 */
	public function add_order_capture_action( $actions ) {
		
		/**
		 * @var \WC_Order $theorder
		 */
		global $theorder;
		
		$method = Compatibility::get_prop( $theorder, 'payment_method' );
		if ( $this->id != $method ) {
			return $actions;
		}
		
		$ps_order = new Psigate_Order( $theorder );
		
		$is_captured            = $ps_order->get_is_payment_captured();
		$allowed_order_statuses = \WcPsigate\Admin\Capture::get_capture_allowed_order_statuses();
		
		if ( $ps_order->is_subscription()
		     || false != $is_captured
		     || ! in_array( $theorder->get_status(), $allowed_order_statuses ) ) {
			return $actions;
		}
		
		$authorized_amount = $ps_order->get_order_amount_authorized();
		// We need to know how much we authorized
		if ( empty( $authorized_amount ) ) {
			return $actions;
		}
		
		$amount_captured = $ps_order->get_order_amount_captured();
		$amount_allowed  = wc_format_decimal( $authorized_amount - $amount_captured );
		
		$actions['psigate_capture_payment'] = __( 'Capture Payment (' . get_woocommerce_currency_symbol() . $amount_allowed . ')', WC_PsiGate::TEXT_DOMAIN );
		
		return $actions;
	}
	
	/**
	 * Capture payment
	 *
	 * @since 3.2.0
	 *
	 * @param \WC_Order  $order
	 * @param float|null $amount
	 *
	 * @throws \Exception
	 *
	 * @return bool
	 */
	public function capture_payment( $order, $amount = null ) {
		$psigate_order = new Psigate_Order( $order );
		
		$is_captured = $psigate_order->get_is_payment_captured();
		
		// Bail, if we captured the amount already
		if ( false != $is_captured ) {
			return __( 'Order amount is already captured', WC_PsiGate::TEXT_DOMAIN );
		}
		
		try {
			// Get the original transaction ID
			$transaction_id = $psigate_order->get_payment_order_id();
			// Bail, if there is no reference ID
			if ( '' == $transaction_id ) {
				throw new \Exception( __( 'Missing Transaction ID. The order does not have all required information to process process a capture.', WC_PsiGate::TEXT_DOMAIN ) );
			}
			
			$authorized_amount = wc_format_decimal( $psigate_order->get_order_amount_authorized() );
			if ( empty( $authorized_amount ) ) {
				$authorized_amount = $order->get_total();
			}
			
			$amount_captured       = $psigate_order->get_order_amount_captured();
			$amount_captured_total = wc_format_decimal( $amount + $amount_captured );
			// Can't capture more than the initially authorized amount
			if ( $authorized_amount < $amount_captured_total ) {
				throw new \Exception( sprintf( __( "You can't capture more than the initially authorized amount: %s.", WC_PsiGate::TEXT_DOMAIN ), $authorized_amount ) );
			}
			
			if ( 0 == $amount ) {
				$amount = wc_format_decimal( $authorized_amount - $amount_captured );
			}
			
			$parameters = array(
				'OrderID'     => $transaction_id,
				'Subtotal'    => wc_format_decimal( $amount, 2 ),
				'CardAction'  => '2',
				'PaymentType' => 'CC',
				'StoreID'     => $this->StoreID,
			);
			
			// Debug log
			WC_PsiGate::add_debug_log( 'Capture request: ' . print_r( $parameters, true ) );
			
			$parameters['Passphrase'] = $this->Passphrase;
			
			// Get the correct api url
			$url = $this->get_xml_api_url();
			
			$parameters = apply_filters( 'wc_psigate_process_capture_parameters', $parameters, $order, $amount );
			
			// Build xml from the parameters array
			$xml = $this->psi->array_to_xml( $parameters, 'Order' );
			
			// Perform the request
			$result = $this->psi->process_request( $xml, $url );
			
			// Debug log
			WC_PsiGate::add_debug_log( 'Capture response: ' . print_r( $result->get_response(), true ) );
			
			if ( 'APPROVED' == $result->get_value( 'Approved' ) ) { //Payment APPROVED
				// Debug log
				WC_PsiGate::add_debug_log( 'Capture completed.' );
				
				// Add order note
				$order->add_order_note(
					sprintf(
						__(
							'Captured Completed for %s. Capture ID: %s.',
							WC_PsiGate::TEXT_DOMAIN
						),
						$amount,
						$result->get_value( 'TransRefNumber' )
					)
				);
				
				// If we are running authorizations only, we will need to mark the order captured or not and how much we authorized
				$psigate_order->save_is_payment_captured( true );
				$psigate_order->save_order_amount_captured( $amount );
			} else {
				$error = $this->get_transaction_error_details( $result->get_value( 'ErrMsg' ) );
				
				$error_message = '';
				if ( is_array( $error ) ) {
					$error_message = sprintf( __( 'Error Code: %s Message: %s', WC_PsiGate::TEXT_DOMAIN ), $error['code'], $error['message'] );
				} elseif ( $error ) {
					$error_message = sprintf( __( 'Error Message: %s', WC_PsiGate::TEXT_DOMAIN ), $error );
				}
				
				$message = sprintf( __( 'Capture declined. %s', WC_PsiGate::TEXT_DOMAIN ), $error_message );
				
				// Debug log
				WC_PsiGate::add_debug_log( 'Capture failed: ' . print_r( $message, true ) );
				
				throw new Exception( $message );
			}
			
			return true;
		}
		catch ( \Exception $e ) {
			$order->add_order_note( $e->getMessage() );
			
			return $e->getMessage();
		}
	}
	
	/**
	 * Delete customer saved card
	 *
	 * @since 1.1
	 *
	 * @param int $account_id
	 *
	 * @throws Exception
	 */
	function delete_customer_card( $account_id ) {
		
		// Get the card to delete
		$card_to_delete = $this->get_saved_card( $account_id );
		
		// Get the correct api url
		$url = $this->get_account_manager_api_url();
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Deleting credit card from customer: ' . get_current_user_id() );
		
		$parameters = array(
			'Action'    => 'AMA14',
			'Condition' => array(
				'AccountID' => $card_to_delete['account_id'],
				'SerialNo'  => $card_to_delete['serial_no'],
			),
		);
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Request parameters: ' . print_r( $parameters, true ) );
		
		// Add the API credentials
		$this->add_acc_manager_credentials( $parameters );
		
		// Build an xml from the array
		$xml = $this->psi->array_to_xml( $parameters, 'Request' );
		
		// Perform the request
		$result = $this->psi->process_request( $xml, $url );
		
		// Debug log
		WC_PsiGate::add_debug_log( 'Delete credit card response: ' . print_r( $result->get_response(), true ) );
		
		// Account is registered successfully
		if ( 'RPA-0058' == $result->get_value( 'ReturnCode' ) ) {
			delete_user_meta( get_current_user_id(), '_psigate_saved_cards', $card_to_delete );
		} else {
			throw new Exception(
				sprintf(
					__(
						'Card could not be deleted. Please try again or contact the site administrator.'
						. ' Error Code: %s.'
						. ' Error Message: %s'
					),
					$result->get_value( 'ReturnCode' ), $result->get_value( 'ReturnMessage' )
				)
			);
		}
	}
	
	/**
	 * @param WC_Order $order
	 *
	 * @return string
	 */
	public function generate_gateway_order_id( WC_Order $order ) {
		$psigate_order = new \WcPsigate\Psigate_Order( $order );
		
		return $psigate_order->get_order_number() . $psigate_order->suffix_connector() . $psigate_order->get_attempts_suffix();
	}
	
	/**
	 * Generate the Account Manager payment request order items for normal order
	 *
	 * @since 1.1
	 *
	 * @param WC_Order $order
	 * @param float    $amount_to_charge
	 *
	 * @return array
	 */
	protected function get_request_items( WC_Order $order, $amount_to_charge = null ) {
		if ( null === $amount_to_charge ) {
			$amount_to_charge = $order->get_total();
		}
		
		$parameters       = array();
		$i                = 0;
		$total_from_items = 0;
		
		$item_quantity_divider = apply_filters( 'wc_psigate_item_qty_divider', 'X' );
		
		if ( 0 < count( $order->get_items() ) ) {
			foreach ( $order->get_items() as $item ) {
				if ( Compatibility::get_item_quantity( $item ) ) {
					$product = Compatibility::get_product_from_item( $item, $order );
					
					$item_meta = Compatibility::wc_display_item_meta( $item );
					
					$item_name = Compatibility::get_item_quantity( $item ) . ' ' . $item_quantity_divider . ' ' . Compatibility::get_item_name( $item );
					if ( $item_meta ) {
						$item_name .= ' (' . $item_meta . ')';
					}
					
					$total_from_items += $order->get_line_subtotal( $item ) + $order->get_line_tax( $item );
					
					$parameters[ $i ] = $this->build_request_item(
						$this->html_entity_decode_numeric( $item_name ),
						( $product->get_sku() ) ? $product->get_sku() : $product->id,
						1,
						$this->format_amount( $order->get_line_subtotal( $item ) ),
						$this->format_amount( $order->get_line_tax( $item ) )
					);
				}
				$i ++;
			}
		}
		
		if ( 0 < Compatibility::get_total_shipping( $order ) ) {
			
			$total_from_items += Compatibility::get_total_shipping( $order ) + $order->get_shipping_tax();
			
			$parameters[ $i ] = $this->build_request_item(
				__( 'Shipping', WC_PsiGate::TEXT_DOMAIN ),
				'shipping',
				1,
				$this->format_amount( Compatibility::get_total_shipping( $order ) ),
				$this->format_amount( $order->get_shipping_tax() )
			);
			$i ++;
		}
		
		// Add the order discount
		if ( 0 < $order->get_total_discount() ) {
			$total_discount   = $order->get_total_discount();
			$total_from_items = $total_from_items - $total_discount;
			
			$parameters[ $i ] = $this->build_request_item(
				__( 'Discount', WC_PsiGate::TEXT_DOMAIN ),
				'discount',
				1,
				- $this->format_amount( $total_discount ),
				0
			);
			$i ++;
		}
		
		$fees = $order->get_items( 'fee' );
		if ( 0 < $fees ) {
			foreach ( $fees as $fee ) {
				$fee_name         = Compatibility::get_item_quantity( $fee ) . ' ' . $item_quantity_divider . ' ' . Compatibility::get_item_name( $fee );
				$total_from_items += $order->get_line_subtotal( $fee ) + $order->get_line_tax( $fee );
				
				$parameters[ $i ] = $this->build_request_item(
					$this->html_entity_decode_numeric( $fee_name ),
					sanitize_title( $fee_name ),
					1,
					$this->format_amount( $order->get_line_subtotal( $fee ) ),
					$this->format_amount( $order->get_line_tax( $fee ) )
				);
				$i ++;
			}
		}
		
		$total_from_items = $this->format_amount( $total_from_items );
		$amount_to_charge = $this->format_amount( $amount_to_charge );
		if ( $total_from_items != $amount_to_charge ) {
			WC_PsiGate::add_debug_log( 'get_request_items: total_from_items: ' . print_r( $total_from_items, true ) );
			WC_PsiGate::add_debug_log( 'get_request_items: amount_to_charge: ' . print_r( $amount_to_charge, true ) );
			
			$outstanding_fee_name = __( 'Outstanding line item amount', WC_PsiGate::TEXT_DOMAIN );
			
			$parameters[ $i ] = $this->build_request_item(
				$outstanding_fee_name,
				sanitize_title( $outstanding_fee_name ),
				1,
				$this->format_amount( $total_from_items - $amount_to_charge ),
				0
			);
		}
		
		return $parameters;
	}
	
	/**
	 * Formats the XML API items in the request
	 *
	 * @since 1.5.0
	 *
	 * @param $items_array
	 *
	 * @return array
	 */
	public function get_request_items_xml_api( $items_array ) {
		$parameters = array();
		foreach ( $items_array as $i => $item ) {
			if ( 'shipping' == $item['ProductID'] ) {
				continue;
			}
			
			$parameters['Item'][ $i ]['ItemDescription'] = $item['Description'];
			$parameters['Item'][ $i ]['ItemID']          = $item['ProductID'];
			$parameters['Item'][ $i ]['ItemQty']         = $item['Quantity'];
			$parameters['Item'][ $i ]['ItemPrice']       = $this->format_amount( $item['Price'] );
		}
		
		return $parameters;
	}
	
	/**
	 * Formats the AM API items in the request
	 *
	 * @since 1.5.0
	 *
	 * @param $items_array
	 *
	 * @return array
	 */
	public function get_request_items_am_api( $items_array ) {
		$parameters = array();
		foreach ( $items_array as $i => $item ) {
			$parameters['ItemInfo'][ $i ]['Description'] = $item['Description'];
			$parameters['ItemInfo'][ $i ]['ProductID']   = $item['ProductID'];
			$parameters['ItemInfo'][ $i ]['Quantity']    = $item['Quantity'];
			$parameters['ItemInfo'][ $i ]['Price']       = $item['Price'];
			$parameters['ItemInfo'][ $i ]['Tax1']        = $item['Tax'];
		}
		
		return $parameters;
	}
	
	/**
	 * Builds a single item details array
	 * Here so unify the item format
	 *
	 * @since 1.5.0
	 *
	 * @param      $description
	 * @param      $item_id
	 * @param      $qty
	 * @param      $price
	 * @param null $tax
	 *
	 * @return array
	 */
	public function build_request_item( $description, $item_id, $qty, $price, $tax = null ) {
		return array(
			'Description' => $description,
			'ProductID'   => $item_id,
			'Quantity'    => $qty,
			'Price'       => $price,
			'Tax'         => $tax,
		);
	}
	
	/**
	 * Generate the Account Manager payment request order items for Subscription payment
	 *
	 * @since 1.1
	 *
	 * @param WC_Order $order
	 * @param float    $amount_to_charge Amount to charge. It is used for Subscriptions renewal payments.<br />
	 *                                   We can't really itemize the renewal order since it can contain payments for missed periods, so we will charge one item with the total amount.<br />
	 *                                   It should not be a problem for now since there can be only one item per Subs order.
	 *
	 * @return array
	 */
	public function get_request_subscription_items( WC_Order $order, $amount_to_charge = null ) {
		return $this->get_request_items( $order, $amount_to_charge );
	}
	
	/**
	 * Set the credit card details in object variables
	 *
	 * TODO: Refactor to return a Card object instead of setting global props
	 *
	 * @since 1.1
	 */
	private function set_credit_card_details() {
		$cc_exp_date        = $this->format_exp_date( WC_PsiGate::get_post( 'psigate-card-expiry' ) );
		$this->cc_exp_month = $cc_exp_date['month'];
		
		$year              = 2 < strlen( $cc_exp_date['year'] ) ? substr( $cc_exp_date['year'], - 2 ) : $cc_exp_date['year'];
		$this->cc_exp_year = $year;
		
		$this->cc_number = $this->remove_white_space( WC_PsiGate::get_post( 'psigate-card-number' ) );
		$this->cc_cvc    = $this->remove_white_space( WC_PsiGate::get_post( 'psigate-card-cvc' ) );
	}
	
	/**
	 * Add Account Manager API credentials to the request parameters
	 *
	 * @since 1.1
	 *
	 * @param array $parameters
	 */
	private function add_acc_manager_credentials( array &$parameters ) {
		$parameters['CID']      = $this->acc_cid;
		$parameters['UserID']   = $this->acc_user_id;
		$parameters['Password'] = $this->acc_password;
	}
	
	/**
	 * Get a saved card from the customer cards
	 *
	 * @since 1.1
	 *
	 * @param int $card_id The array key of the card
	 *
	 * @return array The information array of the card
	 */
	public function get_saved_card( $card_id ) {
		$saved_cards = get_user_meta( get_current_user_id(), '_psigate_saved_cards', false );
		$saved_card  = $saved_cards[ $card_id ];
		
		return $saved_card;
	}
	
	/**
	 * Loop through the user saved cards and find the one matching the account_id
	 *
	 * @since 1.1
	 *
	 * @param int $account_id
	 *
	 * @return array|boolean
	 */
	public function get_card_from_account_id( $account_id ) {
		$saved_cards = get_user_meta( get_current_user_id(), '_psigate_saved_cards', false );
		
		foreach ( $saved_cards as $key => $value ) {
			if ( $value['account_id'] == $account_id ) {
				return $saved_cards[ $key ];
			}
		}
		
		return false;
	}
	
	/**
	 * Return the order number with stripped # or n° ( french translations )
	 *
	 * @param WC_Order $order
	 *
	 * @return string
	 */
	public function get_order_number( WC_Order $order ) {
		return str_replace( array( '#', 'n°' ), '', $order->get_order_number() );
	}
	
	/**
	 * Format the card type to match the card image names
	 *
	 * @since 1.1
	 *
	 * @param string $card_type
	 *
	 * @return string
	 */
	private function format_card_type( $card_type ) {
		if ( 'MC' == $card_type ) {
			$card_type = 'mastercard';
		} else {
			$card_type = strtolower( $card_type );
		}
		
		return $card_type;
	}
	
	/**
	 * @param int $order_id
	 *
	 */
	function receipt_page( $order_id ) {
		
		echo '<p>' . __( 'Thank you for your order.', WC_PsiGate::TEXT_DOMAIN ) . '</p>';
		
		try {
			echo $this->html_api_request->generate_payment_form( $order_id );
		}
		catch ( Exception $ex ) {
			echo $ex->getMessage();
		}
	}
	
	/**
	 * Check if this gateway is enabled and available in the user's country
	 */
	function is_available() {
		if ( 'yes' == $this->enabled ) {
			if ( ! $this->StoreID ) {
				return false;
			}
			if ( ! $this->Passphrase ) {
				return false;
			}
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * Check and validate the card fields
	 *
	 * @param mixed $cc_number    Credit card number
	 * @param mixed $cc_cvc       Card Security Code
	 * @param mixed $cc_exp_month Expiration month
	 * @param mixed $cc_exp_year  Expiration year
	 *
	 * @return void
	 **/
	private function validate_card_fields( $cc_number = '', $cc_cvc = '', $cc_exp_year = '', $cc_exp_month = '' ) {
		//Check credit card number
		if ( empty( $cc_number ) ) {
			wc_add_notice( __( 'Credit Card Number Required.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
		} else {
			if ( ! is_numeric( $cc_number ) || ( strlen( $cc_number ) > 19 || strlen( $cc_number ) < 12 ) ) {
				wc_add_notice( __( 'Invalid Credit Card Number length.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
			}
		}
		
		//Check credit card CV2 number
		if ( empty( $cc_cvc ) ) {
			wc_add_notice( __( 'CVC number required.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
		} else {
			//Check CVN
			if ( ! is_numeric( $cc_cvc ) ) {
				wc_add_notice( __( 'Invalid Card Verification Code. Only numbers are allowed.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
			}
			
			if ( strlen( $cc_cvc ) > 4 || strlen( $cc_cvc ) < 3 ) {
				wc_add_notice( __( 'Invalid Card Verification Code length. The CVC has to be 3 or 4 digits long. ', WC_PsiGate::TEXT_DOMAIN ), 'error' );
			}
		}
		
		if ( empty( $cc_exp_month ) ) {
			wc_add_notice( __( 'Expiration month required.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
		}
		
		if ( empty( $cc_exp_year ) ) {
			wc_add_notice( __( 'Expiration year required.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
		}
		
		//Check exp date
		$current_year  = date( 'y' );
		$current_month = date( 'n' );
		if ( ! is_numeric( $cc_exp_year ) ||
		     $cc_exp_year < $current_year ||
		     $cc_exp_year > ( $current_year + 10 ) ||
		     ! is_numeric( $cc_exp_month ) ||
		     $cc_exp_month < 0 ||
		     $cc_exp_month > 12 ||
		     ( $cc_exp_year == $current_year && $cc_exp_month < $current_month )
		) {
			wc_add_notice( __( 'Invalid expiration date.', WC_PsiGate::TEXT_DOMAIN ), 'error' );
		}
	}
	
	/**
	 * Format and return the expiration month and year
	 *
	 * @param string $expiration_date
	 *
	 * @return array
	 */
	public function format_exp_date( $expiration_date ) {
		$cc_exp_date = str_replace( ' ', '', explode( '/', $expiration_date ) );
		
		return array( 'month' => $cc_exp_date[0], 'year' => $cc_exp_date[1] );
	}
	
	/**
	 * Remove white space from a string
	 *
	 * @param $string $string
	 *
	 * @return string
	 */
	public function remove_white_space( $string ) {
		return str_replace( ' ', '', $string );
	}
	
	/**
	 * Return the correct XML API URL
	 *
	 * @return string
	 */
	public function get_xml_api_url() {
		if ( 'yes' == $this->testmode ) {
			return $this->testurl;
		} else {
			return $this->liveurl;
		}
	}
	
	/**
	 * Return the correct Account Manager API URL
	 *
	 * @return string
	 */
	public function get_account_manager_api_url() {
		if ( 'yes' == $this->testmode ) {
			return $this->test_manager_url;
		} else {
			return $this->live_manager_url;
		}
	}
	
	/**
	 * Decodes all HTML entities, including numeric and hexadecimal ones.
	 *
	 * @param mixed $string
	 * @param mixed $quote_style
	 * @param mixed $charset
	 *
	 * @return string decoded HTML
	 */
	function html_entity_decode_numeric( $string, $quote_style = ENT_COMPAT, $charset = 'utf-8' ) {
		$string = html_entity_decode( $string, $quote_style, $charset );
		$string = preg_replace_callback( '~&#x([0-9a-fA-F]+);~i', array(
			$this,
			'chr_utf8_callback'
		), $string );
		$string = preg_replace_callback( '~&#([0-9]+);~', array( $this, 'chr_utf8' ), $string );
		$string = str_ireplace( '&apos;', "'", $string );
		
		return $string;
	}
	
	/**
	 * Callback helper
	 *
	 * @param $matches
	 *
	 * @return string
	 */
	function chr_utf8_callback( $matches ) {
		return $this->chr_utf8( hexdec( $matches[1] ) );
	}
	
	/**
	 * Multi-byte chr(): Will turn a numeric argument into a UTF-8 string.
	 *
	 * @param mixed $num
	 *
	 * @return string
	 */
	function chr_utf8( $num ) {
		if ( $num < 128 ) {
			return chr( $num );
		}
		if ( $num < 2048 ) {
			return chr( ( $num >> 6 ) + 192 ) . chr( ( $num & 63 ) + 128 );
		}
		if ( $num < 65536 ) {
			return chr( ( $num >> 12 ) + 224 ) . chr( ( ( $num >> 6 ) & 63 ) + 128 ) . chr( ( $num & 63 ) + 128 );
		}
		if ( $num < 2097152 ) {
			return chr( ( $num >> 18 ) + 240 ) . chr( ( ( $num >> 12 ) & 63 ) + 128 ) . chr( ( ( $num >> 6 ) & 63 ) + 128 ) . chr( ( $num & 63 ) + 128 );
		}
		
		return '';
	}
	
	/**
	 * Extract error message and code from the PsiGate error.
	 * Format is errcode:errmessage.
	 *
	 * @since 1.2
	 *
	 * @param $err_msg
	 *
	 * @return array
	 */
	public function get_transaction_error_details( $err_msg ) {
		if ( '' == $err_msg || false === strpos( $err_msg, ':' ) ) {
			return $err_msg;
		}
		
		$split = explode( ':', $err_msg );
		
		return array( 'code' => $split[0], 'message' => $split[1] );
	}
	
	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_subscription( WC_Order $order ) {
		return false;
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.3
	 *
	 * @param WC_Order $order
	 *
	 * @return bool
	 */
	public function order_contains_pre_order( WC_Order $order ) {
		return false;
	}
	
	/**
	 * Formats and returns a the passed string
	 *
	 * @since 1.3
	 *
	 * @param string $string String to be formatted
	 * @param int    $limit  Limit characters of the string
	 *
	 * @return string
	 */
	public function format_string( $string, $limit ) {
		if ( strlen( $string ) > $limit ) {
			$string = substr( $string, 0, ( $limit - 3 ) ) . '...';
		}
		
		return $string;
	}
	
	/**
	 * Formats an amount the way PsiGate expects it
	 *
	 * @param float $amount
	 *
	 * @return float
	 */
	public function format_amount( $amount ) {
		return number_format( $amount, 2, '.', '' );
	}
	
	/**
	 * Checks, if the HTML API Features are enabled
	 *
	 * @return string
	 */
	public function has_html_api_features() {
		return $this->interac_enable;
	}
	
	public function suffix_connector() {
		return apply_filters( 'psigate_order_suffix_connector', '-' );
	}
	
	public function cart_contains_subscription() {
		if ( ! class_exists( 'WC_Subscriptions_Cart' ) ) {
			return false;
		}
		
		return \WC_Subscriptions_Cart::cart_contains_subscription();
	}
	
	public function cart_contains_pre_order() {
		if ( ! class_exists( 'WC_Pre_Orders_Cart' ) ) {
			return false;
		}
		
		return \WC_Pre_Orders_Cart::cart_contains_pre_order();
	}
	
	/**
	 * Get the order ID. Check to see if SON and SONP is enabled and
	 *
	 * @global WC_Seq_Order_Number     $wc_seq_order_number
	 * @global WC_Seq_Order_Number_Pro $wc_seq_order_number_pro
	 *
	 * @param string                   $order_number
	 *
	 * @return int|bool
	 */
	public function get_sonp_order_id( $order_number ) {
		
		@ob_start();
		
		// Find the order ID from the custom order number, if we have SON or SONP enabled
		if ( class_exists( 'WC_Seq_Order_Number' ) ) {
			if ( function_exists( 'wc_sequential_order_numbers' ) ) {
				$order_id = wc_sequential_order_numbers()->find_order_by_order_number( $order_number );
			} else {
				global $wc_seq_order_number;
				$order_id = $wc_seq_order_number->find_order_by_order_number( $order_number );
			}
			
			if ( 0 === $order_id ) {
				$order_id = $order_number;
			}
		} elseif ( class_exists( 'WC_Seq_Order_Number_Pro' ) ) {
			
			if ( function_exists( 'wc_seq_order_number_pro' ) ) {
				$order_id = wc_seq_order_number_pro()->find_order_by_order_number( $order_number );
			} else {
				global $wc_seq_order_number_pro;
				$order_id = $wc_seq_order_number_pro->find_order_by_order_number( $order_number );
			}
			
			if ( 0 === $order_id ) {
				$order_id = $order_number;
			}
		} else {
			$order_id = $order_number;
		}
		
		// Remove any error notices generated during the process
		@ob_clean();
		
		return $order_id;
	}
	
	public function processing_authorizations_only() {
		return '1' == $this->authorize;
	}
} //end PsiGate class