<?php

namespace WcPsigate;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Order wrapper for gateway order data
 *
 * @since  1.5.0
 * @author VanboDevelops | Ivan Andreev
 *
 *        Copyright: (c) 2018-2019 VanboDevelops
 *        License: GNU General Public License v3.0
 *        License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
class Psigate_Order {
	
	/**
	 * @var \WC_Order
	 */
	public $order;
	
	/**
	 * @param $order
	 */
	public function __construct( \WC_Order $order ) {
		$this->order = $order;
	}
	
	public function save_meta_data_to_subscription( $account_id, $serial_number ) {
		// Also store it on the subscriptions being purchased or paid for in the order
		if ( wcs_order_contains_subscription( $this->order ) ) {
			$subscriptions = wcs_get_subscriptions_for_order( $this->order );
		} elseif ( wcs_order_contains_renewal( $this->order ) ) {
			$subscriptions = wcs_get_subscriptions_for_renewal_order( $this->order );
		} else {
			$subscriptions = array();
		}
		
		foreach ( $subscriptions as $subscription ) {
			// Debug log
			\WC_PsiGate::add_debug_log( 'Saving details to subscription: ' . print_r( $subscription->get_id(), true ) );
			
			$psigate_subscription = new Psigate_Order( $subscription );
			
			$psigate_subscription->save_account_id( $account_id );
			$psigate_subscription->save_serial_number( $serial_number );
		}
	}
	
	/**---------------------------------
	 * GETTERS
	 * -----------------------------------*/
	
	/**
	 * Returns the order id
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	public function get_payment_order_id() {
		return Compatibility::get_meta( $this->order, '_psigate_order_id', true );
	}
	
	/**
	 * Returns the saved Card Account ID to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	public function get_account_id() {
		return Compatibility::get_meta( $this->order, '_psigate_account_id', true );
	}
	
	/**
	 * Returns the saved Card Serial Number to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	public function get_serial_number() {
		return Compatibility::get_meta( $this->order, '_psigate_serial_no', true );
	}
	
	/**
	 * Returns the payment captured data
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_is_payment_captured() {
		$value = Compatibility::get_meta( $this->order, '_psigate_is_payment_captured', true );
		
		if ( '' === $value ) {
			$value = false;
		}
		
		return $value;
	}
	
	/**
	 * Returns the amount captured
	 *
	 * @since 1.5.0
	 *
	 * @return float
	 */
	public function get_order_amount_captured() {
		return Compatibility::get_meta( $this->order, '_psigate_order_amount_captured', true );
	}
	
	/**
	 * Returns the amount authorized in the transaction
	 *
	 * @since 1.5.0
	 *
	 * @return float
	 */
	public function get_order_amount_authorized() {
		return Compatibility::get_meta( $this->order, '_psigate_order_amount_authorized', true );
	}
	
	public function get_order_payment_attempts() {
		return Compatibility::get_meta( $this->order, '_psigate_order_payment_attempts', true );
	}
	
	/**
	 * Iterates and saves an order request suffix. Used to remove duplication of requests
	 * @return int|string
	 */
	public function get_attempts_suffix() {
		$attempts        = $this->get_order_payment_attempts();
		$attempts_suffix = 0;
		
		if ( is_numeric( $attempts ) ) {
			$attempts_suffix = $attempts;
			
			$attempts_suffix ++;
		}
		
		// Save the incremented attempts
		$this->save_order_payment_attempts( $attempts_suffix );
		
		return $attempts_suffix;
	}
	
	public function suffix_connector() {
		return apply_filters( 'psigate_order_suffix_connector', '-' );
	}
	
	/**
	 * Return the order number with stripped # or n° ( french translations )
	 *
	 * @return string
	 */
	public function get_order_number() {
		return str_replace( array( '#', 'n°' ), '', $this->order->get_order_number() );
	}
	
	/**---------------------------------------------------
	 * CREATE
	 * ---------------------------------------------------*/
	
	/**
	 * Saves order id to the order
	 *
	 * This is the original transaction_id/settlement_id for the payment
	 *
	 * @since 1.5.0
	 *
	 * @param string $value
	 */
	public function save_payment_order_id( $value ) {
		Compatibility::update_meta( $this->order, '_psigate_order_id', wc_clean( $value ) );
	}
	
	/**
	 * Saves the Card Account ID to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @param string $account_id
	 *
	 * @return bool|int
	 */
	public function save_account_id( $account_id ) {
		return Compatibility::update_meta( $this->order, '_psigate_account_id', wc_clean( $account_id ) );
	}
	
	/**
	 * Saves the Card Serial Number to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @param string $serial_number
	 *
	 * @return bool|int
	 */
	public function save_serial_number( $serial_number ) {
		return Compatibility::update_meta( $this->order, '_psigate_serial_no', wc_clean( $serial_number ) );
	}
	
	/**
	 * Marks the order the amount captured
	 *
	 * @param bool $is_captured
	 *
	 * @since 1.5.0
	 */
	public function save_is_payment_captured( $is_captured = false ) {
		Compatibility::update_meta( $this->order, '_psigate_is_payment_captured', wc_clean( $is_captured ) );
	}
	
	/**
	 * Marks the order payment as captured or not
	 *
	 * @since 1.5.0
	 *
	 * @param bool $amount (optional) If not present the order total will be saved
	 */
	public function save_order_amount_captured( $amount = false ) {
		if ( false === $amount ) {
			$amount = $this->order->get_total();
		}
		
		Compatibility::update_meta( $this->order, '_psigate_order_amount_captured', wc_clean( $amount ) );
	}
	
	/**
	 * Saves the amount authorized in the transaction
	 *
	 * @since 1.5.0
	 *
	 * @param bool $amount
	 */
	public function save_order_amount_authorized( $amount = false ) {
		if ( false === $amount ) {
			$amount = $this->order->get_total();
		}
		
		Compatibility::update_meta( $this->order, '_psigate_order_amount_authorized', wc_clean( $amount ) );
	}
	
	/**
	 * Saves the order payment attempts
	 *
	 * @param $value
	 */
	public function save_order_payment_attempts( $value ) {
		Compatibility::update_meta( $this->order, '_psigate_order_payment_attempts', wc_clean( $value ) );
	}
	
	/**
	 * @param string $transaction_id
	 */
	public function complete_order( $transaction_id ) {
		if ( $this->is_pre_order_with_tokenization() ) {
			// Now that we have the info need for future payment, mark the order pre-ordered
			\WC_Pre_Orders_Order::mark_order_as_pre_ordered( $this->order );
		} else {
			$this->order->payment_complete( $transaction_id );
		}
	}
	
	/**---------------------------------------------------
	 * DELETE
	 * ---------------------------------------------------*/
	
	/**
	 * Deletes the billing ID from the order
	 *
	 * @since 1.5.0
	 *
	 * @return bool|int
	 */
	public function delete_payment_order_id() {
		return Compatibility::delete_meta( $this->order, '_psigate_order_id' );
	}
	
	/**
	 * Deletes the saved Card Account ID to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	public function delete_account_id() {
		return Compatibility::delete_meta( $this->order, '_psigate_account_id' );
	}
	
	/**
	 * Deletes the saved Card Serial Number to the given order.
	 *
	 * @since 1.5.0
	 *
	 * @return mixed
	 */
	public function delete_serial_number() {
		return Compatibility::delete_meta( $this->order, '_psigate_serial_no' );
	}
	
	/**
	 * Deletes the is payment captured mark
	 *
	 * @since 1.5.0
	 */
	public function delete_is_payment_captured() {
		return Compatibility::delete_meta( $this->order, '_psigate_is_payment_captured' );
	}
	
	/**
	 * Deletes the amount captured value
	 *
	 * @since 1.5.0
	 */
	public function delete_order_amount_captured() {
		return Compatibility::delete_meta( $this->order, '_psigate_order_amount_captured' );
	}
	
	/**
	 * Deletes the amount authorized meta
	 *
	 * @since 1.5.0
	 *
	 * @return bool|int
	 */
	public function delete_order_amount_authorized() {
		return Compatibility::delete_meta( $this->order, '_psigate_order_amount_authorized' );
	}
	
	/**
	 * Deletes the payment type details
	 *
	 * @since 1.5.0
	 *
	 * @return bool|int
	 */
	public function delete_payment_type_detials() {
		if ( Compatibility::is_wc_3_0() ) {
			$this->order->delete_meta_data( '_psigate_transaction_payment_type' );
			$this->order->delete_meta_data( '_psigate_transaction_card_brand' );
			$this->order->delete_meta_data( '_psigate_transaction_card_last_four' );
			$this->order->delete_meta_data( '_psigate_transaction_card_expiration' );
			
			return $this->order->save();
		}
		
		delete_post_meta( Compatibility::get_order_id( $this->order ), '_psigate_transaction_payment_type' );
		delete_post_meta( Compatibility::get_order_id( $this->order ), '_psigate_transaction_card_brand' );
		delete_post_meta( Compatibility::get_order_id( $this->order ), '_psigate_transaction_card_last_four' );
		delete_post_meta( Compatibility::get_order_id( $this->order ), '_psigate_transaction_card_expiration' );
		
		return true;
	}
	
	/**---------------------------------------------------
	 * Functional Checks
	 * ---------------------------------------------------*/
	
	/**
	 * Returns true, if order contains Subscription
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function contains_subscription() {
		if ( ! \WC_PsiGate::is_subscriptions_active() ) {
			return false;
		}
		
		if ( wcs_order_contains_subscription( $this->order )
		     || wcs_order_contains_renewal( $this->order ) ) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Returns whether or not the order is a WC_Subscription
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_subscription() {
		if ( ! \WC_PsiGate::is_subscriptions_active() ) {
			return false;
		}
		
		return wcs_is_subscription( $this->order );
	}
	
	/**
	 * Returns true, if order contains Pre-Order
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function contains_pre_order() {
		if ( ! \WC_PsiGate::is_pre_orders_active() ) {
			return false;
		}
		
		return \WC_Pre_Orders_Order::order_contains_pre_order( $this->order );
	}
	
	/**
	 * Returns true if the order is a pre-order and it requires tokenization(charged at release)
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_pre_order_with_tokenization() {
		if ( ! \WC_PsiGate::is_pre_orders_active() ) {
			return false;
		}
		
		return \WC_Pre_Orders_Order::order_contains_pre_order( $this->order )
		       && \WC_Pre_Orders_Order::order_requires_payment_tokenization( $this->order );
	}
}