<?php
/*
 * Plugin Name: WooCommerce PsiGate Payment Gateway
 * Plugin URI: http://woothemes.com/woocommerce
 * Description: Allows you to use <a href="http://www.psigate.com/">PsiGate</a> payment processor with the WooCommerce plugin.
 * Version: 1.5.3
 * Author: VanboDevelops
 * Author URI: http://www.vanbodevelops.com
 * Woo: 18714:6cfa2a5e4b92169a72e91181e382ecaa
 * WC requires at least: 2.6.0
 * WC tested up to: 3.7.0
 *
 *	Copyright: (c) 2012 - 2019 VanboDevelops
 *	License: GNU General Public License v3.0
 *	License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) ) {
	require_once( 'woo-includes/woo-functions.php' );
}

/**
 * Plugin updates
 */
woothemes_queue_update( plugin_basename( __FILE__ ), '6cfa2a5e4b92169a72e91181e382ecaa', '18714' );

if ( ! is_woocommerce_active() ) {
	return;
}

class WC_PsiGate {
	
	const MIN_PHP_VERSION = '5.3.0';
	const MIN_WC_VERSION = '2.6.0';
	/**
	 * Text domain string. Constant
	 */
	const TEXT_DOMAIN = 'wc_psigate';
	/**
	 * The plugin version
	 */
	const VERSION = '1.5.3';
	/**
	 * The files and folders version
	 * Should be changes every time there is a new class file added or one deleted
	 * @since 2.0
	 */
	const FILES_VERSION = '1.0.3';
	/**
	 * WC Logger object
	 * @var object
	 */
	private static $log;
	/**
	 * Plugin URL
	 * @var string
	 */
	private static $plugin_url;
	/**
	 * Plugin Path
	 * @var string
	 */
	private static $plugin_path;
	/**
	 * Is WC Subscriptions active
	 * @var bool
	 */
	private static $is_subscriptions_active;
	/**
	 * Hold WC Subscriptions main version
	 * @var bool
	 */
	private static $is_subscriptions_version;
	/**
	 * Is WC Pre-Orders active
	 * @var bool
	 */
	private static $is_pre_orders_active;
	/**
	 * Do we have debug mode enabled
	 * @var bool
	 */
	private static $is_debug_enabled;
	/**
	 * @var \WcPsigate\Admin\Admin
	 */
	public $admin;
	
	public function __construct() {
		$this->load_autoloader();
		
		// Add required files
		add_action( 'plugins_loaded', array( $this, 'load_gateway_files' ) );
		
		// Add a 'Settings' link to the plugin action links
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array(
			$this,
			'settings_support_link'
		), 10, 4 );
		
		add_action( 'init', array( $this, 'load_text_domain' ) );
		add_action( 'init', array( $this, 'load_plugin_essentials' ) );
		
		add_action( 'template_redirect', array( $this, 'manage_saved_cards' ) );
		
		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_psigate_gateway' ) );
	}
	
	/**
	 * Loads the plugin autoloader
	 *
	 * @since 1.5.0
	 */
	public function load_autoloader() {
		require_once( 'includes/autoloader.php' );
		
		$loader = new \WcPsigate\Autoloader( self::plugin_path(), self::FILES_VERSION, 'includes' );
		
		spl_autoload_register( array( $loader, 'load_classes' ) );
	}
	
	/**
	 * Localisation
	 **/
	public function load_text_domain() {
		load_plugin_textdomain( self::TEXT_DOMAIN, false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}
	
	public function load_plugin_essentials() {
		if ( is_admin() ) {
			$this->admin = new WcPsigate\Admin\Admin();
		}
		
		$scripts = new \WcPsigate\Scripts();
		$scripts->hooks();
	}
	
	/**
	 * Add 'Settings' link to the plugin actions links
	 *
	 * @since 1.1
	 * @return array associative array of plugin action links
	 */
	public function settings_support_link( $actions, $plugin_file, $plugin_data, $context ) {
		return array_merge(
			array( 'settings' => '<a href="' . \WcPsigate\Compatibility::gateway_settings_page( 'psigate' ) . '">' . __( 'Settings', self::TEXT_DOMAIN ) . '</a>' ),
			$actions
		);
	}
	
	/**
	 * Get the correct gateway class name to load
	 *
	 * @since 1.1
	 * @return string Class name
	 */
	public static function get_gateway_class() {
		if ( self::is_pre_orders_active()
		     || self::is_subscriptions_active() ) {
			$methods = 'WC_Gateway_PsiGate_Addons';
		} else {
			$methods = 'WC_Gateway_PsiGate';
		}
		
		return $methods;
	}
	
	/**
	 * Load gateway files
	 *
	 * @since 1.1
	 */
	public function load_gateway_files() {
		
		include_once( 'includes/class-wc-gateway-psigate.php' );
		
		if ( self::is_pre_orders_active()
		     || self::is_subscriptions_active() ) {
			include_once( 'includes/class-wc-gateway-psigate-addons.php' );
		}
		
		// Add the cards management
		if ( \WcPsigate\Compatibility::is_wc_3_0() ) {
			add_action( 'woocommerce_account_content', array( $this, 'display_saved_cards' ), 20 );
		} else {
			add_action( 'woocommerce_after_my_account', array( $this, 'display_saved_cards' ) );
		}
	}
	
	/**
	 * Safely get POST variables
	 *
	 * @since 1.1
	 *
	 * @param string $name POST variable name
	 *
	 * @return string The variable value
	 */
	public static function get_post( $name ) {
		if ( isset( $_POST[ $name ] ) ) {
			return $_POST[ $name ];
		}
		
		return null;
	}
	
	/**
	 * Safely get GET variables
	 *
	 * @since 1.1
	 *
	 * @param string $name GET variable name
	 *
	 * @return string The variable value
	 */
	public static function get_get( $name ) {
		if ( isset( $_GET[ $name ] ) ) {
			return $_GET[ $name ];
		}
		
		return null;
	}
	
	/**
	 * Get an array field
	 *
	 * @param string $name    Field name
	 * @param array  $array   The array
	 * @param string $default Default value
	 *
	 * @return string
	 */
	public static function get_field( $name, $array, $default = '' ) {
		if ( ! is_array( $array ) ) {
			return $default;
		}
		
		if ( isset( $array[ $name ] ) ) {
			return $array[ $name ];
		}
		
		return $default;
	}
	
	/**
	 * Add the gateway to WooCommerce
	 *
	 * @since 1.1
	 *
	 * @param array $methods
	 *
	 * @return array
	 */
	function add_psigate_gateway( $methods ) {
		$methods[] = self::get_gateway_class();
		
		return $methods;
	}
	
	/**
	 * Add debug log message
	 *
	 * @since 1.1
	 *
	 * @param string $message
	 */
	public static function add_debug_log( $message ) {
		if ( ! is_object( self::$log ) ) {
			self::$log = \WcPsigate\Compatibility::get_wc_logger();
		}
		
		if ( self::is_debug_enabled() ) {
			self::$log->add( 'psigate', $message );
		}
	}
	
	/**
	 * Check, if debug logging is enabled
	 *
	 * @since 1.2
	 * @return bool
	 */
	public static function is_debug_enabled() {
		if ( null !== self::$is_debug_enabled ) {
			return self::$is_debug_enabled;
		} else {
			$psigate = get_option( 'woocommerce_psigate_settings' );
			
			self::$is_debug_enabled = ( 'yes' == $psigate['debug'] );
			
			return self::$is_debug_enabled;
		}
	}
	
	/**
	 * Get the plugin url
	 *
	 * @since 1.1
	 * @return string
	 */
	public static function plugin_url() {
		if ( self::$plugin_url ) {
			return self::$plugin_url;
		}
		
		return self::$plugin_url = untrailingslashit( plugins_url( '/', __FILE__ ) );
	}
	
	/**
	 * Get the plugin path
	 *
	 * @since 1.1
	 * @return string
	 */
	public static function plugin_path() {
		if ( self::$plugin_path ) {
			return self::$plugin_path;
		}
		
		return self::$plugin_path = untrailingslashit( plugin_dir_path( __FILE__ ) );
	}
	
	/**
	 * Detect if WC Subscriptions is active
	 *
	 * @since 1.1
	 * @return bool True if active, False if not
	 */
	public static function is_subscriptions_active() {
		if ( is_bool( self::$is_subscriptions_active ) ) {
			return self::$is_subscriptions_active;
		}
		
		self::$is_subscriptions_active = false;
		if ( class_exists( 'WC_Subscriptions' ) || function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_active = true;
		}
		
		return self::$is_subscriptions_active;
	}
	
	/**
	 * Get back the Subsctiptions version.
	 *
	 * @since 1.3
	 * @return bool Main Subscriptions version number (e.i. 1, 2, 3), False, if Subscriptions is not active
	 */
	public static function get_subscriptions_version() {
		if ( null !== self::$is_subscriptions_version ) {
			return self::$is_subscriptions_version;
		}
		
		self::$is_subscriptions_version = false;
		
		if ( function_exists( 'wcs_order_contains_subscription' ) ) {
			self::$is_subscriptions_version = 2;
		} elseif ( class_exists( 'WC_Subscriptions' ) ) {
			self::$is_subscriptions_version = 1;
		}
		
		return self::$is_subscriptions_version;
	}
	
	/**
	 * Detect if Pre-Orders is active
	 *
	 * @since 1.1
	 * @return bool True if actiove, False if not
	 */
	public static function is_pre_orders_active() {
		if ( is_bool( self::$is_pre_orders_active ) ) {
			return self::$is_pre_orders_active;
		}
		
		self::$is_pre_orders_active = false;
		if ( class_exists( 'WC_Pre_Orders' ) ) {
			self::$is_pre_orders_active = true;
		}
		
		return self::$is_pre_orders_active;
	}
	
	/**
	 * Manage the customer saved credit cards
	 *
	 * @since 1.1
	 */
	public function display_saved_cards() {
		
		// Customer should be logged in
		if ( ! is_user_logged_in() ) {
			return;
		}
		
		// Init the gateway
		$gateway_name = self::get_gateway_class();
		/**
		 * @var WC_Gateway_PsiGate $gateway
		 */
		$gateway = new $gateway_name;
		
		// Don't show if the gateway is not available for use
		if ( ! $gateway->is_available() ) {
			return;
		}
		
		$saved_cards = get_user_meta( get_current_user_id(), '_psigate_saved_cards', false );
		
		// Don't show if there are not profiles or the save customers is not enabled
		if ( ! $saved_cards || 'no' == $gateway->save_customers ) {
			return;
		}
		
		wc_get_template(
			'myaccount/myaccount-psigate-saved-cards.php',
			array(
				'saved_cards' => $saved_cards,
			),
			'',
			WC_PsiGate::plugin_path() . '/templates/'
		);
	}
	
	/**
	 * Manage the saved credit cards
	 *
	 * @since 1.1
	 */
	public function manage_saved_cards() {
		try {
			// Customer should be logged in
			if ( ! is_user_logged_in() ) {
				return;
			}
			
			// No need to continue, if an update is not submitted
			if ( null == self::get_post( 'psigate-delete-card' ) ) {
				return;
			}
			
			if ( '' != self::get_post( 'psigate-delete-card' ) && is_account_page() ) {
				
				// Init the gateway
				$gateway_name = self::get_gateway_class();
				
				/**
				 * @var WC_Gateway_PsiGate $gateway
				 */
				$gateway = new $gateway_name;
				
				// First verify the request was send from the manage card form
				$this->verify_request( 'psigate-nonce', 'psigate-verification' );
				
				$gateway->delete_customer_card( (int) self::get_post( 'psigate-delete-card' ) );
				
				wc_add_notice( __( 'Card deleted successfully', WC_PsiGate::TEXT_DOMAIN ), 'success' );
			}
		}
		catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}
	
	/**
	 * Verify a form request action
	 *
	 * @since 1.1
	 *
	 * @param type $action
	 *
	 * @throws Exception
	 */
	public function verify_request( $name, $action ) {
		if ( ! wp_verify_nonce( self::get_post( $name ), $action ) ) {
			throw new Exception( __( 'Cannot verify the request, please try again.', WC_PsiGate::TEXT_DOMAIN ) );
		}
	}
}

new WC_PsiGate(); //end PsiGate class