<?php

    /**
     * WC_Gateway_SagePaymentUSA_API_AddOns class.
     *
     * @extends WC_Gateway_SagePaymentUSA_API
     * Adds subscriptions support support.
     */
    class WC_Gateway_SagePaymentUSA_API_AddOns extends WC_Gateway_SagePaymentUSA_API {

        /**
         * __construct function.
         *
         * @access public
         * @return void
         */
        public function __construct() {
            parent::__construct();

            // Subscriptions
            if ( class_exists( 'WC_Subscriptions_Order' ) ) {
                
                // if ( function_exists( 'wcs_order_contains_subscription' ) ) {
                if ( version_compare( WC_Subscriptions::$version,'2.0', '<' ) ) {
                    // Pre Subscriptions 2.0
                    add_action( 'scheduled_subscription_payment_' . $this->id, array( $this, 'process_scheduled_subscription_payment' ), 10, 3 );
                    add_filter( 'woocommerce_subscriptions_renewal_order_meta_query', array( $this, 'remove_renewal_order_meta' ), 10, 4 );
                } else {
                    // Subscriptions 2.0
                    add_action( 'woocommerce_scheduled_subscription_payment_' . $this->id, array( $this, 'woocommerce_process_scheduled_subscription_payment' ), 10, 2 );
                    add_filter( 'wcs_renewal_order_meta_query', array( $this, 'remove_renewal_order_meta' ), 10, 3 );
                }

                // display the credit card used for a subscription in the "My Subscriptions" table
                if ( class_exists( 'WC_Payment_Token_CC' ) ) {
                    add_filter( 'woocommerce_my_subscriptions_payment_method', array( $this, 'maybe_render_subscription_payment_method' ), 10, 2 );

                    add_action( 'woocommerce_subscriptions_changed_failing_payment_method_$this->id', array( $this, 'update_failing_payment_method' ), 10, 3 );
                }

            }

        }

        /**
         * process scheduled subscription payment pre 2.0
         */
        function process_scheduled_subscription_payment( $amount_to_charge, $order, $product_id = NULL ) {

            if( !is_object( $order ) ) {
                $order = new WC_Order( $order );
            }

            // WooCommerce 3.0 compatibility 
            $order_id   = is_callable( array( $order, 'get_id' ) ) ? $order->get_id() : $order->id;

            // Get parent order ID
            $subscription       = new WC_Subscription( $order_id );
            $parent_order       = is_callable( array( $subscription, 'get_parent' ) ) ? $subscription->get_parent() : $subscription->order;
            $parent_order_id    = is_callable( array( $parent_order, 'get_id' ) ) ? $parent_order->get_id() : $parent_order->id;

            // Get the token from the parent order
            $token_value        = get_post_meta( $parent_order_id, '_SageToken', TRUE );

            // Setup variables for the transaction.
            $order_variables = $this->get_order_variables( $order );

            // Set up the request for Sage
            $requestData = [
                "Ecommerce" => [
                    "OrderNumber" => $order->get_order_number(),
                    "Amounts"   => [
                        "Total"         => $order_variables['order_total'],
                        "tax"           => $order->get_total_tax(),
                        "shipping"      => $order_variables['total_shipping'],
                    ],
                    "customer"  => [
                        "email"         =>  $order_variables['billing_email'],
                        "telephone"     =>  preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
                    ],
                    "billing"   => [
                        "name"          =>  $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
                        "address"       =>  $order_variables['billing_address_1'],
                        "city"          =>  $order_variables['billing_city'],
                        "state"         =>  $order_variables['billing_state'],
                        "postalCode"    =>  $order_variables['billing_postcode'],
                        "country"       =>  $order_variables['billing_country']
                    ],
                    "shipping"  => [
                        "name"          =>  $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
                        "address"       =>  $order_variables['shipping_address_1'],
                        "city"          =>  $order_variables['shipping_city'],
                        "state"         =>  $order_variables['shipping_state'],
                        "postalCode"    =>  $order_variables['shipping_postcode'],
                        "country"       =>  $order_variables['shipping_country']
                    ],
                ],
                "Vault"     => [
                    "Token"         => $token_value,
                    "Operation"     => "Read" // CRUD
                ],
            ];

            $requestData = apply_filters( 'woocommerce_sagepaymentsapi_version1api_data', $requestData, $order );

            // sale url
            $url    = $this->apiendpoint . "bankcard/v1/charges";
            $result = $this->apirequest( $url, $requestData, "Sale", "POST" );

            if( is_wp_error( $result ) ) {

                WC_Subscriptions_Manager::process_subscription_payment_failure_on_order( $order, $product_id );
                $order->add_order_note( $result->get_error_message() );

            } else {

                $response   = json_decode($result['body'], TRUE);

                // Store the transaction information from Sage
                update_post_meta( $order_id, '_sageresult' , $response );

                if( $response['status'] === 'Approved' ) {

                    /**
                     * Successful payment
                     */
                    $order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />
                                                Status: '           . $response['status'] . '<br />
                                                Approval Msg: '     . $response['message'] .'<br />
                                                Reference: '        . $response['reference'] );

                    // Get subs to update the order
                    WC_Subscriptions_Manager::process_subscription_payments_on_order( $order );

                    // Mark as paid
                    $order->payment_complete( $response['reference'] );

                } else {

                    // Payment failed, update things accordingly
                    WC_Subscriptions_Manager::process_subscription_payment_failure_on_order( $order, $product_id );

                    $cancelNote = __('Payment failed', 'woocommerce-spusa') . 
                                    __('Payment was rejected due to an error', 'woocommerce-spusa') . 
                                    ': "' . $response['message'] . '". ';
                    $order->add_order_note( $cancelNote );

                }

            }

        } // process scheduled subscription payment

        /**
         * process scheduled subscription payment for Subscriptions 2.0
         */
        function woocommerce_process_scheduled_subscription_payment( $amount_to_charge, $order ) {

            if( !is_object( $order ) ) {
                $order = new WC_Order( $order );
            }

            // WooCommerce 3.0 compatibility 
            $order_id   = is_callable( array( $order, 'get_id' ) ) ? $order->get_id() : $order->id;

            // Check WC version - changes for WC 3.0.0
            $pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

            /**
             * Get parent order ID
             */
            $subscriptions = wcs_get_subscriptions_for_renewal_order( $order_id );
            foreach( $subscriptions as $subscription ) {

                $parent_order      = is_callable( array( $subscription, 'get_parent' ) ) ? $subscription->get_parent() : $subscription->order;

                $parent_order_id   = is_callable( array( $parent_order, 'get_id' ) ) ? $parent_order->get_id() : $parent_order->id;
                $subscription_id   = is_callable( array( $subscription, 'get_id' ) ) ? $subscription->get_id() : $subscription->id;

            }

            // Get the token from the parent order
            $token_value = get_post_meta( $parent_order_id, '_SageToken', TRUE );

            // Setup variables for the transaction.
            $order_variables = $this->get_order_variables( $order );

            // Set up the request for Sage
            $requestData = [
                "Ecommerce" => [
                    "OrderNumber" => $order->get_order_number(),
                    "Amounts"   => [
                        "Total"         => $order_variables['order_total'],
                        "tax"           => $order->get_total_tax(),
                        "shipping"      => $order_variables['total_shipping'],
                    ],
                    "customer"  => [
                        "email"         =>  $order_variables['billing_email'],
                        "telephone"     =>  preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
                    ],
                    "billing"   => [
                        "name"          =>  $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
                        "address"       =>  $order_variables['billing_address_1'],
                        "city"          =>  $order_variables['billing_city'],
                        "state"         =>  $order_variables['billing_state'],
                        "postalCode"    =>  $order_variables['billing_postcode'],
                        "country"       =>  $order_variables['billing_country']
                    ],
                    "shipping"  => [
                        "name"          =>  $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
                        "address"       =>  $order_variables['shipping_address_1'],
                        "city"          =>  $order_variables['shipping_city'],
                        "state"         =>  $order_variables['shipping_state'],
                        "postalCode"    =>  $order_variables['shipping_postcode'],
                        "country"       =>  $order_variables['shipping_country']
                    ],
                ],
                "Vault"     => [
                    "Token"         => $token_value,
                    "Operation"     => "Read" // CRUD
                ],
            ];

            $requestData = apply_filters( 'woocommerce_sagepaymentsapi_version1api_data', $requestData, $order );

            // sale url
            $url    = $this->apiendpoint . "bankcard/v1/charges";
            $result = $this->apirequest( $url, $requestData, "Sale", "POST" );

            if( is_wp_error( $result ) ) {

                WC_Subscriptions_Manager::process_subscription_payment_failure_on_order( $order );
                $order->add_order_note( $result->get_error_message() );

            } else {

                $response   = json_decode($result['body'], TRUE);

                if( isset( $response['status'] ) && $response['status'] === 'Approved' ) {

                    /**
                     * Successful payment
                     */
                    $order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />
                                                Status: '           . $response['status'] . '<br />
                                                Approval Msg: '     . $response['message'] .'<br />
                                                Reference: '        . $response['reference'] );

                    // Store the transaction information from Sage
                    update_post_meta( $order_id, '_sageresult' , $response );

                    
                    // Get subs to update the order
                    WC_Subscriptions_Manager::process_subscription_payments_on_order( $order );

                    // Mark as paid
                    $order->payment_complete( $response['reference'] );

                } else {

                    // Payment failed, update things accordingly
                    WC_Subscriptions_Manager::process_subscription_payment_failure_on_order( $order );

                    $cancelNote = __('Payment failed', 'woocommerce-spusa') . 
                                    __('Payment was rejected due to an error', 'woocommerce-spusa') . 
                                    ': "' . $response['message'] . '". ';
                    $order->add_order_note( $cancelNote );

                }

            }

        } // process scheduled subscription payment

        /**
         * Update the renewal order with the transaction info from Sage 
         * and update the original order with the renewal order transaction information.
         */
        private function add_notes_scheduled_subscription_order( $sageresult, $order_id, $original_order_id, $VendorTxCode ) {

            $order = new WC_Order( $order_id );

            /**
             * Successful payment
             */
            $successful_ordernote = '';

            foreach ( $sageresult as $key => $value ) {
                $successful_ordernote .= $key . ' : ' . $value . "\r\n";
            }

            $order->add_order_note( __('Payment completed', 'woocommerce_sagepayform') . '<br />' . $successful_ordernote );

            update_post_meta( $order_id, '_transaction_id', str_replace( array('{','}'),'',$sageresult['VPSTxId'] ) );
            update_post_meta( $order_id, '_VPSTxId' , str_replace( array('{','}'),'',$sageresult['VPSTxId'] ) );
            update_post_meta( $order_id, '_SecurityKey' , $sageresult['SecurityKey'] );
            update_post_meta( $order_id, '_TxAuthNo' , $sageresult['TxAuthNo'] );
            delete_post_meta( $order_id, '_CV2Result' );
            delete_post_meta( $order_id, '_3DSecureStatus' );

            // update the original order with the renewal order transaction information
            update_post_meta( $original_order_id, '_RelatedVPSTxId' , str_replace( array('{','}'),'',$sageresult['VPSTxId'] ) );
            update_post_meta( $original_order_id, '_RelatedVendorTxCode' , $VendorTxCode );
            update_post_meta( $original_order_id, '_RelatedSecurityKey' , $sageresult['SecurityKey'] );
            update_post_meta( $original_order_id, '_RelatedTxAuthNo' , $sageresult['TxAuthNo'] );

        }

        /**
         * Don't transfer Stripe customer/token meta when creating a parent renewal order.
         *
         * @access public
         * @param array $order_meta_query MySQL query for pulling the metadata
         * @param int $original_order_id Post ID of the order being used to purchased the subscription being renewed
         * @param int $renewal_order_id Post ID of the order created for renewing the subscription
         * @param string $new_order_role The role the renewal order is taking, one of 'parent' or 'child'
         * @return void
         */
        public function remove_renewal_order_meta( $order_meta_query, $original_order_id, $renewal_order_id, $new_order_role = NULL ) {
            if ( 'parent' == $new_order_role ) {
                $order_meta_query .= " AND `meta_key` NOT IN ( '_VPSTxId', '_SecurityKey', '_TxAuthNo', '_RelatedVPSTxId', '_RelatedSecurityKey', '_RelatedTxAuthNo', '_CV2Result', '_3DSecureStatus' ) ";
            }
            return $order_meta_query;
        }

        /**
         * Update the customer_id for a subscription after using SagePay to complete a payment to make up for
         * an automatic renewal payment which previously failed.
         *
         * @access public
         * @param WC_Order $original_order The original order in which the subscription was purchased.
         * @param WC_Order $renewal_order The order which recorded the successful payment (to make up for the failed automatic payment).
         * @param string $subscription_key A subscription key of the form created by @see WC_Subscriptions_Manager::get_subscription_key()
         * @return void
         */
        public function update_failing_payment_method( $original_order, $renewal_order, $subscription_key ) {
            update_post_meta( $original_order->id, '_SagePayDirectToken', get_post_meta( $new_renewal_order->id, '_SagePayDirectToken', true ) );
        }

        /**
         * Render the payment method used for a subscription in the "My Subscriptions" table
         *
         * @param string $payment_method_to_display the default payment method text to display
         * @param array $subscription_details the subscription details
         * @param WC_Order $order the order containing the subscription
         * @return string the subscription payment method
         */
        public function maybe_render_subscription_payment_method( $payment_method_to_display, $subscription ) {
            // bail for other payment methods
            if ( $this->id != $subscription->payment_method || ! $subscription->customer_user ) {
                return $payment_method_to_display;
            }

            $sage_token     = get_post_meta( $subscription->order->id, '_SagePayDirectToken', true );
            $sage_token_id  = $this->get_token_id( $sage_token );

            $token = new WC_Payment_Token_CC();
            $token = WC_Payment_Tokens::get( $sage_token_id );

            if( $token ) {
                $payment_method_to_display = sprintf( __( 'Via %s card ending in %s', 'woocommerce_sagepayform' ), $token->get_card_type(), $token->get_last4() );
            }

            return $payment_method_to_display;
        }

        /**
         * Get the Token ID from the database using the token from Sage
         * @param  [type] $token [description]
         * @return [type]        [description]
         */
        function get_token_id( $token ) {
            global $wpdb;

            $id = NULL;

            if ( $token ) {
                $tokens = $wpdb->get_row( $wpdb->prepare(
                    "SELECT token_id FROM {$wpdb->prefix}woocommerce_payment_tokens WHERE token = %s",
                    $token
                ) );
            }

            return $tokens->token_id;
        }

    }
