<?php

    /**
     * WC_Gateway_SagePaymentUSA_API class.
     *
     * @extends WC_Payment_Gateway / WC_Payment_Gateway_CC
     *
     * WC_Payment_Gateway_CC is new for WC 2.6
     */
    if( class_exists( 'WC_Payment_Gateway_CC' ) ) {
        class _WC_Gateway_SagePaymentUSA_API extends WC_Payment_Gateway_CC {}
    } else {
        class _WC_Gateway_SagePaymentUSA_API extends WC_Payment_Gateway {}
    }

    class WC_Gateway_SagePaymentUSA_API extends _WC_Gateway_SagePaymentUSA_API {

        /**
         * __construct function.
         *
         * @access public
         * @return void
         */
        public function __construct() {
            $this->id                   		= 'sagepaymentsusaapi';
            $this->method_title         		= __( 'Sage Payment Solutions API', 'woocommerce-spusa' );
            $this->method_description   		= __( 'Activate Sage Payment Solutions API and let your customers pay via Sage Payment Solutions. For help and trouble shooting visit the <a href="'.SPUSADOCSURL.'" target="_blank"> plugin docs page</a> ', 'woocommerce-spusa' );
            $this->icon                 		= apply_filters( 'wc_sagepaymentsusaapi_icon', '' );
            $this->has_fields           		= true;
            $this->liveurl              		= 'https://gateway.sagepayments.net/cgi-bin/eftbankcard.dll?transaction';
            $this->default_order_button_text	= __( 'Pay Securely with Sage', 'woocommerce-spusa' );
            $this->default_legacyordernotes		= false;
            $this->default_forcetokens			= false;

            // Load the form fields
            $this->init_form_fields();

            // Load the settings.
            $this->init_settings();

            // API Details
            $this->production_ClientID		= "IxZLfpktvq4AHmNOIJGBNUJS1P9CftvQ";
			$this->production_ClientSecret	= "WwVxkKOoVl1yGmci";
			$this->test_ClientID			= "RVhgZuumTYf7DNLmcZETqvOpnALpGsPj";
			$this->test_ClientSecret		= "WpDmRIoSIeV4SjGu";

			$this->default_M_id_testing		= "137251282749";
			$this->default_M_key_testing	= "P2Q9X2W5R9B4";

            // Get setting values
            $this->enabled			= $this->settings['enabled'];
            $this->title			= $this->settings['title'];
            $this->description		= $this->settings['description'];
            $this->status			= isset( $this->settings['status'] ) ? $this->settings['status'] : 'testing';
            $this->apiversion		= isset( $this->settings['apiversion'] ) && $this->settings['apiversion'] === '0' ? 0 : 1;
            // Set the Client ID and Client Secret for production/testing
            $this->clientId 		= isset( $this->settings['status'] ) && $this->settings['status'] === 'live' ? $this->production_ClientID : $this->test_ClientID;
            $this->clientSecret 	= isset( $this->settings['status'] ) && $this->settings['status'] === 'live' ? $this->production_ClientSecret : $this->test_ClientSecret;

            $this->M_id				= $this->settings['M_id'];
			$this->M_key			= $this->settings['M_key'];
			$this->M_id_testing		= isset( $this->settings['M_id_testing'] )  ? $this->settings['M_id_testing']  : $this->default_M_id_testing;
			$this->M_key_testing	= isset( $this->settings['M_key_testing'] ) ? $this->settings['M_key_testing'] : $this->default_M_key_testing;
			
			$this->T_code			= $this->settings['T_code'];
			$this->cvv				= $this->settings['cvv'];
			$this->cardtypes		= $this->settings['cardtypes'];
			$this->debug			= isset( $this->settings['debug'] ) && $this->settings['debug'] == 'yes' ? true : false;
			$this->order_button_text= isset( $this->settings['order_button_text'] ) ? $this->settings['order_button_text'] : $this->default_order_button_text;
			$this->legacyordernotes = isset( $this->settings['legacyordernotes'] ) ? $this->settings['legacyordernotes'] : $this->default_legacyordernotes;
			$this->forcetokens 		= isset( $this->settings['forcetokens'] ) ? $this->settings['forcetokens'] : $this->default_forcetokens;

			if ( $this->status == 'live' ) {
				$this->merchant_id  = $this->M_id;
				$this->merchant_key = $this->M_key;
				$this->apiendpoint	= "https://api.sagepayments.com/";
			} else {
				$this->merchant_id  = $this->M_id_testing;
				$this->merchant_key = $this->M_key_testing;	
				$this->apiendpoint	= "https://api-cert.sagepayments.com/";		
			}

            // Hooks
			add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );

			// API
            add_action( 'woocommerce_api_wc_gateway_sagepaymentsusaapi', array( $this, 'check_sagepay_response' ) );

            add_action( 'valid-sagepaymentsusaapi-request', array( $this, 'successful_request' ) );
            add_action( 'woocommerce_receipt_sagepaymentsusaapi', array( $this, 'receipt_page' ) );
			
			// SSL Check
			add_action( 'admin_notices', array( $this,'sagepaymentsusaapi_ssl_check') );

			// Scripts
			add_action( 'wp_enqueue_scripts', array( $this, 'sagepaymentsusaapi_scripts' ) );

			/**
			 * What this gateway supports
			 * @var array
			 */
			$this->supports = array(
  									'products',
  									'refunds',
									);

			// Add support for additional methods if using API version 1
			if( $this->apiversion === 1 ) {
				$this->supports = array(
            						'products',
            						'refunds',
									'subscriptions',
									'subscription_cancellation',
									'subscription_reactivation',
									'subscription_suspension',
									'subscription_amount_changes',
									'subscription_payment_method_change',
									'subscription_payment_method_change_customer',
									'subscription_payment_method_change_admin',
									'subscription_date_changes',
									'multiple_subscriptions',
            						'pre-orders',
									'tokenization',
									'default_credit_card_form'
								);
				
				// Capture authorised payments
				add_action ( 'woocommerce_order_action_sage_process_payment', array( $this, 'process_pre_order_release_payment' ) );
	            
	            // Pre-Orders
	            if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
	                add_action( 'wc_pre_orders_process_pre_order_completion_payment_' . $this->id, array( $this, 'process_pre_order_release_payment' ) );
	            }

			}

			// Add test card info to the description if in test mode
			if ( $this->status != 'live' ) {
				$this->description .= ' ' . sprintf( __( '<br />TEST MODE ENABLED.<br />In test mode, you can use Visa card number 4111 1111 1111 1111 with any CVC and a valid expiration date or check the documentation (<a href="%s">Test card details for your test transactions</a>) for more card numbers.', 'woocommerce_sagepayform' ), 'https://docs.woocommerce.com/document/sage-payments-usa/#section-3' );
				$this->description  = trim( $this->description );
			}
				
        } // END __construct

        /**
         * init_form_fields function.
         *
         * @access public
         * @return void
         */
        function init_form_fields() {
        	require( SPUSAPLUGINPATH . 'assets/php/function-init_form_fields.php' );
		}

		/**
		 * Returns the plugin's url without a trailing slash
		 */
		public function get_plugin_url() {

			return str_replace('/classes','',untrailingslashit( plugins_url( '/', __FILE__ ) ) );
		}
		
		/**
		 * Check if SSL is enabled and notify the user
	 	 */
		function sagepaymentsusaapi_ssl_check() {
	     
		     if ( get_option('woocommerce_force_ssl_checkout')=='no' && $this->enabled=='yes' ) :
	     
		     	echo '<div class="error"><p>'.sprintf(__('Sage Payment Solutions API is enabled and the <a href="%s">force SSL option</a> is disabled; your checkout is not secure! Please enable SSL and ensure your server has a valid SSL certificate.', 'woocommerce-spusa'), admin_url('admin.php?page=woocommerce')).'</p></div>';
	     
	     	endif;
		}

		/**
		 * Enqueue Scipts
		 */
		function sagepaymentsusaapi_scripts() {
			wp_enqueue_style( 'wc-spusa', $this->get_plugin_url().'/assets/css/checkout.css' );

			if ( ! wp_script_is( 'jquery-payment', 'registered' ) ) {
				wp_register_script( 'jquery-payment', $this->get_plugin_url().'/assets/js/jquery.payment.js', array( 'jquery' ), '1.0.2', true );
			}

			if ( ! wp_script_is( 'wc-credit-card-form', 'registered' ) ) {
				wp_register_script( 'wc-credit-card-form', $this->get_plugin_url().'/assets/js/credit-card-form.js', array( 'jquery', 'jquery-payment' ), WOOCOMMERCE_VERSION, true );
			}
		}

		/**
		 * Enqueues our tokenization script to handle some of the new form options.
		 */
		public function tokenization_script() {
			wp_enqueue_script(
				'sagepay-tokenization-form',
				SPUSAPLUGINURL.'assets/js/tokenization-form.js',
				array( 'jquery' ),
				WC()->version
			);
		}

		/**
		 * Add selected card icons to payment method label, defaults to Visa/MC/Amex/Discover
		 */
		public function get_icon() {
			global $woocommerce;

			$icon = '';

			if ( $this->icon ) :
		
				if ( get_option('woocommerce_force_ssl_checkout')=='no' ) :
					// use icon provided by filter
					$icon = '<img src="' . esc_url( $this->icon ) . '" alt="' . esc_attr( $this->title ) . '" />';			
				else :
					// use icon provided by filter
					$icon = '<img src="' . esc_url( WC_HTTPS::force_https_url( $this->icon ) ) . '" alt="' . esc_attr( $this->title ) . '" />';		
				endif;

			elseif ( ! empty( $this->cardtypes ) ) :

				if ( get_option('woocommerce_force_ssl_checkout')=='no' ) {

					// display icons for the selected card types
					foreach ( $this->cardtypes as $card_type ) {

						$icon .= '<img src="' . 
									esc_url( $this->get_plugin_url() . '/assets/card-' . 
									strtolower( str_replace(' ','-',$card_type) ) . '.png' ) . '" alt="' . 
									esc_attr( strtolower( $card_type ) ) . '" />';
					}

				} else {

					// display icons for the selected card types
					foreach ( $this->cardtypes as $card_type ) {

						$icon .= '<img src="' . 
									esc_url( WC_HTTPS::force_https_url( $this->get_plugin_url() ) . '/assets/card-' . 
									strtolower( str_replace(' ','-',$card_type) ) . '.png' ) . '" alt="' . 
									esc_attr( strtolower( $card_type ) ) . '" />';
					}

				}

			endif;

			return apply_filters( 'woocommerce_gateway_icon', $icon, $this->id );
		}

	    /**
		 * Credit Card Fields.
		 *
		 * Core credit card form which gateways can used if needed.
		 */
    	function sagepay_credit_card_form() {

			wp_enqueue_script( 'wc-credit-card-form' );

			$fields = array(
				'card-number-field' => '<p class="form-row form-row-wide">
					<label for="' . $this->id . '-card-number">' . __( "Card Number", 'woocommerce' ) . ' <span class="required">*</span></label>
					<input id="' . $this->id . '-card-number" class="input-text wc-credit-card-form-card-number" type="text" maxlength="20" autocomplete="off" placeholder="•••• •••• •••• ••••" name="' . $this->id . '-card-number" />
				</p>',
				'card-expiry-field' => '<p class="form-row form-row-first">
					<label for="' . $this->id . '-card-expiry">' . __( "Expiry (MM/YY)", 'woocommerce' ) . ' <span class="required">*</span></label>
					<input id="' . $this->id . '-card-expiry" class="input-text wc-credit-card-form-card-expiry" type="text" autocomplete="off" placeholder="MM / YY" name="' . $this->id . '-card-expiry" />
				</p>',
				'card-cvc-field' => '<p class="form-row form-row-last">
					<label for="' . $this->id . '-card-cvc">' . __( "Card Code", 'woocommerce' ) . ' <span class="required">*</span></label>
					<input id="' . $this->id . '-card-cvc" class="input-text wc-credit-card-form-card-cvc" type="text" autocomplete="off" placeholder="CVC" name="' . $this->id . '-card-cvc" />
				</p>'
			);
			?>
			<fieldset id="<?php echo $this->id; ?>-cc-form">
				<?php do_action( 'woocommerce_credit_card_form_before', $this->id ); ?>
				<?php echo $fields['card-number-field']; ?>
				<?php echo $fields['card-expiry-field']; ?>
				<?php echo $fields['card-cvc-field']; ?>
				<?php do_action( 'woocommerce_credit_card_form_after', $this->id ); ?>
				<div class="clear"></div>
			</fieldset>
			<?php
    	}

		/**
    	 * Payment form on checkout page
    	 */
		public function payment_fields() {

			if ( is_add_payment_method_page() ) {
				$pay_button_text = __( 'Add Card', 'woocommerce-spusa' );
			} else {
				$pay_button_text = '';
			}

        	if ( get_option('woocommerce_force_ssl_checkout')=='no' && $this->status=='live' ) :
		     	echo '<div class="error"><p>'.sprintf(__('Sage Payment Solutions API is enabled and the force SSL option is disabled; your checkout is not secure! Please enable SSL and ensure your server has a valid SSL certificate.', 'woocommerce-spusa') ).'<br /></p></div>';
			endif;

			echo '<div id="sagepay-payment-data">';

			if ( $this->description ) {
				echo apply_filters( 'wc_sagepay_description', wp_kses_post( $this->description ) );
			}

			// Saved cards
			if ( class_exists( 'WC_Payment_Token_CC' ) && $this->apiversion === 1 ) {
				$this->tokenization_script();
				$this->saved_payment_methods();
			}

			// CC Form
			$this->sagepay_credit_card_form();

            // Save card option
			if ( class_exists( 'WC_Payment_Token_CC' ) && $this->apiversion === 1 ) {
				$this->save_payment_method_checkbox();
			}

			echo '</div>';

		}

		/**
	     * Validate the payment form
	     */
		function validate_fields() {
			global $woocommerce;

			try {

				// Check WC version - changes for WC 3.0.0
				$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

				if ( $pre_wc_30 ) {
					$card_number 		= isset($_POST[$this->id . '-card-number']) ? woocommerce_clean($_POST[$this->id . '-card-number']) : '';
					$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? woocommerce_clean($_POST[$this->id . '-card-cvc']) : '';
					$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? woocommerce_clean($_POST[$this->id . '-card-expiry']) : '';
					// Check the credit card form for token variables
					$existing_token 	= isset($_POST['wc-sagepaymentsusaapi-payment-token']) ? woocommerce_clean( $_POST['wc-sagepaymentsusaapi-payment-token'] ) : false;
					$new_token 			= isset($_POST['wc-sagepaymentsusaapi-new-payment-method']) ? woocommerce_clean( $_POST['wc-sagepaymentsusaapi-new-payment-method'] ) : false;
				} else {
					$card_number 		= isset($_POST[$this->id . '-card-number']) ? wc_clean($_POST[$this->id . '-card-number']) : '';
					$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? wc_clean($_POST[$this->id . '-card-cvc']) : '';
					$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? wc_clean($_POST[$this->id . '-card-expiry']) : '';
					// Check the credit card form for token variables
					$existing_token 	= isset($_POST['wc-sagepaymentsusaapi-payment-token']) ? wc_clean( $_POST['wc-sagepaymentsusaapi-payment-token'] ) : false;
					$new_token 			= isset($_POST['wc-sagepaymentsusaapi-new-payment-method']) ? wc_clean( $_POST['wc-sagepaymentsusaapi-new-payment-method'] ) : false;
				}

				/**
				 * Check if we need to validate card form
				 */
				if ( $existing_token === false || $existing_token === 'new' ) {

					// Format values
					$card_number    = str_replace( array( ' ', '-' ), '', $card_number );
					$card_expiry    = array_map( 'trim', explode( '/', $card_expiry ) );
					$card_exp_month = str_pad( $card_expiry[0], 2, "0", STR_PAD_LEFT );
					$card_exp_year  = $card_expiry[1];
					// Make sure the expiry is only 2 digits
					$card_exp_year  = substr( $card_exp_year, -2, 2 );

					// Validate values
					if ( ! ctype_digit( $card_cvc ) ) {
						throw new Exception( __( 'Card security code is invalid (only digits are allowed) ' . $this->id . ' ' . $card_number . ' ' . $card_cvc, 'woocommerce-spusa' ) );
					}

					if (
						! ctype_digit( $card_exp_month ) ||
						! ctype_digit( $card_exp_year ) ||
						$card_exp_month > 12 ||
						$card_exp_month < 1 ||
						$card_exp_year < date('y')
					) {
						throw new Exception( __( 'Card expiration date is invalid, please use MM/YY', 'woocommerce-spusa' ) );
					}	

					if ( empty( $card_number ) || ! ctype_digit( $card_number ) ) {
						throw new Exception( __( 'Card number is invalid', 'woocommerce-spusa' ) );
					}

					return true;

				} else {
					return true;
				}

			} catch( Exception $e ) {
				wc_add_notice( $e->getMessage(), 'error' );
				return false;
			}
		}

		/**
		 * Process the payment and return the result
		 */
		function process_payment( $order_id ) {
			global $woocommerce;

			$order = new WC_Order( $order_id );

			if( !isset( $this->apiversion ) || $this->apiversion === 0 ) {

				$res = $this->apilegacy( $order_id );

				/**
				 * Retreive response
				 */
				if ( $res[0]['body'][1] == 'A' ) {
					
					/**
					 * Successful payment
					 */
					$order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />
												Approval Code: ' 	. substr($res[0]['body'], 2, 6) . '<br />
												Approval Msg: ' 	. substr($res[0]['body'], 8, 32) .'<br />
												Card Type: ' 		. $this->get_card_brand( $res[1] ) .'<br />
												Card Number: ' 		. 'XXXX-XXXX-XXXX-'.substr( $res[1],-4 ) .'<br />
												Expiry Date: ' 		. $res[2] . $res[3] . '<br />
												Reference: ' 		. substr($res[0]['body'], 46, 10) );
 
					
					$order->payment_complete( substr($res[0]['body'], 46, 10) );
		
					$woocommerce->cart->empty_cart();

					/**
					 * Empty awaiting payment session
					 */
					unset( WC()->session->order_awaiting_payment );
						
					/**
					 * Return thank you redirect
					 */
					if ( method_exists( $order, 'get_checkout_order_received_url' ) ) {
                        $redirect = $order->get_checkout_order_received_url();
                	} else {
                        $redirect = add_query_arg( 'key', $order->order_key, add_query_arg( 'order', $order_id, get_permalink( get_option( 'woocommerce_thanks_page_id' ) ) ) );
                	}

                    // Return thank you page redirect
                    return array(
                        'result'	=> 'success',
                        'redirect'	=> $redirect
                    );
	
				} else {
					
					$cancelNote = __('Payment failed', 'woocommerce-spusa') . 
									' (Response Code: ' . substr($res[0]['body'], 2, 6) . '). ' . 
									__('Payment was rejected due to an error', 'woocommerce-spusa') . 
									': "' . substr($res[0]['body'], 8, 32) . '". ';
					$order->add_order_note( $cancelNote );

					wc_add_notice( __('Payment error', 'woocommerce-spusa') . ': ' . substr($res[0]['body'], 8, 32), 'error' );

					if ( $this->debug == true ) {

						wc_add_notice( __('Internal error', 'woocommerce-spusa'), 'error' );

   						wc_add_notice(__('Debugging is on', 'woocommerce-spusa') . '', 'error');

   						wc_add_notice( '<p>' . 'XXXX-XXXX-XXXX-'. substr( $res[1],-4 ) . '<br />' .
   										$res[2] . $res[3], 'error' );

   						wc_add_notice( print_r( str_replace('&','<br />',$data), TRUE ), 'error' );
   					}

				}

			} else {
				
				$result = $this->apiversion1( $order_id );

				// Deal with $0 orders first, this result will just contain the token information.
				if( isset( $result['response']['data'] ) && $result['response']['status'] === 1 ) {

					// Mark the order as paid
					$order->payment_complete();

					// Make sure the cart is empty
					WC()->cart->empty_cart();

					/**
					 * Empty awaiting payment session
					 */
					unset( WC()->session->order_awaiting_payment );
						
					/**
					 * Return thank you redirect
					 */
					if ( method_exists( $order, 'get_checkout_order_received_url' ) ) {
			            $redirect = $order->get_checkout_order_received_url();
			    	} else {
			            $redirect = add_query_arg( 'key', $order->order_key, add_query_arg( 'order', $order->id, get_permalink( get_option( 'woocommerce_thanks_page_id' ) ) ) );
			    	}

			        // Return thank you page redirect
			        return array(
			            'result'	=> 'success',
			            'redirect'	=> $redirect
			        );
				
				} elseif( $result['response']['status'] === 'Approved' ) {

					/**
					 * Successful payment
					 */
					$result['response']['Card Type']   = $result['cardtype'];
					$result['response']['Card Number'] = $result['cardnumber'];
					$result['response']['Expiry Date'] = $result['cardexpiry'];

					// Add transaction information to order notes
					if( $this->legacyordernotes ) {

						$order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />
													Status: ' 			. $result['response']['status'] . '<br />
													Approval Msg: ' 	. $result['response']['message'] .'<br />
													Card Type: ' 		. $result['response']['Card Type'] .'<br />
													Card Number: ' 		. $result['response']['Card Number'] .'<br />
													Expiry Date: ' 		. $result['response']['Expiry Date'] . '<br />
													Reference: ' 		. $result['response']['reference'] );

					} else {

						foreach ( $result['response'] as $key => $value ) {
	                    	$successful_ordernote .= ucwords($key) . ' : ' . $value . "\r\n";
	                	}
	                	$order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />' . $successful_ordernote );

					}

	                // Store the transaction information from Sage
	                update_post_meta( $order_id, '_sageresult' , $result['response'] );


					// Mark the order as paid
					if ( class_exists('WC_Pre_Orders') && WC_Pre_Orders_Order::order_contains_pre_order( $order_id ) ) {
						// mark order as pre-ordered / reduce order stock
						WC_Pre_Orders_Order::mark_order_as_pre_ordered( $order );
					} else {
						$order->payment_complete( $result['response']['reference'] );
						// Check if this is an authorized only order
						if ( $this->T_code == "02" ) {
							$order->update_status( 'authorized', _x( 'Payment authorized, you will need to capture this payment before shipping. Use the "Capture Authorised Payment" option in the "Order Actions" dropdown.<br /><br />', 'woocommerce-spusa' ) );
						}
					}

					// Make sure the cart is empty
					WC()->cart->empty_cart();

					/**
					 * Empty awaiting payment session
					 */
					unset( WC()->session->order_awaiting_payment );
						
					/**
					 * Return thank you redirect
					 */
					if ( method_exists( $order, 'get_checkout_order_received_url' ) ) {
			            $redirect = $order->get_checkout_order_received_url();
			    	} else {
			            $redirect = add_query_arg( 'key', $order->order_key, add_query_arg( 'order', $order->id, get_permalink( get_option( 'woocommerce_thanks_page_id' ) ) ) );
			    	}

			        // Return thank you page redirect
			        return array(
			            'result'	=> 'success',
			            'redirect'	=> $redirect
			        );

				} else {

					foreach ( $result['response'] as $key => $value ) {
	                    $successful_ordernote .= $key . ' : ' . $value . "\r\n";
	                }

	                $order->add_order_note( __('Payment Failed', 'woocommerce-spusa') . '<br />' . $successful_ordernote );

					if ( $this->debug == true ) {
						wc_add_notice( '<pre>' . print_r( str_replace('&','<br />',$result), TRUE ) . '</pre>', 'error' );
					} else {
						wc_add_notice( __('Payment error', 'woocommerce-spusa') . ': ' . $result['response']['message'], 'error' );
						// Add failed transaction information to log when debugging is off.
						$this->sage_debug( $result, $this->id, __('Sage Response : ', 'woocommerce-spusa'), TRUE );
					}

				}

			}

		}

		/**
		 * Process the payment using the API Version 1 and return the result
		 */
		function apiversion1( $order_id ) {	
			global $woocommerce;

			$order = new WC_Order( $order_id );

			// Check WC version - changes for WC 3.0.0
			$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

			// Set token variables to false
			$token_status 	= false;
			$new_token		= false;
			$existing_token = false;
			$card_number 	= '';
			$card_exp_month = '';
			$card_exp_year 	= '';

			if ( $pre_wc_30 ) {
				$card_number 		= isset($_POST[$this->id . '-card-number']) ? woocommerce_clean($_POST[$this->id . '-card-number']) : '';
				$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? woocommerce_clean($_POST[$this->id . '-card-cvc']) : '';
				$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? woocommerce_clean($_POST[$this->id . '-card-expiry']) : '';
				// Check the credit card form for token variables
				$existing_token 	= isset($_POST['wc-sagepaymentsusaapi-payment-token']) ? woocommerce_clean( $_POST['wc-sagepaymentsusaapi-payment-token'] ) : false;
				$new_token 			= isset($_POST['wc-sagepaymentsusaapi-new-payment-method']) ? woocommerce_clean( $_POST['wc-sagepaymentsusaapi-new-payment-method'] ) : false;
			} else {
				$card_number 		= isset($_POST[$this->id . '-card-number']) ? wc_clean($_POST[$this->id . '-card-number']) : '';
				$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? wc_clean($_POST[$this->id . '-card-cvc']) : '';
				$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? wc_clean($_POST[$this->id . '-card-expiry']) : '';
				// Check the credit card form for token variables
				$existing_token 	= isset($_POST['wc-sagepaymentsusaapi-payment-token']) ? wc_clean( $_POST['wc-sagepaymentsusaapi-payment-token'] ) : false;
				$new_token 			= isset($_POST['wc-sagepaymentsusaapi-new-payment-method']) ? wc_clean( $_POST['wc-sagepaymentsusaapi-new-payment-method'] ) : false;
			}

			// Setup variables for the transaction.
			$order_variables = $this->get_order_variables( $order );

			// Authorization or Sale
			$type = $this->get_txtype( $order_id, $order_variables['order_total'] );

			// Existing Token?
			if( isset( $existing_token ) && $existing_token != 'new' ) {

				$token = new WC_Payment_Token_CC();
				$token = WC_Payment_Tokens::get( $existing_token );

				if ( $token ) {
					$customer_id = $pre_wc_30 ? $order->customer_user : $order->get_user_id();
					if ( $token->get_user_id() == $customer_id ) {
						// This is a valid token and belongs to this customer, we can process the order using it.
						$token_status 		= true;
						$token_value 		= $token->get_token();

						$card_type 			= $token->get_card_type();
						$masked_card_number = 'XXXX-XXXX-XXXX-' . $token->get_last4(); 
						$card_exp_month 	= $token->get_expiry_month();
						$card_exp_year		= $token->get_expiry_year();

						// Update Order with token info
						update_post_meta( $order_id, '_SageToken' , $token_value );

						// If the $order_total is 0 we don't need to go any further.
						if( $order_variables['order_total'] == 0 ) {

							// create a vaultResponse array
							$vaultResponse = array();
							$vaultResponse['status'] = 1;
							$vaultResponse['data']	 = $token_value;

							return array( 
								'response' 	 => $vaultResponse,
								'cardtype' 	 => $token->get_card_type(),
								'cardnumber' => 'XXXX-XXXX-XXXX-' . $token->get_last4(), 
								'cardexpiry' => $token->get_expiry_month() . $token->get_expiry_year()
							);

						}
					}
				}

			} else {
				// Format card values
				$card_number    	= str_replace( array( ' ', '-' ), '', $card_number );
				$card_expiry    	= array_map( 'trim', explode( '/', $card_expiry ) );
				$card_exp_month 	= str_pad( $card_expiry[0], 2, "0", STR_PAD_LEFT );
				$card_exp_year  	= $card_expiry[1];
				// Make sure the expiry is only 2 digits
				$card_exp_year  	= substr( $card_exp_year, -2, 2 );
				$card_type 			= $this->get_card_brand( $card_number );
				$masked_card_number = 'XXXX-XXXX-XXXX-'.substr( $card_number,-4 );
			}

		    // Check if we need to create a new token
		    // Tokens are required for Pre-Orders, Subscriptions and Authorizations
		    if ( !$token_status &&  ( 
					$new_token || 
					( class_exists( 'WC_Pre_Orders_Order' ) && WC_Pre_Orders_Order::order_contains_pre_order( $order_id ) ) || 
					( function_exists( 'wcs_order_contains_subscription' ) && wcs_order_contains_subscription( $order_id ) ) ||
					( $type == 'Authorization')
				) ) {
			    
			    $requestData = [
			        "CardData" => [
		                "Number" 		=> $card_number,
		                "Expiration" 	=> $card_exp_month.$card_exp_year
			        ]
			    ];

			    // token url
			    $url 				= $this->apiendpoint . "bankcard/v1/tokens";
			    $result 			= $this->apirequest( $url, $requestData, false, "POST" );
			    $decoded_result 	= json_decode($result['body'], TRUE);
			    $vaultResponse 		= $decoded_result['vaultResponse'];

			    // Set token_status to true is token is successfully created
			    if( $vaultResponse['data'] && $vaultResponse['status'] === 1 ) {

					$token_status 	= true;
					$token_value 	= $vaultResponse['data'];

					// Store the token for future use
					$this->save_token( $token_value, $card_number, $card_exp_month, $card_exp_year );

					// Update Order with new token info
					update_post_meta( $order_id, '_SageToken' , $token_value );

				}

				// If the $order_total is 0 we don't need to go any further.
				if( $order_variables['order_total'] == 0 ) {
					return array( 
						'response' 	 => $vaultResponse,
						'cardtype' 	 => $card_type,
						'cardnumber' => $masked_card_number, 
						'cardexpiry' => $card_exp_month . $card_exp_year
					);
				}

		    }

		    /**
		     * Build $requestData
		     */
		    if( $token_status && $token_value && $token_value != '' ) {
		    	// If $token_status is true and we have a $token_value then use the token.
			    $requestData = [
			        "Ecommerce" => [
			            "OrderNumber" => $order->get_order_number(),
			            "Amounts" 	=> [
			                "Total" 		=> $order_variables['order_total'],
			                "tax" 			=> $order->get_total_tax(),
			  				"shipping" 		=> $order_variables['total_shipping'],
			            ],
			            "customer" 	=> [
			  				"email" 		=>  $order_variables['billing_email'],
			  				"telephone" 	=>  preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
						],
				        "billing" 	=> [
						    "name" 			=>  $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
						    "address" 		=>  $order_variables['billing_address_1'],
						    "city" 			=>  $order_variables['billing_city'],
						    "state" 		=>  $order_variables['billing_state'],
						    "postalCode" 	=>  $order_variables['billing_postcode'],
						    "country" 		=>  $order_variables['billing_country']
				    	],
				    	"shipping" 	=> [
				      		"name" 			=>  $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
				      		"address" 		=>  $order_variables['shipping_address_1'],
				      		"city" 			=>  $order_variables['shipping_city'],
				      		"state" 		=>  $order_variables['shipping_state'],
				      		"postalCode" 	=>  $order_variables['shipping_postcode'],
				      		"country" 		=>  $order_variables['shipping_country']
				    	],
			        ],
			        "Vault" 	=> [
            			"Token" 		=> $token_value,
            			"Operation" 	=> "Read" // CRUD
        			],
			    ];

		    } elseif( $order_variables['order_total'] != 0 ) {
		    	// This is a non-token transaction
			    $requestData = [
			        "Ecommerce" => [
			            "OrderNumber" => $order->get_order_number(),
			            "Amounts" 	=> [
			                "Total" 		=> $order_variables['order_total'],
			                "tax" 			=> $order->get_total_tax(),
			  				"shipping" 		=> $order_variables['total_shipping'],
			            ],
			            "CardData" 	=> [
			                "Number" 		=> $card_number,
			                "Expiration" 	=> $card_exp_month . $card_exp_year
			            ],
			            "customer" 	=> [
			  				"email" 		=>  $order_variables['billing_email'],
			  				"telephone" 	=>  preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
						],
				        "billing" 	=> [
						    "name" 			=>  $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
						    "address" 		=>  $order_variables['billing_address_1'],
						    "city" 			=>  $order_variables['billing_city'],
						    "state" 		=>  $order_variables['billing_state'],
						    "postalCode" 	=>  $order_variables['billing_postcode'],
						    "country" 		=>  $order_variables['billing_country']
				    	],
				    	"shipping" 	=> [
				      		"name" 			=>  $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
				      		"address" 		=>  $order_variables['shipping_address_1'],
				      		"city" 			=>  $order_variables['shipping_city'],
				      		"state" 		=>  $order_variables['shipping_state'],
				      		"postalCode" 	=>  $order_variables['shipping_postcode'],
				      		"country" 		=>  $order_variables['shipping_country']
				    	],

			        ]
			    ];

			}

		    $requestData = apply_filters( 'woocommerce_sagepaymentsapi_version1api_data', $requestData, $order );

			// url
		    $url	= $this->apiendpoint . "bankcard/v1/charges";
			$result = $this->apirequest( $url, $requestData, $type, "POST" );

			if( is_wp_error( $result ) ) {

				wc_add_notice( __('Internal error.', 'woocommerce-spusa'), 'error' );

				if ( $this->debug == true ) {

					wc_add_notice(__('Debugging is on', 'woocommerce-spusa') , 'error');
					wc_add_notice(__('Error message : ', 'woocommerce-spusa') , 'error');

					$error_string = $result->get_error_message();
					wc_add_notice( '<p>' . $error_string . '</p>', 'error' );					

				}

			} else {

				$response 	= json_decode($result['body'], TRUE);

				/**
				 * Debugging
				 */
	  			if ( $this->debug == true ) {
	  				$this->sage_debug( $response, $this->id, __('Received from Sage : ', 'woocommerce-spusa'), TRUE );
				}

				return array( 
					'response' 	 => $response, 
					'cardtype' 	 => $card_type,
					'cardnumber' => $masked_card_number, 
					'cardexpiry' => $card_exp_month . $card_exp_year
				);
			
			}

		}

		/**
		 * Process the payment using the Legacy API and return the result
		 */
		function apilegacy( $order_id ) {			
			global $woocommerce;

			$order = new WC_Order( $order_id );

			// Check WC version - changes for WC 3.0.0
			$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

			if ( $pre_wc_30 ) {
				$card_number 		= isset($_POST[$this->id . '-card-number']) ? woocommerce_clean($_POST[$this->id . '-card-number']) : '';
				$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? woocommerce_clean($_POST[$this->id . '-card-cvc']) : '';
				$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? woocommerce_clean($_POST[$this->id . '-card-expiry']) : '';
			} else {
				$card_number 		= isset($_POST[$this->id . '-card-number']) ? wc_clean($_POST[$this->id . '-card-number']) : '';
				$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? wc_clean($_POST[$this->id . '-card-cvc']) : '';
				$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? wc_clean($_POST[$this->id . '-card-expiry']) : '';
			}

			// Format values
			$card_number    = str_replace( array( ' ', '-' ), '', $card_number );
			$card_expiry    = array_map( 'trim', explode( '/', $card_expiry ) );
			$card_exp_month = str_pad( $card_expiry[0], 2, "0", STR_PAD_LEFT );
			$card_exp_year  = $card_expiry[1];
			// Make sure the expiry is only 2 digits
			$card_exp_year  = substr( $card_exp_year, -2, 2 );
			
			// set the URL that data will be posted to.
			$eftsecure_url = "https://gateway.sagepayments.net/cgi-bin/eftbankcard.dll?transaction";

			// Setup variables for the transaction.
			$order_variables = $this->get_order_variables( $order );

			if ( $this->status == 'live' ) {
				$m_id  = $this->M_id;
				$m_key = $this->M_key;
			} else {
				$m_id  = $this->M_id_testing;
				$m_key = $this->M_key_testing;			
			}

			// make your query.
			$data	 = array(
				"m_id"			=> urlencode( $m_id ),
				"m_key"			=> urlencode( $m_key ),
				"T_ordernum"	=> urlencode( $order->get_order_number() ),
				"T_amt"			=> urlencode( $order_variables['order_total'] ),
				"C_name"		=> $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
				"C_address"		=> $order_variables['billing_address_1'],
				"C_state"		=> $order_variables['billing_state'],
				"C_city"		=> $order_variables['billing_city'],
				"C_zip"			=> $order_variables['billing_postcode'],
				"C_country"		=> $order_variables['billing_country'],
				"C_email"		=> $order_variables['billing_email'],
				"C_telephone"	=> preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
				"C_cardnumber"	=> urlencode( $card_number ),
				"C_exp"			=> urlencode( $card_exp_month . $card_exp_year ),
				"C_cvv"			=> urlencode( $card_cvc ),
				"T_code"		=> urlencode( $this->T_code ),
				"C_ship_name"	=> $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
				"C_ship_address"=> $$order_variables['shipping_address_1'],
				"C_ship_city"	=> $order_variables['shipping_city'],
				"C_ship_state"	=> $order_variables['shipping_state'],
				"C_ship_zip"	=> $order_variables['shipping_postcode'],
				"C_ship_country"=> $order_variables['shipping_country']
			);

			$data = apply_filters( 'woocommerce_sagepaymentsapi_data', $data, $order );

			/**
			 * Debugging
			 */
  			if ( $this->debug == true ) {
  				$this->sage_debug( $data, $this->id, __('Sent to Sage : ', 'woocommerce-spusa'), TRUE );
			}

			/**
			 * Convert the $data array for Sage
			 */
			$data = http_build_query( $data, '', '&' );

			/**
			 * Send the info to Sage for processing, use wp_remote_post cos Mike Jolley says so!
			 */
			$contents = array(
							'method' 		=> 'POST',
							'timeout' 		=> 45,
							'redirection' 	=> 5,
							'httpversion' 	=> '1.0',
							'blocking' 		=> true,
							'headers' 		=> array(),
							'body' 			=> $data,
							'cookies' 		=> array()
    					);

			$res = wp_remote_post( $eftsecure_url, $contents );

			if( is_wp_error( $res ) ) {

				wc_add_notice( __('Internal error.', 'woocommerce-spusa'), 'error' );

   				if ( $this->debug == true ) {

					wc_add_notice(__('Debugging is on', 'woocommerce-spusa') , 'error');
   					wc_add_notice(__('Error message : ', 'woocommerce-spusa') , 'error');

   					$error_string = $res->get_error_message();
   					wc_add_notice( '<p>' . $error_string . '</p>', 'error' );

   				}

			} else {
 
				return array( $res, $card_number, $card_exp_month, $card_exp_year );
			
			}
	
		}

		/**
		 * [process_refund description]
		 * @param  [type] $order_id [description]
		 * @param  [type] $amount   [description]
		 * @param  string $reason   [description]
		 * @return [type]           [description]
		 */
		public function process_refund( $order_id, $amount = NULL, $reason = '' ) {
  			if( !isset( $this->apiversion ) || $this->apiversion === 0 ) {
				// OLD API REFUNDS
				return $this->legacyrefund( $order_id, $amount, $reason );
			} else {
				return $this->apiversion1refund( $order_id, $amount, $reason );
			}
  			
		}

		/**
		 * [legacyrefund description]
		 * @param  [type] $order_id [description]
		 * @param  [type] $amount   [description]
		 * @param  string $reason   [description]
		 * @return [type]           [description]
		 */
		private function legacyrefund( $order_id, $amount = NULL, $reason = '' ) {

			$order = new WC_Order( $order_id );

			// set the URL that will be posted to.
			$eftsecure_url = "https://gateway.sagepayments.net/cgi-bin/eftbankcard.dll?transaction";

			if ( !$order->billing_state || $order->billing_state == '' ) :
				$state = $order->billing_city;
			else :
				$state = $order->billing_state;
			endif;
				 
			// make your query.
			if ( $this->status == 'live' ) {
				$data  = "m_id=" 		. 	$this->M_id;
				$data .= "&m_key=" 		. 	$this->M_key;
			} else {
				$data  = "m_id=" 		. 	$this->M_id_testing;
				$data .= "&m_key=" 		. 	$this->M_key_testing;			
			}

			$data .= "&T_ordernum=" 	. 	urlencode( $order->get_order_number() );
			$data .= "&T_amt=" 			. 	urlencode( $amount ); 
			$data .= "&C_name=" 		. 	urlencode( $order->billing_first_name . ' ' . $order->billing_last_name );
			$data .= "&C_address=" 		. 	urlencode( $order->billing_address_1 );
			$data .= "&C_state=" 		. 	urlencode( $state );
			$data .= "&C_city=" 		. 	urlencode( $order->billing_city );
			$data .= "&C_zip=" 			. 	urlencode( $order->billing_postcode );
			$data .= "&C_country=" 		. 	urlencode( $order->billing_country );
			$data .= "&C_email=" 		. 	urlencode( $order->billing_email );
			$data .= "&T_code=" 		. 	urlencode( '10' );
			$data .= "&T_REFERENCE=" 	. 	get_post_meta( $order_id, '_transaction_id', true );
				
			/**
			 * Send the info to Sage for processing
			 */
			$res = wp_remote_post( 
						$eftsecure_url, array(
							'method' 		=> 'POST',
							'timeout' 		=> 45,
							'redirection' 	=> 5,
							'httpversion' 	=> '1.0',
							'blocking' 		=> true,
							'headers' 		=> array(),
							'body' 			=> $data,
							'cookies' 		=> array()
	    			)
				);

			if( is_wp_error( $res ) ) {
				return false;
			} else {
				if ( $res['body'][1] == 'A' ) {
					/**
					 * Successful refund
					 */
					$order->add_order_note( __('Refund completed', 'woocommerce-spusa') . '<br />
												Reference: ' 		. substr($res['body'], 46, 10) );
					return true;

				} else {
					return false;
				}

			}

		}

		/**
		 * [apiversion1refund description]
		 * @param  [type] $order_id [description]
		 * @param  [type] $amount   [description]
		 * @param  string $reason   [description]
		 * @return [type]           [description]
		 */
		private function apiversion1refund( $order_id, $amount = NULL, $reason = '' ) {

			$order = new WC_Order( $order_id );

			if ( $this->status == 'live' ) {
				$m_id  = $this->M_id;
				$m_key = $this->M_key;
			} else {
				$m_id  = $this->M_id_testing;
				$m_key = $this->M_key_testing;			
			}
				
			// we'll need a new nonce and timestamp:
			$nonce = uniqid();
			$timestamp = (string)time();
			    
			// notice that the URL now uses "credits" instead of "charges"
			// by referencing the sale transaction, we can issue a refund to 
			// the card that was used without passing through the card #, exp, etc.
			$verb = "POST";
			$url = $this->apiendpoint . "bankcard/v1/credits/" . get_post_meta( $order_id, '_transaction_id', true );
			// in this example, we're doing a partial refund
			$requestData = [
			    "Amount" => $amount
			];

			// convert to json for transport
			$payload = json_encode($requestData);

			// the request is authorized via an HMAC header that we generate by
			// concatenating certain info, and then hashing it using our client key
			$toBeHashed = $verb . $url . $payload . $m_id . $nonce . $timestamp;
			$hmac 		= $this->getHmac($toBeHashed, $this->clientSecret);

			/**
			 * Build Header
			 */
			$header = array(
			            "clientId"		=> $this->clientId,
			            "merchantId"	=> $m_id,
			            "merchantKey"	=> $m_key,
			            "nonce"			=> $nonce,
			            "timestamp"		=> $timestamp,
			            "authorization"	=> $hmac,
			            "content-type"	=> "application/json",
			);

			/**
			 * Send the info to Sage for processing, use wp_remote_post cos Mike Jolley says so!
			 */
			$contents = array(
						'method' 		=> $verb,
						'timeout' 		=> 45,
						'redirection' 	=> 5,
						'httpversion' 	=> '1.0',
						'blocking' 		=> true,
						'headers' 		=> $header,
						'body' 			=> $payload,
						'cookies' 		=> array()
	 		);

			$result = wp_remote_post( $url, $contents );

			if( is_wp_error( $result ) ) {
				return false;
			} else {
				$response 	= json_decode($result['body'], TRUE);
				if ( $response['status'] === 'Approved' ) {
						
						$order->add_order_note( __('Refund completed', 'woocommerce-spusa') . '<br />
													Reference: ' 		. $response['reference'] );
					return true;
				} else {
					return false;
				}
			}

		}

		/**
		 * [apirequest description]
		 * @param  [type] $url         [description]
		 * @param  [type] $requestData [description]
		 * @return [type]              [description]
		 */
		function apirequest( $url, $requestData, $type = NULL, $verb = "POST" ) {

			// Build the URL
			if( $type ) {
				$url = $url . "?type=" . $type;
			}

			// the nonce can be any unique identifier -- guids and timestamps work well
		    $nonce = uniqid();
		    
		    // a standard unix timestamp. a request must be received within 60s
		    // of its timestamp header.
		    $timestamp = (string)time();

			// convert to json for transport
		    $payload = json_encode($requestData);

		    // the request is authorized via an HMAC header that we generate by
		    // concatenating certain info, and then hashing it using our client key
		    $toBeHashed = $verb . $url . $payload . $this->merchant_id . $nonce . $timestamp;
		    $hmac 		= $this->getHmac($toBeHashed, $this->clientSecret);

		    /**
		     * Build Header
		     */
		    $header = array(
		                "clientId"		=> $this->clientId,
		                "merchantId"	=> $this->merchant_id,
		                "merchantKey"	=> $this->merchant_key,
		                "nonce"			=> $nonce,
		                "timestamp"		=> $timestamp,
		                "authorization"	=> $hmac,
		                "content-type"	=> "application/json",
		            );
			/**
			 * Send the info to Sage for processing
			 */
			$contents = array(
						'method' 		=> $verb,
						'timeout' 		=> 45,
						'redirection' 	=> 5,
						'httpversion' 	=> '1.0',
						'blocking' 		=> true,
						'headers' 		=> $header,
						'body' 			=> $payload,
						'cookies' 		=> array()
					);

			return wp_remote_post( $url, $contents );

		}

        /**
         * [sagepay_debug description]
         * @param  Array   $tolog   contents for log
         * @param  String  $id      payment gateway ID
         * @param  String  $message additional message for log
         * @param  boolean $start   is this the first log entry for this transaction
         */
        public static function sage_debug( $tolog = NULL, $id, $message = NULL, $start = FALSE ) {

        	$logger 	 = new stdClass();
            $logger->log = new WC_Logger();

            /**
             * If this is the start of the logging for this transaction add the header
             */
            if( $start ) {

                $logger->log->add( $id, __('=============================================', 'woocommerce-spusa') );
                $logger->log->add( $id, __(' ', 'woocommerce-spusa') );
                $logger->log->add( $id, __('Sage Log', 'woocommerce-spusa') );
                $logger->log->add( $id, __('' .date('d M Y, H:i:s'), 'woocommerce-spusa') );
                $logger->log->add( $id, __(' ', 'woocommerce-spusa') );

            }

            /**
             * Make sure we mask the card number
             */
            if( isset( $tolog["C_cardnumber"] ) && $tolog["C_cardnumber"] != '' ) {
                $tolog["C_cardnumber"] = substr( $tolog["C_cardnumber"], 0, 4 ) . str_repeat( "*", strlen($tolog["C_cardnumber"]) - 8 ) . substr( $tolog["C_cardnumber"], -4 );
            }

            $logger->log->add( $id, __('=============================================', 'woocommerce-spusa') );
            $logger->log->add( $id, $message );
            $logger->log->add( $id, print_r( $tolog, TRUE ) );
            $logger->log->add( $id, __('=============================================', 'woocommerce-spusa') );

        }

        /**
         * [getHmac description]
         * @param  [type] $toBeHashed [description]
         * @param  [type] $privateKey [description]
         * @return [type]             [description]
         */
        function getHmac($toBeHashed, $privateKey){
   
	        $hmac = hash_hmac(
	            "sha512", 		// use the SHA-512 algorithm...
	            $toBeHashed, 	// ... to hash the combined string...
	            $privateKey, 	// .. using your private dev key to sign it.
	            true 			// (php returns hexits by default; override this)
	        );

	        // convert to base-64 for transport
	        $hmac_b64 = base64_encode($hmac);
	        return $hmac_b64;
	    }

		/**
		 * [save_token description]
		 * @param  [type] $token        [description]
		 * @param  [type] $card_type    [description]
		 * @param  [type] $last4        [description]
		 * @param  [type] $expiry_month [description]
		 * @param  [type] $expiry_year  [description]
		 * @return [type]               [description]
		 */
		function save_token( $token_value, $card_number, $expiry_month, $expiry_year ) {
					
			$token = new WC_Payment_Token_CC();

			$token->set_token( $token_value );
			$token->set_gateway_id( $this->id );
			$token->set_card_type( $this->get_card_brand( $card_number ) );
			$token->set_last4( substr( $card_number, -4 ) );
			$token->set_expiry_month( $expiry_month );
			$token->set_expiry_year( 2000+$expiry_year );
			$token->set_user_id( get_current_user_id() );

			$token->save();

		}

	    /**
	     * [get_card_brand description]
	     * @param  [type] $cardnumber [description]
	     * @return [type]             [description]
	     */
	    function get_card_brand_old( $cardnumber ) {

	    	$cardnumber = preg_replace( '/\D/', '', $cardnumber );

	    	if ( version_compare( $environment['php_version'], '7.0', '<' ) ) {

	    		require( SPUSAPLUGINPATH . 'assets/php/carddetect.php' );

	    		$detector = new CardDetect\Detector();
				return $detector->detect( $cardnumber );

	    	} else {

		    	$bin = substr( $cardnumber, 0, 8 );

		    	// BIN List check
	    		$url = 'https://lookup.binlist.net/' . $bin;
				$res = wp_remote_get( $url );
	 
				if ( ! is_wp_error( $res ) ) {

					// Successful BIN list check.
					$result = json_decode( wp_remote_retrieve_body( $res ) );
					$brand  = $result->brand;
					$scheme = $result->scheme;

					return $result->brand ? $result->brand : $result->scheme;

				}

		    	return ' ';
		    }

	    }

		function get_card_brand($str, $format = 'string')
		    {
		        if (empty($str)) {
		            return false;
		        }

		        $matchingPatterns = [
		            'Visa' 			=> '/^4[0-9]{12}(?:[0-9]{3})?$/',
		            'Mastercard' 	=> '/^5[1-5][0-9]{14}$/',
		            'Amex' 			=> '/^3[47][0-9]{13}$/',
		            'Diners' 		=> '/^3(?:0[0-5]|[68][0-9])[0-9]{11}$/',
		            'Discover' 		=> '/^6(?:011|5[0-9]{2})[0-9]{12}$/',
		            'JCB' 			=> '/^(?:2131|1800|35\d{3})\d{11}$/',
		            'any' 			=> '/^(?:4[0-9]{12}(?:[0-9]{3})?|5[1-5][0-9]{14}|6(?:011|5[0-9][0-9])[0-9]{12}|3[47][0-9]{13}|3(?:0[0-5]|[68][0-9])[0-9]{11}|(?:2131|1800|35\d{3})\d{11})$/'
		        ];

		        $ctr = 1;
		        foreach ($matchingPatterns as $key=>$pattern) {
		            if (preg_match($pattern, $str)) {
		                return $format == 'string' ? $key : $ctr;
		            }
		            $ctr++;
		        }
		    }



		/**
		 * Use the $this->T_code from settings unless the order contains a pre-order or the order value is 0
		 *
		 * @param  {[type]} $order_id [description]
		 * @param  {[type]} $amount   [description]
		 * @return {[type]}           [description]
		 */
		function get_txtype( $order_id, $amount ) {

			if( class_exists( 'WC_Pre_Orders' ) && WC_Pre_Orders_Order::order_contains_pre_order( $order_id ) ) {
				return 'Authorization';
			} elseif( $amount == 0 ) {
				return 'Authorization';
			} else {
				return $this->T_code == '01'? "Sale" : "Authorization";
			}

		}

		/**
		 * Add payment method via account screen.
		 */
		public function add_payment_method() {

			if( is_user_logged_in() ) {     
			
				// Check WC version - changes for WC 3.0.0
				$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

				if ( $pre_wc_30 ) {
					$card_number 		= isset($_POST[$this->id . '-card-number']) ? woocommerce_clean($_POST[$this->id . '-card-number']) : '';
					$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? woocommerce_clean($_POST[$this->id . '-card-cvc']) : '';
					$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? woocommerce_clean($_POST[$this->id . '-card-expiry']) : '';
				} else {
					$card_number 		= isset($_POST[$this->id . '-card-number']) ? wc_clean($_POST[$this->id . '-card-number']) : '';
					$card_cvc 			= isset($_POST[$this->id . '-card-cvc']) ? wc_clean($_POST[$this->id . '-card-cvc']) : '';
					$card_expiry		= isset($_POST[$this->id . '-card-expiry']) ? wc_clean($_POST[$this->id . '-card-expiry']) : '';
				}

				// Format values
				$card_number    	= str_replace( array( ' ', '-' ), '', $card_number );
				$card_expiry    	= array_map( 'trim', explode( '/', $card_expiry ) );
				$card_exp_month 	= str_pad( $card_expiry[0], 2, "0", STR_PAD_LEFT );
				$card_exp_year  	= $card_expiry[1];

			    $requestData = [
			        "CardData" => [
		                "Number" 		=> $card_number,
		                "Expiration" 	=> $card_exp_month.$card_exp_year
			        ]
			    ];

			    // token url
			    $url 				= $this->apiendpoint . "bankcard/v1/tokens";
			    $result 			= $this->apirequest( $url, $requestData, false, "POST" );
			    $decoded_result 	= json_decode($result['body'], TRUE);
			    $vaultResponse 		= $decoded_result['vaultResponse'];

			    // Set token_status to true is token is successfully created
			    if( $vaultResponse['data'] && $vaultResponse['status'] === 1 ) {

					$token_status 	= true;
					$token_value 	= $vaultResponse['data'];

					// Store the token for future use
					$this->save_token( $token_value, $card_number, $card_exp_month, $card_exp_year );

					return array(
						'result'   => 'success',
						'redirect' => wc_get_endpoint_url( 'payment-methods' ),
					);

				} else {
					wc_add_notice( __( 'There was a problem adding the card. ' . $result['body'], 'woocommerce-spusa' ), 'error' );
					return;
				}

			} else {
				wc_add_notice( __( 'There was a problem adding the card. Please make sure you are logged in.', 'woocommerce-spusa' ), 'error' );
				return;
			}

		}

        /**
         * [process_pre_order_payments description]
         * @return [type] [description]
         */
        function process_pre_order_release_payment( $order ) {

        	$preauth = FALSE;

        	// WooCommerce 3.0 compatibility  
            $order_id   = is_callable( array( $order, 'get_id' ) ) ? $order->get_id() : $order->id;

            // Check WC version - changes for WC 3.0.0
			$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

            $token_value 	= get_post_meta( $order_id, '_SageToken', TRUE );
            $transaction_id = $order->get_transaction_id();

            // Setup variables for the transaction.
			$order_total 			= $pre_wc_30 ? $order->order_total : $order->get_total();
			$total_shipping 		= $pre_wc_30 ? $order->get_total_shipping() : $order->get_shipping_total();

            // Token order
            if( isset( $token_value ) && $token_value != '' && empty($transaction_id) ) {

			// Setup variables for the transaction.
			$order_variables = $this->get_order_variables( $order );

			// Set up the request for Sage
		    $requestData = [
		        "Ecommerce" => [
		            "OrderNumber" => $order->get_order_number(),
		            "Amounts" 	=> [
		                "Total" 		=> $order_variables['order_total'],
		                "tax" 			=> $order->get_total_tax(),
		  				"shipping" 		=> $order_variables['total_shipping'],
		            ],
		            "customer" 	=> [
		  				"email" 		=>  $order_variables['billing_email'],
		  				"telephone" 	=>  preg_replace( '/\D/', '', $order_variables['billing_phone'] ),
					],
			        "billing" 	=> [
					    "name" 			=>  $order_variables['billing_first_name'] . ' ' . $order_variables['billing_last_name'],
					    "address" 		=>  $order_variables['billing_address_1'],
					    "city" 			=>  $order_variables['billing_city'],
					    "state" 		=>  $order_variables['billing_state'],
					    "postalCode" 	=>  $order_variables['billing_postcode'],
					    "country" 		=>  $order_variables['billing_country']
			    	],
			    	"shipping" 	=> [
			      		"name" 			=>  $order_variables['shipping_first_name'] . ' ' . $order_variables['shipping_last_name'],
			      		"address" 		=>  $order_variables['shipping_address_1'],
			      		"city" 			=>  $order_variables['shipping_city'],
			      		"state" 		=>  $order_variables['shipping_state'],
			      		"postalCode" 	=>  $order_variables['shipping_postcode'],
			      		"country" 		=>  $order_variables['shipping_country']
			    	],
		        ],
		        "Vault" 	=> [
        			"Token" 		=> $token_value,
        			"Operation" 	=> "Read" // CRUD
    			],
		    ];

			    $requestData = apply_filters( 'woocommerce_sagepaymentsapi_version1api_data', $requestData );

			    // sale url
			    $url 	= $this->apiendpoint . "bankcard/v1/charges";
			    $result = $this->apirequest( $url, $requestData, "Sale", "POST" );

			} else {
				// pre-authorized transaction
				$preauth = TRUE;

			    $requestData = [
		            "Amounts" 	=> [
		                "Total" 		=> $order_total,
		                "tax" 			=> $order->get_total_tax(),
		  				"shipping" 		=> $total_shipping,
		            ],
			    ];
				$url 	= $this->apiendpoint . "bankcard/v1/charges/" . $transaction_id; 
				$result = $this->apirequest( $url, $requestData, NULL, "PUT" );

			}


			if( is_wp_error( $result ) ) {

				$order->add_order_note( $result->get_error_message() );

			} else {

				// Check if this is a token transaction
				if( isset( $token_value ) && $token_value != '' && !$preauth ) {
					$response 	= json_decode($result['body'], TRUE);
				} else {
					$response 	= $result['response'];
				}

				// Token transaction
				if( $response['status'] === 'Approved' ) {

					// Set the order to Processing / Completed
					if ( !class_exists( 'WC_Pre_Orders_Order' ) || (class_exists( 'WC_Pre_Orders' ) && !WC_Pre_Orders_Order::order_contains_pre_order( $order_id )) ) {
						// Successfully capture authorized token payment
						$order->set_status( ($order->needs_processing() ? 'processing' : 'completed'), __('Payment completed', 'woocommerce-spusa') . '<br />Approval Msg: ' . $response['message'] . '<br />Reference: ' . $response['reference'] . '<br />' );
						$order->save();
					} else {
						// Successful Pre-Order Capture
						$order->add_order_note( __('Payment completed', 'woocommerce-spusa') . '<br />
													Status: ' 			. $response['status'] . '<br />
													Approval Msg: ' 	. $response['message'] .'<br />
													Reference: ' 		. $response['reference'] );

						$order->payment_complete( $response['reference'] );
					}

				} elseif ( $response['code'] === 200 && $response['message'] === 'OK' ) {
					// Pre-authed transaction that does not use a token
					// Set the order to Processing / Completed
					$order->set_status( ($order->needs_processing() ? 'processing' : 'completed'), __('Payment completed', 'woocommerce-spusa') . '<br />Approval Msg: ' . $response['message'] . '<br />' );
					$order->save();

				} else {

					$cancelNote = __('Payment failed<br />', 'woocommerce-spusa') . 
									__('Payment was rejected due to an error', 'woocommerce-spusa') . 
									': "' . $response['message'] . '". ';
					$order->add_order_note( $cancelNote );

				}

			}

        }

		public function get_order_variables( $order ) {

			// Check WC version - changes for WC 3.0.0
			$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

			$order_variables = array();

			// Setup variables for the transaction.
			$order_variables['order_total'] 		= $pre_wc_30 ? $order->order_total : $order->get_total();
			$order_variables['total_shipping'] 		= $pre_wc_30 ? $order->get_total_shipping() : $order->get_shipping_total();

			$order_variables['billing_first_name'] 	= $pre_wc_30 ? $order->billing_first_name : $order->get_billing_first_name();
			$order_variables['billing_last_name'] 	= $pre_wc_30 ? $order->billing_last_name : $order->get_billing_last_name();
			$order_variables['billing_address_1'] 	= $pre_wc_30 ? $order->billing_address_1 : $order->get_billing_address_1();
			$order_variables['billing_address_2'] 	= $pre_wc_30 ? $order->billing_address_2 : $order->get_billing_address_2();
			$order_variables['billing_city'] 		= $pre_wc_30 ? $order->billing_city : $order->get_billing_city();
			$order_variables['billing_state'] 		= $pre_wc_30 ? $order->billing_state : $order->get_billing_state();
			$order_variables['billing_postcode'] 	= $pre_wc_30 ? $order->billing_postcode : $order->get_billing_postcode();
			$order_variables['billing_country'] 	= $pre_wc_30 ? $order->billing_country : $order->get_billing_country();
			$order_variables['billing_email'] 		= $pre_wc_30 ? $order->billing_email : $order->get_billing_email();
			$order_variables['billing_phone'] 		= $pre_wc_30 ? $order->billing_phone : $order->get_billing_phone();

			$order_variables['shipping_first_name'] = $pre_wc_30 ? $order->shipping_first_name : $order->get_shipping_first_name();
			$order_variables['shipping_last_name'] 	= $pre_wc_30 ? $order->shipping_last_name : $order->get_shipping_last_name();
			$order_variables['shipping_address_1'] 	= $pre_wc_30 ? $order->shipping_address_1 : $order->get_shipping_address_1();
			$order_variables['shipping_address_2'] 	= $pre_wc_30 ? $order->shipping_address_2 : $order->get_shipping_address_2();
			$order_variables['shipping_city'] 		= $pre_wc_30 ? $order->shipping_city : $order->get_shipping_city();
			$order_variables['shipping_state'] 		= $pre_wc_30 ? $order->shipping_state : $order->get_shipping_state();
			$order_variables['shipping_postcode'] 	= $pre_wc_30 ? $order->shipping_postcode : $order->get_shipping_postcode();
			$order_variables['shipping_country'] 	= $pre_wc_30 ? $order->shipping_country : $order->get_shipping_country();

			// Sort out the billing state field.
			if ( !$order_variables['billing_state'] || $order_variables['billing_state'] == '' ) {
				$order_variables['billing_state'] = $order_variables['billing_city'];
			}

			// Sort out the shipping state field.
			if ( !$order_variables['shipping_state'] || $order_variables['shipping_state'] == '' ) {
				$order_variables['shipping_state'] = $order_variables['shipping_city'];
			}

			return $order_variables;

		}

		// Modified Save Payment Method Checkbox
		public function save_payment_method_checkbox() {
			
			if ( $this->forcetokens ) {

				printf(
					'<p class="form-row woocommerce-SavedPaymentMethods-saveNew">
						<input id="wc-%1$s-new-payment-method" name="wc-%1$s-new-payment-method" type="hidden" value="true" style="width:auto;" />
					</p>',
					esc_attr( $this->id )
				);

			} else {

				printf(
					'<p class="form-row woocommerce-SavedPaymentMethods-saveNew">
						<input id="wc-%1$s-new-payment-method" name="wc-%1$s-new-payment-method" type="checkbox" value="true" style="width:auto;" />
						<label for="wc-%1$s-new-payment-method" style="display:inline;">%2$s</label>
					</p>',
					esc_attr( $this->id ),
					esc_html__( 'Save to account', 'woocommerce' )
				);

			}

		}

	} // END CLASS