<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * The base USA ePay API class.
 *
 * @since 2.0.0
 */
class API extends Framework\SV_WC_API_Base implements Framework\SV_WC_Payment_Gateway_API {


	/** the default production domain */
	const PRODUCTION_DOMAIN = 'usaepay.com';

	/** the test endpoint domain */
	const TEST_DOMAIN = 'sandbox.usaepay.com';


	/** @var string API key */
	protected $api_key;

	/** @var string pin */
	protected $pin;

	/** @var string  */
	protected $api_endpoint_key;

	/** @var string payment gateway ID */
	protected $gateway_id;

	/** @var \WC_Order WooCommerce order object */
	protected $order;


	/**
	 * Constructs the class.
	 *
	 * @since 2.0.0
	 *
	 * @param string $gateway_id payment gateway ID calling this API
	 * @param string $environment current API environment, either `production` or `test`
	 * @param string $api_key API key
	 * @param string $pin API pin
	 */
	public function __construct( $gateway_id, $environment, $api_key, $pin ) {

		$this->gateway_id = $gateway_id;
		$this->api_key    = $api_key;
		$this->pin        = $pin;

		$domain   = $this->get_api_domain( $environment );
		$endpoint = $this->get_api_endpoint_key( $environment );

		$this->request_uri = "https://{$domain}/api/{$endpoint}/";

		$this->set_request_content_type_header( 'application/json' );
		$this->set_request_accept_header( 'application/json' );
	}


	/**
	 * Gets the base API domain.
	 *
	 * @since 2.0.0
	 *
	 * @param string $environment current API environment, either `production` or `test`
	 * @return string API domain
	 */
	protected function get_api_domain( $environment ) {

		$domain = self::TEST_DOMAIN;

		if ( 'production' === $environment ) {

			/**
			 * Filters the API domain to allow for alternate high-availability domains.
			 *
			 * Normally, this should not be overridden since USA ePay automatically load-balances
			 * their primary domain, but there may be some instances in which a merchant needs to
			 * run all their transactions through a specific server, which can be done with this filter.
			 *
			 * @see https://help.usaepay.info/developer/reference/highavailability/
			 *
			 * @since 2.0.0
			 *
			 * @param string $domain the API domain
			 */
			$domain = apply_filters( 'wc_usa_epay_api_production_domain', self::PRODUCTION_DOMAIN );
		}

		return $domain;
	}


	/**
	 * Gets the API endpoint key.
	 *
	 * @since 2.0.0
	 *
	 * @param string $environment current API environment, either `production` or `test`
	 * @return string API endpoint
	 */
	protected function get_api_endpoint_key( $environment ) {

		/**
		 * Filters the endpoint key of the API URL.
		 *
		 * The default `v2` endpoint key will work for everyone, but is throttled.
		 * High-volume merchants will want to create a new endpoint key and filter it here.
		 *
		 * @see https://help.usaepay.info/developer/rest-api/#api-rate-limits
		 *
		 * @since 2.0.0
		 *
		 * @param string $endpoint_key the API endpoint key
		 * @param string $environment the current API environment, either `production` or `test`
		 */
		return apply_filters( 'wc_usa_epay_api_endpoint', 'v2', $environment );
	}


	/**
	 * Performs a credit card charge.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order WooCommerce order object
	 * @return API\Response
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function credit_card_charge( \WC_Order $order ) {

		$this->order = $order;

		$request = $this->get_new_request( 'sale' );

		$request->set_charge_data( $order );
		$request->set_send_receipt( wc_usa_epay()->get_gateway( $this->gateway_id )->should_send_receipt_email() );

		return $this->perform_request( $request );
	}


	/**
	 * Performs a credit card authorization.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order WooCommerce order object
	 * @return API\Response
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function credit_card_authorization( \WC_Order $order ) {

		$this->order = $order;

		$request = $this->get_new_request( 'sale' );

		$request->set_authorization_data( $order );
		$request->set_send_receipt( wc_usa_epay()->get_gateway( $this->gateway_id )->should_send_receipt_email() );

		return $this->perform_request( $request );
	}


	/**
	 * Captures an authorized credit card payment.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return API\Response
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function credit_card_capture( \WC_Order $order ) {

		$this->order = $order;

		$request = $this->get_new_request( 'sale' );

		$request->set_capture_data( $order );

		return $this->perform_request( $request );
	}


	/**
	 * Refunds a transaction.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return API\Response
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function refund( \WC_Order $order ) {

		$this->order = $order;

		$request = $this->get_new_request( 'sale' );

		$request->set_refund_data( $order );

		return $this->perform_request( $request );
	}


	/**
	 * Voids a transaction.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return API\Response
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function void( \WC_Order $order ) {

		$this->order = $order;

		$request = $this->get_new_request( 'sale' );

		$request->set_void_data( $order );

		return $this->perform_request( $request );
	}


	/** Tokenization methods **************************************************/


	/**
	 * Determines if the API supports getting tokenized payment methods.
	 *
	 * @see Framework\SV_WC_Payment_Gateway_API::supports_get_tokenized_payment_methods()
	 *
	 * @since 2.0.0
	 *
	 * @return false
	 */
	public function supports_get_tokenized_payment_methods() {

		return false;
	}


	/**
	 * Determines if the API supports updating tokenized payment methods.
	 *
	 * @see Framework\SV_WC_Payment_Gateway_API::supports_update_tokenized_payment_method()
	 *
	 * @since 2.0.2
	 *
	 * @return false
	 */
	public function supports_update_tokenized_payment_method() {

		return false;
	}


	/**
	 * Determines if the API supports removing tokenized payment methods.
	 *
	 * @see Framework\SV_WC_Payment_Gateway_API::supports_remove_tokenized_payment_method()
	 *
	 * @since 2.0.0
	 *
	 * @return false
	 */
	public function supports_remove_tokenized_payment_method() {

		return false;
	}


	/** Validation methods ****************************************************/


	/**
	 * Performs the API request and returns the parsed response.
	 *
	 * Overridden to add authenication headers and allow filtering the merchant ID.
	 *
	 * @since 2.0.0
	 *
	 * @param API\Request $request request object
	 * @return API\Response response object
	 * @throws Framework\SV_WC_API_Exception
	 */
	protected function perform_request( $request ) {

		/**
		 * Filters the USA ePay API Key.
		 *
		 * @since 2.0.0
		 *
		 * @param string $api_key API key
		 * @param \WC_Order|null $order order object
		 */
		$this->api_key = apply_filters( 'wc_usa_epay_api_request_api_key', $this->api_key, $this->get_order() );

		/**
		 * Filters the USA ePay API Pin.
		 *
		 * @since 2.0.0
		 *
		 * @param string $pin API pin
		 * @param \WC_Order|null $order order object
		 */
		$this->pin = apply_filters( 'wc_usa_epay_api_request_pin', $this->pin, $this->get_order() );

		$this->set_request_header( 'Authorization', 'Basic ' . $this->api_auth_key() );

		return parent::perform_request( $request );
	}


	/**
	 * Validates the response for errors before parsing the data.
	 *
	 * @see Framework\SV_WC_API_Base::do_pre_parse_response_validation()
	 *
	 * @since 2.0.0
	 *
	 * @return true
	 */
	protected function do_pre_parse_response_validation() {

		return true;
	}


	/**
	 * Validates the response for errors after parsing the data.
	 *
	 * @see Framework\SV_WC_API_Base::do_post_parse_response_validation()
	 *
	 * @since 2.0.0
	 *
	 * @return true
	 * @throws Framework\SV_WC_API_Exception
	 */
	protected function do_post_parse_response_validation() {

		$response_code = (int) $this->get_response_code();

		// if response code is not 200 or there is an error code
		if ( 300 <= $response_code || 0 < (int) $this->get_response()->get_error_code() ) {
			throw new Framework\SV_WC_API_Exception( $this->get_response()->get_status_message() );
		}

		return true;
	}


	/** Helper methods ********************************************************/


	/**
	 * Generates and returns an API auth key used in the authentication header.
	 *
	 * @since 2.0.0
	 *
	 * @return string API hash
	 */
	private function api_auth_key() {

		$seed     = bin2hex( openssl_random_pseudo_bytes( 8 ) );
		$prehash  = $this->api_key . $seed . $this->pin;
		$api_hash = 's2/' . $seed . '/' . hash( 'sha256', $prehash );

		return base64_encode( $this->api_key . ':' . $api_hash );
	}


	/**
	 * Builds and returns a new API request object.
	 *
	 * @since 2.0.0
	 *
	 * @param string $type request type to get
	 * @return API\Request
	 * @throws Framework\SV_WC_API_Exception
	 */
	protected function get_new_request( $type = '' ) {

		switch ( $type ) {

			case 'sale':
				$request_handler  = '\\SkyVerge\\WooCommerce\\USAePay\\API\\Request\\Sale';
				$response_handler = '\\SkyVerge\\WooCommerce\\USAePay\\API\\Response\\Sale';
				break;

			default:
				throw new Framework\SV_WC_API_Exception( 'Invalid request type' );
		}

		$this->set_response_handler( $response_handler );

		return new $request_handler;
	}


	/**
	 * Gets the order associated with the request.
	 *
	 * @since 2.0.0
	 *
	 * @return \WC_Order|null
	 */
	public function get_order() {

		return $this->order;
	}


	/**
	 * Gets the ID for the API.
	 *
	 * Used primarily to namespace the action name for broadcasting requests.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	protected function get_api_id() {

		return $this->gateway_id;
	}


	/**
	 * Gets the main plugin instance.
	 *
	 * @see Framework\SV_WC_API_Base::get_plugin()
	 *
	 * @since 2.0.0
	 *
	 * @return Plugin
	 */
	protected function get_plugin() {

		return wc_usa_epay();
	}


	/* No-op methods **********************************************************/


	/**
	 * No-op: we do not currently support echeck transactions.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function check_debit( \WC_Order $order ) { }


	/**
	 * No-op: we do not currently support tokenization
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function tokenize_payment_method( \WC_Order $order ) { }


	/**
	 * No-op: we do not currently support tokenization
	 *
	 * @since 2.0.2
	 *
	 * @param \WC_Order $order order object
	 */
	public function update_tokenized_payment_method( \WC_Order $order ) { }


	/**
	 * No-op: we do not currently support tokenization
	 *
	 * @since 2.0.0
	 *
	 * @param string $token payment method token
	 * @param string $customer_id unique customer ID
	 */
	public function remove_tokenized_payment_method( $token, $customer_id ) { }


	/**
	 * No-op: we do not currently support tokenization
	 *
	 * @since 2.0.0
	 *
	 * @param string $customer_id unique customer ID
	 */
	public function get_tokenized_payment_methods( $customer_id ) { }


}
