<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\API;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay API base request class.
 *
 * @since 2.0.0
 */
class Request extends Framework\SV_WC_API_JSON_Request implements Framework\SV_WC_Payment_Gateway_API_Request {


	/**
	 * Gets the billing data from an order.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return array billing data
	 */
	protected function get_billing_data( \WC_Order $order ) {

		$data  = $this->get_address_data( $order, 'billing' );
		$phone = $order->get_billing_phone();

		$data['phone'] = Framework\SV_WC_Helper::str_truncate( preg_replace( '/[^0-9]/', '', $phone ), 20, '' ); // digits only

		return $data;
	}


	/**
	 * Gets the shipping data from an order.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return array shipping data
	 */
	protected function get_shipping_data( \WC_Order $order ) {

		return $this->get_address_data( $order, 'shipping' );
	}


	/**
	 * Gets address data from an order, formatted for the USA ePay API.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @param string $type address type, `billing` or `shipping`
	 * @return array address data
	 */
	protected function get_address_data( \WC_Order $order, $type = 'billing' ) {

		$address = [
			'firstname'  => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_first_name" ] ) ? $order->{"get_{$type}_first_name"}( 'edit' ) : '', 64 ),
			'lastname'   => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_last_name" ] )  ? $order->{"get_{$type}_last_name"}( 'edit' )  : '', 64 ),
			'street'     => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_address_1" ] )  ? $order->{"get_{$type}_address_1"}( 'edit' )  : '', 64 ),
			'street2'    => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_address_2" ] )  ? $order->{"get_{$type}_address_2"}( 'edit' )  : '', 64 ),
			'city'       => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_city" ] )       ? $order->{"get_{$type}_city"}( 'edit' )       : '', 32 ),
			'state'      => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_state" ] )      ? $order->{"get_{$type}_state"}( 'edit' )      : '', 2  ),
			'country'    => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_country" ] )    ? $order->{"get_{$type}_country"}( 'edit' )    : '', 3  ),
			'postalcode' => Framework\SV_WC_Helper::str_truncate( is_callable( [ $order, "get_{$type}_postcode" ] )   ? $order->{"get_{$type}_postcode"}( 'edit' )   : '', 16 ),
		];

		return array_filter( $address );
	}


	/**
	 * Gets payment data from an order, formatted for the USA ePay API.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 * @return array payment data
	 */
	protected function get_payment_data( \WC_Order $order ) {

		$payment_data = array(
			'cardholder' => $order->get_formatted_billing_full_name(),
			'number'     => $order->payment->account_number,
			'expiration' => $order->payment->exp_month . $order->payment->exp_year,
			'avs_street' => $order->get_billing_address_1( 'edit' ),
			'avs_zip'    => $order->get_billing_postcode( 'edit' ),
		);

		if ( isset( $order->payment->csc ) ) {
			$payment_data['cvc'] = $order->payment->csc;
		}

		return $payment_data;
	}


	/**
	 * Gets the request data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_data() {

		/**
		 * Filters the USA ePay API request data.
		 *
		 * @since 2.0.0
		 *
		 * @param array $data request data
		 */
		$this->data = (array) apply_filters( 'wc_usa_epay_credit_card_api_request_data', $this->data );

		return $this->data;
	}


}
