<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\API\Request;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;
use SkyVerge\WooCommerce\USAePay\API\Request;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay Sale request class.
 *
 * @since 2.0.0
 */
class Sale extends Request {


	/** @var string API URL path */
	protected $path = 'transactions';

	/** @var string the sale command being executed */
	protected $command;


	/**
	 * Sets the credit card charge transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_charge_data( \WC_Order $order ) {

		$this->command = 'cc:sale';

		$this->set_payment_data( $order );
	}


	/**
	 * Sets the credit card authorization transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_authorization_data( \WC_Order $order ) {

		$this->command = 'cc:authonly';

		$this->set_payment_data( $order );
	}


	/**
	 * Sets the credit card capture transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_capture_data( \WC_Order $order ) {

		$this->command = 'cc:capture';

		$this->set_data( array(
			'refnum' => $order->capture->trans_id,
			'amount' => $order->capture->amount,
		) );
	}


	/**
	 * Sets the refund transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_refund_data( \WC_Order $order ) {

		$this->command = 'cc:refund';

		$this->set_data( array(
			'refnum' => $order->refund->trans_id,
			'amount' => $order->refund->amount,
		) );
	}


	/**
	 * Sets the void transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_void_data( \WC_Order $order ) {

		$this->command = 'cc:void';

		$this->set_data( array(
			'refnum' => $order->refund->trans_id,
			'amount' => $order->refund->amount,
		) );
	}


	/**
	 * Sets the necessary transaction data for a new payment.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	protected function set_payment_data( \WC_Order $order ) {

		$data = [
			'orderid'          => Framework\SV_WC_Helper::str_truncate( $order->get_order_number(), 30, '' ),
			'amount'           => $order->payment_total,
			'amount_detail'    => [
				'tax'      => $order->get_total_tax(),
				'shipping' => $order->get_shipping_total(),
				'discount' => $order->get_total_discount(),
			],
			'creditcard'       => $this->get_payment_data( $order ),
			'billing_address'  => $this->get_billing_data( $order ),
			'shipping_address' => $this->get_shipping_data( $order ),
			'email'            => Framework\SV_WC_Helper::str_truncate( $order->get_billing_email( 'edit' ), 32 ),
			'clientip'         => Framework\SV_WC_Helper::str_truncate( $order->get_customer_ip_address( 'edit' ), 30, '' ),
		];

		$this->set_data( $data );
	}


	/**
	 * Sets data for the current request.
	 *
	 * @since 2.0.0
	 *
	 * @param array|null $data request data
	 */
	protected function set_data( $data ) {

		foreach ( $data as $key => $value ) {

			if ( empty( $value ) && false !== $value ) {
				unset( $data[ $key ] );
			}
		}

		if ( null !== $this->command ) {
			$data['command'] = $this->command;
		}

		$this->data = $data;
	}


	/**
	 * Sets whether USA ePay should send a receipt email or not.
	 *
	 * @since 2.0.0
	 *
	 * @param bool $send_receipt whether USA ePay should send a receipt email
	 */
	public function set_send_receipt( $send_receipt ) {

		// $this->data['send_receipt'] = (bool) $send_receipt;

		// TODO: This is a temporary workaround for an issue with the USA ePay REST API. The API does not currently
		// respect the `send_receipt` param listed in documentation, so the only way to control whether a receipt
		// email is sent is by adding or removing the `email` param from the the transaction data. {JB 2018-08-19}

		if ( ! $send_receipt ) {
			unset( $this->data['email'] );
		}
	}


	/**
	 * Masks credit card details for logging.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		$string = parent::to_string_safe();

		// mask credit card number
		if ( preg_match( '/"number":"(\d+)"/', $string, $matches ) && strlen( $matches[1]	) > 4 ) {

			$string = preg_replace( '/"number":"\d+"/', '"number":"' . substr( $matches[1], 0, 1 ) . str_repeat( '*', strlen( $matches[1] ) - 5 ) . substr( $matches[1], -4 ) .'"', $string );
		}

		// mask cvc
		$string = preg_replace( '/"cvc":"\d+"/', '"cvc":"***"', $string );

		return $string;
	}


}
