<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\API\Response;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;
use SkyVerge\WooCommerce\USAePay\API\Response;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay Sale response class.
 *
 * @since 2.0.0
 */
class Sale extends Response implements Framework\SV_WC_Payment_Gateway_API_Authorization_Response {


	/**
	 * Determines if the transaction was approved.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function transaction_approved() {

		return 'A' === $this->result_code;
	}


	/**
	 * Determines if the transaction was held.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function transaction_held() {

		// USA ePay doesn't have this concept
		return false;
	}


	/**
	 * Gets the transaction authorization code.
	 *
	 * @since 2.0.0
	 *
	 * @return string|null
	 */
	public function get_authorization_code() {

		return $this->authcode;
	}


	/**
	 * Gets the transaction ID.
	 *
	 * @since 2.0.0
	 *
	 * @return string|null
	 */
	public function get_transaction_id() {

		return $this->refnum;
	}


	/**
	 * Determines if the AVS was a match.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function avs_match() {

		$matching_codes = array(
			'YYY', // address: match & 5 digit zip: match
			'Y',   // `YYY` alternate
			'YYA', // `YYY` alternate
			'YYD', // `YYY` alternate
			'YYX', // address: match & 9 digit zip: match
			'X',   // `YYX` alternate
			'GGG', // international address: match & zip: match
			'D',   // `GGG` alternate
		);

		return isset( $this->avs->result_code ) && in_array( $this->avs->result_code, $matching_codes, true );
	}


	/**
	 * Determines if AVS was processed.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function avs_processed() {

		return isset( $this->avs );
	}


	/**
	 * Gets the transaction AVS result code.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_avs_result() {

		return isset( $this->avs->result_code ) ? $this->avs->result_code : '';
	}


	/**
	 * Gets the transaction CSC validation result.
	 *
	 * @since 2.0.0
	 *
	 * @return string|null
	 */
	public function get_csc_result() {

		return isset( $this->cvc->result_code ) ? $this->cvc->result_code : null;
	}


	/**
	 * Determines if the CSC was a match.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function csc_match() {

		return isset( $this->cvc->result_code ) && 'M' === $this->cvc->result_code;
	}


	/**
	 * Gets the transaction status code.
	 *
	 * @since 2.0.0
	 *
	 * @return string|null
	 */
	public function get_status_code() {

		if ( null !== $this->get_error_code() ) {

			return $this->get_error_code();
		}

		return $this->result_code;
	}

	/**
	 * Gets the payment type.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_payment_type() {

		return Framework\SV_WC_Payment_Gateway::PAYMENT_TYPE_CREDIT_CARD;
	}


	/**
	 * Gets a customer-friendly error message if available.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_user_message() {

		$message = '';

		if ( isset( $this->error_code ) ) {

			$helper  = new \SkyVerge\WooCommerce\USAePay\API\Message_Helper();
			$message = $helper->get_message( $this->get_error_code() );
		}

		return $message;
	}

}
