<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\Legacy\API;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;
use SkyVerge\WooCommerce\USAePay\Plugin;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay API base request class.
 *
 * @since 2.0.0
 */
class Request implements Framework\SV_WC_Payment_Gateway_API_Request {


	/** @var string request method */
	protected $method = 'POST';

	/** @var array request parameters */
	protected $params = array();

	/** @var array request data */
	protected $data = array();

	/** @var string API key */
	protected $api_key;

	/** @var string API pin */
	protected $api_pin;

	/** @var string the sale command being executed */
	protected $command;


	/**
	 * Constructs the Request class.
	 *
	 * @since 2.0.0
	 *
	 * @param string $api_key USA ePay API key (called a source key in the legacy system)
	 * @param string $api_pin (optional) API pin
	 */
	public function __construct( $api_key, $api_pin = '' ) {

		$this->api_key = $api_key;
		$this->api_pin = $api_pin;
	}


	/**
	 * Sets the credit card charge transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_charge_data( \WC_Order $order ) {

		$this->command = 'sale';

		$this->set_payment_data( $order );
	}


	/**
	 * Sets the credit card authorization transaction data.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function set_authorization_data( \WC_Order $order ) {

		$this->command = 'authonly';

		$this->set_payment_data( $order );
	}


	/**
	 * Sets the necessary transaction data for a new payment.
	 *
	 * @since 2.0.0
	 *
	 * @param \WC_Order $order order object
	 */
	protected function set_payment_data( \WC_Order $order ) {

		$command = $this->command;
		$amount  = $order->payment_total;
		$invoice = $order->get_order_number();

		$data = array(
			'UMkey'         => $this->api_key,
			'UMcommand'     => $command,
			'UMcard'        => $order->payment->account_number,
			'UMexpir'       => $order->payment->exp_month . $order->payment->exp_year,
			'UMamount'      => $amount,
			'UMinvoice'     => $invoice,
			'UMname'        => $order->get_formatted_billing_full_name(),
			'UMzip'         => $order->get_billing_postcode( 'edit' ),
			'UMdescription' => wc_usa_epay()->get_gateway( Plugin::LEGACY_GATEWAY_ID )->get_transaction_description(),
			'UMip'          => $order->get_customer_ip_address( 'edit' ),
			'UMtestmode'    => 0,
			'UMbillfname'   => $order->get_billing_first_name( 'edit' ),
			'UMbilllname'   => $order->get_billing_last_name( 'edit' ),
			'UMbillcompany' => $order->get_billing_company( 'edit' ),
			'UMbillstreet'  => $order->get_billing_address_1( 'edit' ),
			'UMbillstreet2' => $order->get_billing_address_2( 'edit' ),
			'UMbillcity'    => $order->get_billing_city( 'edit' ),
			'UMbillstate'   => $order->get_billing_state( 'edit' ),
			'UMbillzip'     => $order->get_billing_postcode( 'edit' ),
			'UMbillcountry' => $order->get_billing_country( 'edit' ),
			'UMbillphone'   => $order->get_billing_phone( 'edit' ),
			'UMemail'       => $order->get_billing_email( 'edit' ),
			'UMshipfname'   => $order->get_shipping_first_name( 'edit' ),
			'UMshiplname'   => $order->get_shipping_last_name( 'edit' ),
			'UMshipcompany' => $order->get_shipping_company( 'edit' ),
			'UMshipstreet'  => $order->get_shipping_address_1( 'edit' ),
			'UMshipstreet2' => $order->get_shipping_address_2( 'edit' ),
			'UMshipcity'    => $order->get_shipping_city( 'edit' ),
			'UMshipstate'   => $order->get_shipping_state( 'edit' ),
			'UMshipzip'     => $order->get_shipping_postcode( 'edit' ),
			'UMshipcountry' => $order->get_shipping_country( 'edit' ),
			'UMsoftware'    => wc_usa_epay()->get_plugin_name() . ' Legacy v' . Plugin::VERSION,
			'UMtimeout'     => '45',
		);

		// add CVV data if present
		if ( isset( $order->payment->csc ) && $order->payment->csc ) {

			$data['UMcvv2'] = $order->payment->csc;
		}

		// add an authentication hash if an API pin is present
		if ( '' !== $this->api_pin )  {

			$seed    = microtime(true) . rand();
			$prehash = "{$this->command}:{$this->api_pin}:{$amount}:{$invoice}:{$seed}";

			// if sha1 is available,  create sha1 hash,  else use md5
			if ( function_exists('sha1' ) ) {

				$hash = 's/' . $seed . '/' . sha1( $prehash ) . '/n';

			} else {

				$hash = 'm/' . $seed . '/' . md5( $prehash ) . '/n';
			}

			$data['UMhash'] = $hash;
		}

		$this->data = $data;
	}


	/**
	 * Sets whether a customer receipt should be sent by USA ePay or not.
	 *
	 * @since 2.0.0
	 *
	 * @param bool $send_receipt whether to send a customer receipt
	 */
	public function set_send_receipt( $send_receipt ) {

		$this->data['UMcustreceipt'] = $send_receipt ? 'yes' : 0;
	}


	/**
	 * Returns the method for this request.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_method() {

		return $this->method;
	}


	/**
	 * Gets the request path.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_path() {

		return '';
	}


	/**
	 * Gets the request query params.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_params() {

		return $this->params;
	}


	/**
	 * Gets the string representation of the request data.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string() {

		return http_build_query( $this->get_data(), '', '&' );
	}


	/**
	 * Gets the string representation of the request data with all sensitive
	 * information masked or removed.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		$string = $this->to_string();

		// mask the API key
		if ( ! empty( $this->data['UMkey'] ) ) {

			$string = str_replace( $this->data['UMkey'], str_repeat( '*', strlen( $this->data['UMkey'] ) ), $string );
		}

		// mask the card number
		if ( ! empty( $this->data['UMcard'] ) ) {

			$string = str_replace( $this->data['UMcard'], str_repeat( '*', strlen( $this->data['UMcard'] ) ), $string );
		}

		// mask the card CVV2
		if ( ! empty( $this->data['UMcvv2'] ) ) {

			$string = str_replace( $this->data['UMcvv2'], str_repeat( '*', strlen( $this->data['UMcvv2'] ) ), $string );
		}

		return $string;
	}


	/**
	 * Gets the request data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_data() {

		/**
		 * Filters the USA ePay Legacy API request data.
		 *
		 * @since 2.0.0
		 *
		 * @param array $data request data
		 */
		$this->data = (array) apply_filters( 'wc_usa_epay_legacy_credit_card_api_request_data', $this->data );

		return $this->data;
	}


}
