<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\Legacy\API;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;
use SkyVerge\WooCommerce\USAePay\API\Message_Helper;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay API base response class.
 *
 * @since 2.0.0
 */
class Response implements Framework\SV_WC_Payment_Gateway_API_Authorization_Response {


	/** @var array response data */
	protected $data = array();


	/**
	 * Constructs the class.
	 *
	 * @since 2.0.0
	 *
	 * @param string $raw_data URL query string with the response data
	 */
	public function __construct( $raw_data ) {

		parse_str( $raw_data, $data );

		$this->data = $data;
	}


	/**
	 * Determines if the transaction was approved.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function transaction_approved() {

		return 'A' === $this->get_status_code();
	}


	/**
	 * Determines if the transaction was held.
	 *
	 * @since 2.0.0
	 *
	 * @return false
	 */
	public function transaction_held() {

		return false; // the legacy API doesn't hold orders
	}


	/**
	 * Gets the transaction result code.
	 *
	 * @since 2.0.0
	 *
	 * @return null
	 */
	public function get_status_code() {

		return isset( $this->data['UMresult'] ) ? $this->data['UMresult'] : '';}


	/**
	 * Gets the transaction result message.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_status_message() {

		return isset( $this->data['UMstatus'] ) ? $this->data['UMstatus'] : '';
	}


	/**
	 * Gets the transaction ID.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_transaction_id() {

		return isset( $this->data['UMrefNum'] ) ? $this->data['UMrefNum'] : '';
	}


	/**
	 * Gets the transaction authorization code.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_authorization_code() {

		return isset( $this->data['UMauthCode'] ) ? $this->data['UMauthCode'] : '';
	}


	/**
	 * Gets the AVS result code.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_avs_result() {

		return isset( $this->data['UMavsResultCode'] ) ? $this->data['UMavsResultCode'] : '';
	}


	/**
	 * Gets the CSC validation result code
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_csc_result() {

		return isset( $this->data['UMcvv2ResultCode'] ) ? $this->data['UMcvv2ResultCode'] : '';
	}


	/**
	 * Gets the payment type: 'credit-card', 'echeck', etc...
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_payment_type() {

		return Framework\SV_WC_Payment_Gateway::PAYMENT_TYPE_CREDIT_CARD;
	}


	/**
	 * Gets the user-friendly message for detailed declined messages.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_user_message() {

		$helper = new Message_Helper();

		return $helper->get_message( $this->get_error_code() );
	}


	/**
	 * Gets the error code.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_error_code() {

		return isset( $this->data['UMerror'] ) ? $this->data['UMerror'] : '';
	}


	/**
	 * Gets the string representation of the request data.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string() {

		return print_r( $this->data, true );
	}


	/**
	 * Gets the string representation of the response with any sensitive data
	 * masked or removed.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		return $this->to_string();
	}


	/**
	 * Returns true if the CSC check was successful.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function csc_match() {

		return 'M' === $this->get_csc_result();
	}


}
