<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay\Legacy;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;
use SkyVerge\WooCommerce\USAePay\Plugin;

defined( 'ABSPATH' ) or exit;

/**
 * WooCommerce USA ePay Legacy Gateway class.
 *
 * @since 2.0.0
 */
class Gateway extends Framework\SV_WC_Payment_Gateway_Direct {

	/** @var string API Key */
	protected $api_key;

	/** @var string test API Key */
	protected $test_api_key;

	/** @var string Pin */
	protected $pin;

	/** @var string test Pin */
	protected $test_pin;

	/** @var string legacy gateway url override */
	protected $legacy_gateway_url;

	/** @var string whether or not a receipt email should be sent by USA ePay */
	protected $receipt_email;

	/** @var string transaction description */
	protected $transaction_description;

	/** @var API API handler instance */
	protected $api;


	/**
	 * Constructs the gateway.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		parent::__construct(
			Plugin::LEGACY_GATEWAY_ID,
			wc_usa_epay(),
			array(
				'method_title'       => __( 'USA ePay Legacy', 'woocommerce-gateway-usa-epay' ),
				'method_description' => __( 'USA ePay allows customers to checkout using a credit card', 'woocommerce-gateway-usa-epay' ),
				'supports'           => array(
					self::FEATURE_PRODUCTS,
					self::FEATURE_CARD_TYPES,
					self::FEATURE_PAYMENT_FORM,
					self::FEATURE_CREDIT_CARD_CHARGE,
					self::FEATURE_CREDIT_CARD_CHARGE_VIRTUAL,
					self::FEATURE_CREDIT_CARD_AUTHORIZATION,
					self::FEATURE_DETAILED_CUSTOMER_DECLINE_MESSAGES,
				),
				'environments' => array(
					self::ENVIRONMENT_PRODUCTION => esc_html_x( 'Production', 'software environment', 'woocommerce-gateway-usa-epay' ),
					self::ENVIRONMENT_TEST       => esc_html_x( 'Test', 'software environment', 'woocommerce-gateway-usa-epay' ),
				),
				'payment_type' => self::PAYMENT_TYPE_CREDIT_CARD,
			)
		);
	}


	/**
	 * Gets the gateway form fields.
	 *
	 * @see Framework\SV_WC_Payment_Gateway::get_method_form_fields()
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	protected function get_method_form_fields() {

		return array(
			'transaction_description' => array(
				'title'       => __( 'Transaction Description', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'textarea',
				'description' => __( 'This controls the description that is added to the transaction sent to USA ePay.', 'woocommerce-gateway-usa-epay' ),
				'default'     => 'Order from WooCommerce.'
			),
			'api_key' => array(
				'title'       => __( 'API Key', 'woocommerce-gateway-usa-epay' ),
				'description' => __( 'Your API Key, generated in the Merchant Console.', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'text',
				'class'       => 'environment-field production-field',
			),
			'test_api_key' => array(
				'title'       => __( 'API Key', 'woocommerce-gateway-usa-epay' ),
				'description' => __( 'Your API Key, generated in the Merchant Console.', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'text',
				'class'       => 'environment-field test-field',
			),
			'pin' => array(
				'title'       => __( 'PIN', 'woocommerce-gateway-usa-epay' ),
				'description' => __( 'Pin for API Key. This field is required only if the merchant has set a Pin in the merchant console.', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'text',
				'class'       => 'environment-field production-field',
			),
			'test_pin' => array(
				'title'       => __( 'PIN', 'woocommerce-gateway-usa-epay' ),
				'description' => __( 'Pin for API Key. This field is required only if the merchant has set a Pin in the merchant console.', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'text',
				'class'       => 'environment-field test-field',
			),
			'legacy_gateway_url' => array(
				'title'       => __( 'Gateway URL Override', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'text',
				'description' => __( 'Override for URL of USA ePay gateway processor. Optional, leave blank for default URL.', 'woocommerce-gateway-usa-epay' ),
				'default'     => '',
				'class'       => 'environment-field production-field',
			),
			'receipt_email' => array(
				'title'       => __( 'Receipt Email', 'woocommerce-gateway-usa-epay' ),
				'label'       => __( 'Send receipt email', 'woocommerce-gateway-usa-epay' ),
				'type'        => 'checkbox',
				'description' => __( 'If checked USA ePay will send a receipt email to the customer in addition to the WooCommerce order email.', 'woocommerce-gateway-usa-epay' ),
				'default'     => 'no',
			),
		);
	}


	/**
	 * Overrides the authorization charge form fields to remove the paid capture option.
	 *
	 * @since 2.0.0
	 *
	 * @param array $form_fields gateway form fields
	 * @return array $form_fields gateway form fields
	 */
	protected function add_authorization_charge_form_fields( $form_fields ) {

		$fields = parent::add_authorization_charge_form_fields( $form_fields );

		// 'Capture Paid Orders' not possible on the legacy gateway since it doesn't support capturing.
		unset( $fields['enable_paid_capture'] );

		return $fields;
	}


	/**
	 * Gets the payment form field defaults for the test environment.
	 *
	 * @see Framework\SV_WC_Payment_Gateway::get_payment_method_defaults()
	 *
	 * @since 2.0.0
	 *
	 * @return array payment method defaults
	 */
	public function get_payment_method_defaults() {

		$defaults = parent::get_payment_method_defaults();

		if ( $this->is_test_environment() ) {
			$defaults['account-number'] = '4000100011112224';
			$defaults['expiry']         = '09/' . ( date( 'y' ) + 1 );
		}

		return $defaults;
	}


	/**
	 * Gets the API handler.
	 *
	 * @since 2.0.0
	 *
	 * @return API
	 */
	public function get_api() {

		if ( null === $this->api ) {

			$this->api = new API( $this->get_id(), $this->get_environment(), $this->get_api_key(), $this->get_pin() );
		}

		return $this->api;
	}


	/**
	 * Determines if the gateway is properly configured to perform transactions.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function is_configured() {

		return (bool) $this->get_api_key();
	}


	/**
	 * Gets the API Key.
	 *
	 * @since 2.0.0
	 *
	 * @param string $environment_id gateway environment ID
	 * @return string
	 */
	public function get_api_key( $environment_id = null ) {

		if ( null === $environment_id ) {
			$environment_id = $this->get_environment();
		}

		return self::ENVIRONMENT_PRODUCTION === $environment_id ? $this->api_key : $this->test_api_key;
	}


	/**
	 * Gets the pin.
	 *
	 * @since 2.0.0
	 *
	 * @param string $environment_id gateway environment ID
	 * @return string
	 */
	public function get_pin( $environment_id = null ) {

		if ( null === $environment_id ) {
			$environment_id = $this->get_environment();
		}

		return self::ENVIRONMENT_PRODUCTION === $environment_id ? $this->pin : $this->test_pin;
	}


	/**
	 * Gets the overridden gateway URL.
	 *
	 * @since 2.0.0
	 *
	 * @return string gateway URL
	 */
	public function get_gateway_url_override() {

		return $this->legacy_gateway_url;
	}


	/**
	 * Returns whether USA ePay should send a receipt email or not.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function should_send_receipt_email() {

		return 'yes' === $this->receipt_email;
	}


	/**
	 * Gets the transaction description to send to USA ePay.
	 *
	 * @since 2.0.0
	 *
	 * @return string transaction description
	 */
	public function get_transaction_description() {

		return $this->transaction_description;
	}


}
