<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * The USA ePay Lifecycle class.
 *
 * @since 2.0.0
 */
class Lifecycle extends Framework\Plugin\Lifecycle {


	/**
	 * Lifecycle constructor.
	 *
	 * @since 2.0.3
	 *
	 * @param Plugin $plugin
	 */
	public function __construct( $plugin ) {

		parent::__construct( $plugin );

		$this->upgrade_versions = [
			'2.0.0',
		];
	}


	/**
	 * Installs the plugin.
	 *
	 * @since 2.0.0
	 */
	protected function install() {

		// versions prior to 2.0.0 did not set a version option, so the upgrade method needs to be called manually
		if ( get_option( 'woocommerce_usaepay_settings' ) ) {

			$this->upgrade( '1.9.1' );

			/**
			 * Fires after the plugin has been updated.
			 *
			 * @since 2.0.0
			 */
			do_action( 'wc_' . $this->get_plugin()->get_id() . '_updated' );
		}
	}


	/**
	 * Upgrades to v2.0.0
	 *
	 * @since 2.0.3
	 */
	protected function upgrade_to_2_0_0() {

		$legacy_settings = get_option( 'woocommerce_usaepay_settings', [] );

		if ( ! empty( $legacy_settings ) ) {

			update_option( 'wc_' . $this->get_plugin()->get_id() . '_legacy_active', 'yes' );

			$new_settings = [
				'enabled'                 => isset( $legacy_settings['enabled'] )     ? $legacy_settings['enabled']     : 'no',
				'title'                   => isset( $legacy_settings['title'] )       ? $legacy_settings['title']       : __( 'Credit Card', 'woocommerce-gateway-usa-epay' ),
				'description'             => isset( $legacy_settings['description'] ) ? $legacy_settings['description'] : __( 'Pay securely using your credit card.', 'woocommerce-gateway-usa-epay' ),
				'receipt_email'           => isset( $legacy_settings['custreceipt'] ) ? $legacy_settings['custreceipt'] : 'no',
				'environment'             => isset( $legacy_settings['usesandbox'] ) && 'yes' === $legacy_settings['usesandbox'] ? Framework\SV_WC_Payment_Gateway::ENVIRONMENT_TEST               : Framework\SV_WC_Payment_Gateway::ENVIRONMENT_PRODUCTION,
				'transaction_type'        => isset( $legacy_settings['command'] ) && 'authonly' === $legacy_settings['command']  ? Framework\SV_WC_Payment_Gateway::TRANSACTION_TYPE_AUTHORIZATION : Framework\SV_WC_Payment_Gateway::TRANSACTION_TYPE_CHARGE,
				'transaction_description' => isset( $legacy_settings['trandescription'] ) ? $legacy_settings['trandescription'] : 'Order from WooCommerce.',
				'legacy_gateway_url'      => isset( $legacy_settings['gatewayurl'] ) ? $legacy_settings['gatewayurl'] : '',
				'debug_mode'              => isset( $legacy_settings['debugon'] ) && 'yes' === $legacy_settings['debugon']       ? Framework\SV_WC_Payment_Gateway::DEBUG_MODE_LOG                 : Framework\SV_WC_Payment_Gateway::DEBUG_MODE_OFF,
				'enable_csc'              => isset( $legacy_settings['cvv'] ) && 'yes' === $legacy_settings['cvv'] ? 'yes' : 'no',
				'api_key'                 => '',
				'pin'                     => '',
				'test_api_key'            => '',
				'test_pin'                => '',
			];

			// api credentials
			$env = Framework\SV_WC_Payment_Gateway::ENVIRONMENT_TEST === $new_settings['environment'] ? 'test_' : '';

			$new_settings["{$env}api_key"] = isset( $legacy_settings['sourcekey'] ) ? $legacy_settings['sourcekey'] : '';
			$new_settings["{$env}pin"]     = isset( $legacy_settings['pin'] ) ? $legacy_settings['pin'] : '';

			update_option( 'woocommerce_' . Plugin::CREDIT_CARD_GATEWAY_ID . '_settings', $new_settings );
			update_option( 'woocommerce_' . Plugin::LEGACY_GATEWAY_ID . '_settings', $new_settings );
		}
	}


}
