<?php
/**
 * WooCommerce USA ePay Gateway
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce USA ePay Gateway to newer
 * versions in the future. If you wish to customize WooCommerce USA ePay Gateway for your
 * needs please refer to http://docs.woocommerce.com/document/usa-epay/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2014-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\USAePay;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * WooCommerce USA ePay Gateway main plugin class.
 *
 * @since 2.0.0
 */
class Plugin extends Framework\SV_WC_Payment_Gateway_Plugin {


	/** @var Plugin single instance of this plugin */
	protected static $instance;

	/** string version number */
	const VERSION = '2.3.0';

	/** string the plugin ID */
	const PLUGIN_ID = 'usa_epay';

	/** string credit card gateway class name */
	const CREDIT_CARD_GATEWAY_CLASS_NAME = '\\SkyVerge\\WooCommerce\\USAePay\\Gateway\\Credit_Card';

	/** string credit card gateway id */
	const CREDIT_CARD_GATEWAY_ID = 'usa_epay_credit_card';

	/** string legacy gateway class name */
	const LEGACY_GATEWAY_CLASS_NAME = '\\SkyVerge\\WooCommerce\\USAePay\\Legacy\\Gateway';

	/** string legacy gateway ID */
	const LEGACY_GATEWAY_ID = 'usa_epay_legacy';


	/**
	 * Constructs the class.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {

		parent::__construct(
			self::PLUGIN_ID,
			self::VERSION,
			array(
				'text_domain'        => 'woocommerce-gateway-usa-epay',
				'gateways'           => $this->get_active_gateways(),
				'dependencies'       => $this->get_active_dependencies(),
				'supports'           => array(
					self::FEATURE_CAPTURE_CHARGE,
				),
				'require_ssl'        => true,
				'display_php_notice' => true,
			)
		);

		// handle toggling legacy integration
		add_action( 'admin_action_wc_' . $this->get_id() . '_toggle_legacy', array( $this, 'toggle_legacy' ) );
	}


	/**
	 * Gets the currently active gateways.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	private function get_active_gateways() {

		if ( $this->is_legacy_active() ) {

			$gateways = array(
				self::LEGACY_GATEWAY_ID => self::LEGACY_GATEWAY_CLASS_NAME,
			);

		} else {

			$gateways = array(
				self::CREDIT_CARD_GATEWAY_ID => self::CREDIT_CARD_GATEWAY_CLASS_NAME,
			);
		}

		return $gateways;
	}


	/**
	 * Gets the dependencies for the currently active gateway.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	private function get_active_dependencies() {

		$dependencies = array();

		if ( ! $this->is_legacy_active() ) {
			$dependencies = array(
				'php_extensions' => array(
					'json',
					'openssl',
				),
			);
		}

		return $dependencies;
	}


	/**
	 * Determines if the legacy integration is active.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	private function is_legacy_active() {

		// note: Framework\SV_WC_Plugin::get_id() cannot be used as it's not set early enough in some cases
		return 'yes' === get_option( 'wc_' . self::PLUGIN_ID . '_legacy_active', 'no' );
	}


	/**
	 * Loads and initializes the plugin lifecycle handler.
	 *
	 * @since 2.0.0
	 */
	protected function init_lifecycle_handler() {

		$this->lifecycle_handler = new Lifecycle( $this );
	}


	/**
	 * Signals that TLS is required by this gateway plugin.
	 *
	 * @since 2.2.0
	 *
	 * @return true
	 */
	public function require_tls_1_2() {

		return true;
	}


	/** Admin methods *********************************************************/


	/**
	 * Adds a notice when gateways are switched.
	 *
	 * @see Framework\SV_WC_Plugin::add_admin_notices()
	 *
	 * @since 2.0.0
	 */
	public function add_admin_notices() {

		parent::add_admin_notices();

		if ( isset( $_GET['legacy_toggled'] ) ) {

			$message = $this->is_legacy_active() ? __( 'USA ePay Legacy Gateway is now active.', 'woocommerce-gateway-usa-epay' ) : __( 'USA ePay Gateway is now active.', 'woocommerce-gateway-usa-epay' );

			$this->get_admin_notice_handler()->add_admin_notice( $message, 'legacy-toggled', array( 'dismissible' => false ) );
		}
	}


	/**
	 * Gets the plugin action links.
	 *
	 * @since 2.0.0
	 *
	 * @param array $actions action names => anchor tags
	 * @return array
	 */
	public function plugin_action_links( $actions ) {

		$actions = parent::plugin_action_links( $actions );

		// if the legacy gateway is active, offer switching to the new gateway
		if ( $this->is_legacy_active() ) {

			$toggle_link_text = __( 'Use the USA ePay Gateway', 'woocommerce-gateway-usa-epay' );
			$insert_after_key = 'configure_' . self::LEGACY_GATEWAY_ID;

			// or allow switching to the Legacy gateway if it was previously configured
			// we only want to check that the option exists, regardless of value
		} elseif ( get_option( 'wc_' . self::PLUGIN_ID . '_legacy_active', false ) ) {

			$toggle_link_text = __( 'Use USA ePay Legacy', 'woocommerce-gateway-usa-epay' );
			$insert_after_key = 'configure_' . self::CREDIT_CARD_GATEWAY_ID;

			// otherwise for new installs, consider the latest gateway the only option
		} else {

			return $actions;
		}

		$url  = wp_nonce_url( add_query_arg( 'action', 'wc_' . $this->get_id() . '_toggle_legacy', 'admin.php' ), $this->get_file() );

		$toggle_link = array(
			'toggle_legacy' => sprintf( '<a href="%1$s" title="%2$s">%2$s</a>', esc_url( $url ), esc_html( $toggle_link_text ) ),
		);

		return Framework\SV_WC_Helper::array_insert_after( $actions, $insert_after_key, $toggle_link );
	}


	/**
	 * Handles toggling the legacy integration.
	 *
	 * @since 2.0.0
	 */
	public function toggle_legacy() {

		// security check
		if ( ! wp_verify_nonce( $_GET['_wpnonce'], $this->get_file() ) || ! current_user_can( 'manage_woocommerce' ) ) {

			wp_redirect( wp_get_referer() );

			exit;
		}

		update_option( 'wc_' . self::PLUGIN_ID . '_legacy_active', $this->is_legacy_active() ? 'no' : 'yes' );

		$return_url = add_query_arg( array( 'legacy_toggled' => 1 ), 'plugins.php' );

		// back from whence we came
		wp_redirect( $return_url );
		exit;
	}


	/** Helper methods ******************************************************/


	/**
	 * Gets the main USA ePay instance.
	 *
	 * Ensures only one instance is/can be loaded.
	 *
	 * @see wc_usa_epay()
	 *
	 * @since 2.0.0
	 *
	 * @return Plugin
	 */
	public static function instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}


	/**
	 * Gets the plugin name.
	 *
	 * @see SV_WC_Payment_Gateway::get_plugin_name()
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_plugin_name() {

		return __( 'WooCommerce USA ePay Gateway', 'woocommerce-gateway-usa-epay' );
	}


	/**
	 * Gets the plugin documentation URL.
	 *
	 * @see Framework\SV_WC_Plugin::get_documentation_url()
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_documentation_url() {

		return 'https://docs.woocommerce.com/document/usa-epay/';
	}


	/**
	 * Gets the plugin support URL.
	 *
	 * @see Framework\SV_WC_Plugin::get_support_url()
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_support_url() {

		return 'https://woocommerce.com/my-account/marketplace-ticket-form/';
	}


	/**
	 * Gets the plugin sales page URL.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_sales_page_url() {

		return 'https://woocommerce.com/products/usa-epay/';
	}


	/**
	 * Returns __DIR__
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	protected function get_file() {

		return __DIR__;
	}


}
