<?php
/**
 * Payment confirmation
 *
 * @package SimplePay\Pro\Payments\Payment_Confirmation
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments\Payment_Confirmation;

use SimplePay\Core\API;
use SimplePay\Pro\Customers\Subscription_Management;
use SimplePay\Core\Emails;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Returns the default Payment Confirmation message for "Subscription" payments.
 *
 * @since 4.0.0
 *
 * @return string
 */
function get_subscription_message_default() {
	$email     = new Emails\Email\PaymentConfirmationEmail();
	$has_email = $email->is_enabled();
	$message   = '';

	if ( true === $has_email ) {
		$message = sprintf(
			/* translators: %1$s: Total amount, %2$s: Recurring amount, %3$s: Next invoice date */
			esc_html__( 'Thank you. Your payment of %1$s has been received and your subscription has been activated. You will be charged %2$s from %3$s. Please check your email for additional information.', 'simple-pay' ),
			'{total-amount}',
			'{recurring-amount}',
			'{next-invoice-date}'
		);
	} else {
		$message = sprintf(
			/* translators: %1$s: Total amount, %2$s: Recurring amount, %3$s: Next invoice date */
			esc_html__( 'Thank you. Your payment of %1$s has been received and your subscription has been activated. You will be charged %2$s from %3$s.', 'simple-pay' ),
			'{total-amount}',
			'{recurring-amount}',
			'{next-invoice-date}'
		);
	}

	return $message;
}

/**
 * Returns the default Payment Confirmation message for "Subscription with Trial" payments.
 *
 * @since 4.0.0
 *
 * @return string
 */
function get_trial_message_default() {
	$email     = new Emails\Email\PaymentConfirmationEmail();
	$has_email = $email->is_enabled();
	$message   = '';

	if ( true === $has_email ) {
		$message = sprintf(
			/* translators: %1$s: Recurring amount, %2$s: Next invoice date */
			esc_html__( 'Thank you. Your free trial has been activated and you will be charged %1$s starting %2$s. Please check your email for additional information.', 'simple-pay' ),
			'{recurring-amount}',
			'{next-invoice-date}'
		);

	} else {
		$message = sprintf(
			/* translators: %1$s Recurring amount. %2$s Subscription start date. */
			esc_html__( 'Thank you. Your free trial has been activated and you will be charged %1$s starting %2$s.', 'simple-pay' ),
			'{recurring-amount}',
			'{next-invoice-date}'
		);

	}

	return $message;
}

/**
 * Change the base confirmation message depending on the form type.
 *
 * @since 3.6.0
 *
 * @param string $content Payment confirmation content.
 * @param array  $payment_confirmation_data {
 *   Contextual information about this payment confirmation.
 *
 *   @type \SimplePay\Vendor\Stripe\Customer               $customer Stripe Customer
 *   @type \SimplePay\Core\Abstracts\Form $form Payment form.
 *   @type object                         $subscriptions Subscriptions associated with the Customer.
 *   @type object                         $paymentintents PaymentIntents associated with the Customer.
 * }
 * @return string
 */
function get_content( $content, $payment_confirmation_data ) {
	// Not a subscription.
	if ( empty( $payment_confirmation_data['subscriptions'] ) ) {
		return $content;
	}

	$subscription = current( $payment_confirmation_data['subscriptions'] );
	$type         = $subscription->trial_end ? 'trial' : 'subscription';

	$content = simpay_get_setting(
		$type . '_details',
		call_user_func( __NAMESPACE__ . sprintf( '\get_%s_message_default', $type ) )
	);

	return $content;
}
add_filter( 'simpay_payment_confirmation_content', __NAMESPACE__ . '\\get_content', 10, 2 );

/**
 * Appends the "Update Payment Method" form to the confirmation content.
 *
 * @since 3.7.0
 * @since 4.0.0 Moved to SimplePay\Pro\Customers\Subscription_Management\on_site()
 *
 * @param string $content Payment confirmation shortcode content.
 * @param array  $payment_confirmation_data Array of data to send to the Payment Confirmation smart tags.
 */
function update_payment_method_form( $content, $payment_confirmation_data ) {
	_doing_it_wrong(
		__FUNCTION__,
		__(
			'SimplePay\Pro\Customers\Subscription_Management\on_site() is attached directly. Calling this function directly is not needed.',
			'simple-pay'
		),
		'4.0.0'
	);

	Subscription_Management\on_site( $content, $payment_confirmation_data );
}
