<?php
/**
 * Payments: Subscription
 *
 * @package SimplePay\Core\Payments\Subscription
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments\Subscription;

use SimplePay\Core\API;
use SimplePay\Core\Payments\Stripe_API;
use SimplePay\Core\Legacy;
use SimplePay\Pro\Payment_Methods;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Retrieves a Subscription.
 *
 * @since 3.7.0
 *
 * @param string|array $subscription Subscription ID or {
 *   Arguments used to retrieve a Subscription.
 *
 *   @type string $id Subscription ID.
 * }
 * @param array        $api_request_args {
 *         Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Subscription
 */
function retrieve( $subscription, $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.6.4',
		'\SimplePay\Core\API\Subscriptions\retrieve'
	);

	return API\Subscriptions\retrieve( $subscription, $api_request_args );
}

/**
 * Retrieves Subscriptions.
 *
 * @since 3.8.0
 *
 * @param array $subscriptions Optional arguments used when listing Subscriptions.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return object
 */
function all( $subscriptions = array(), $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.6.4',
		'\SimplePay\Core\API\Subscriptions\all'
	);

	return API\Subscriptions\all( $subscriptions, $api_request_args );
}

/**
 * Creates a Subscription.
 *
 * @since 3.6.0
 *
 * @param array $subscription_args Optional arguments used to create a subscription.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Subscription
 */
function create( $subscription_args = array(), $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.6.4',
		'\SimplePay\Core\API\Subscriptions\create'
	);

	return API\Subscriptions\create( $subscription_args, $api_request_args );
}

/**
 * Updates a Subscription record.
 *
 * @since 3.7.0
 *
 * @param string $subscription_id ID of the Customer to update.
 * @param array  $subscription_args Data to update Customer with.
 * @param array  $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Customer $subscription Stripe Customer.
 */
function update( $subscription_id, $subscription_args, $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.6.4',
		'\SimplePay\Core\API\Subscriptions\update'
	);

	return API\Subscriptions\update(
		$subscription_id,
		$subscription_args,
		$api_request_args
	);
}

/**
 * Cancels a Stripe Subscription.
 *
 * @since 4.0.0
 *
 * @param string $subscription_id Stripe Subscription ID.
 * @param string $schedule When to cancel the Subscription. at_period_end or immediately.
 *                                       Default immediately.
 * @param array  $api_request_args {
 *                 Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \SimplePay\Vendor\Stripe\Subscription
 */
function cancel( $subscription_id, $schedule = 'immediately', $api_request_args = array() ) {
	_deprecated_function(
		__FUNCTION__,
		'4.6.4',
		'\SimplePay\Core\API\Subscriptions\cancel'
	);

	return API\Subscriptions\cancel(
		$subscription_id,
		$schedule,
		$api_request_args
	);
}

/**
 * Generate a unique Subscription key based on a Customer ID.
 *
 * @since 3.7.0
 *
 * @param string $customer_id Customer ID.
 * @return string
 */
function get_subscription_key( $customer_id ) {
	$auth_key         = defined( 'AUTH_KEY' ) ? AUTH_KEY : '';
	$hash             = $customer_id . date( 'Y-m-d H:i:s' ) . $auth_key . uniqid( 'wpsp', true ); // @phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
	$subscription_key = strtolower( md5( $hash ) );

	return $subscription_key;
}

/**
 * Uses the current payment form request to generate arguments for a Subscription.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 * @throws \Exception Thrown if payment form price cannot be located.
 */
function get_args_from_payment_form_request(
	$form,
	$form_data,
	$form_values,
	$customer_id
) {
	// Retrieve price option.
	$price = simpay_payment_form_prices_get_price_by_id(
		$form,
		$form_data['price']['id']
	);

	if ( false === $price ) {
		throw new \Exception(
			__( 'Unable to locate payment form price.', 'simple-pay' )
		);
	}

	$quantity = isset( $form_values['simpay_quantity'] )
		? intval( $form_values['simpay_quantity'] )
		: 1;

	if ( false === $price->is_in_stock( $quantity ) ) {
		throw new \Exception(
			__( 'Sorry, this item is no longer in stock.', 'simple-pay' )
		);
	}

	$subscription_args = array(
		'items'             => array(),
		'add_invoice_items' => array(),
		'customer'          => $customer_id,
		'metadata'          => array(
			'simpay_form_id'          => $form->id,
			'simpay_price_instances'  => sprintf(
				'%s:%d',
				$price->instance_id,
				$quantity
			),
			'simpay_subscription_key' => get_subscription_key( $customer_id ),
		),
		'expand'            => array(
			'latest_invoice.payment_intent',
			'customer',
			'pending_setup_intent',
		),
	);

	// Set payment behavior if not using Stripe Checkout.
	if ( 'stripe_checkout' !== $form->get_display_type() ) {
		$subscription_args['payment_behavior'] = 'default_incomplete';
	}

	// Retrieve price option.
	$price = simpay_payment_form_prices_get_price_by_id(
		$form,
		$form_data['price']['id']
	);

	if ( false === $price ) {
		throw new \Exception(
			__( 'Unable to locate payment form price.', 'simple-pay' )
		);
	}

	$quantity = isset( $form_values['simpay_quantity'] )
		? intval( $form_values['simpay_quantity'] )
		: 1;

	if ( false === $price->is_in_stock( $quantity ) ) {
		throw new \Exception(
			__( 'Sorry, this item is no longer in stock.', 'simple-pay' )
		);
	}

	// Subscription item.
	$item = array();

	$custom_unit_amount = false !== $form_data['customAmount']
		? $form_data['customAmount']
		: $price->unit_amount_min;

	if ( $custom_unit_amount < $price->unit_amount_min ) {
		$custom_unit_amount = $price->unit_amount_min;
	}

	$price_data = array(
		'currency'    => $price->currency,
		'unit_amount' => $custom_unit_amount,
		'recurring'   => array(
			'interval'       => $price->recurring['interval'],
			'interval_count' => $price->recurring['interval_count'],
		),
		'product'     => $price->product_id,
	);

	// Tax behavior on custom amounts.
	$tax_status = get_post_meta( $form->id, '_tax_status', true );

	if ( 'automatic' === $tax_status ) {
		$tax_behavior = get_post_meta( $form->id, '_tax_behavior', true );

		$price_data['tax_behavior'] = $tax_behavior;
	}

	// Optional recurring is being used.
	if (
		isset( $form_data['isRecurring'] ) &&
		true === $form_data['isRecurring']
	) {
		// Defined Price.
		if ( isset( $price->recurring['id'] ) ) {
			$item['price'] = $price->recurring['id'];

			// Custom Price.
		} else {
			$item['price_data'] = $price_data;
		}

		// Custom amount.
	} elseif ( false === simpay_payment_form_prices_is_defined_price( $price->id ) ) {
		$item['price_data'] = $price_data;

		// Defined Price.
	} else {
		$item['price'] = $price->id;
	}

	// Quantity.
	$item['quantity'] = $quantity;

	// Adjustable quantity.
	if ( 'stripe_checkout' === $form->get_display_type() ) {
		$enable_quantity = 'yes' === simpay_get_saved_meta(
			$form->id,
			'_enable_quantity',
			'no'
		);

		$item['adjustable_quantity'] = $enable_quantity
			? array(
				'enabled' => true,
				'minimum' => 1,
			)
			: null;
	}

	// Tax.
	$tax_status = get_post_meta( $form->id, '_tax_status', true );

	if ( empty( $tax_status ) || 'fixed-global' === $tax_status ) {
		// Legacy tax percentage.
		if ( has_filter( 'simpay_form_' . $form->id . '_tax_percent' ) ) {
			$tax_percent = floatval(
				simpay_get_filtered(
					'tax_percent',
					0,
					$form->id
				)
			);

			if ( 0 !== $tax_percent ) {
				$tax = $price->unit_amount * ( $tax_percent / 100 );

				$items[] = array(
					'price_data' => array_merge(
						$price_data,
						array(
							'unit_amount' => $tax,
						)
					),
				);
			}

			// Tax rates.
		} else {
			$tax_rates = simpay_get_payment_form_tax_rates( $form );

			if ( ! empty( $tax_rates ) ) {
				$item['tax_rates'] = wp_list_pluck( $tax_rates, 'id' );
			}
		}

		// Automatic calculation.
	} elseif ( 'automatic' === $tax_status ) {
		$subscription_args['automatic_tax'] = array(
			'enabled' => true,
		);
	}

	$subscription_args['items'][] = $item;

	// Add installment plan metadata if needed.
	$max_charges = isset( $price->recurring['invoice_limit'] )
		? $price->recurring['invoice_limit']
		: 0;

	if ( 0 !== $max_charges ) {
		$charge_count = isset( $price->recurring['trial_period_days'] ) ? -1 : 0;

		$subscription_args['metadata']['simpay_charge_max']   = $max_charges;
		$subscription_args['metadata']['simpay_charge_count'] = $charge_count;
	}

	// Add trial length if needed.
	if ( isset( $price->recurring['trial_period_days'] ) ) {
		$subscription_args['trial_period_days'] =
			$price->recurring['trial_period_days'];
	}

	// Add setup fee invoice item if needed.
	$subscription_args['add_invoice_items'][] = get_setup_fee_from_payment_form_request(
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	// Add fee recovery information if needed.
	$fee_recovery_amount = 0;

	if (
		$form->has_forced_fee_recovery() ||
		isset( $form_values['fee_recovery_toggle'] )
	) {
		$fee_recovery_amount = Payment_Methods\get_form_payment_method_fee_recovery_amount(
			$form,
			sanitize_text_field( $form_values['payment_method_type'] ),
			$price_data['unit_amount']
		);

		if ( 0 !== $fee_recovery_amount ) {
			$subscription_args['items'][] = array(
				'price_data' => array(
					'unit_amount' => $fee_recovery_amount,
					'currency'    => $price_data['currency'],
					'recurring'   => $price_data['recurring'],
					'product'     => $price->product_id,
				),
			);

			// Add metadata for easier Smart Tag usage.
			$subscription_args['metadata']['simpay_fee_recovery_unit_amount'] =
				$fee_recovery_amount;
		}
	}

	/**
	 * Filters arguments used to generate a Subscription from a payment form request.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $subscription_args Arguments used to create the Subscription.
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	$subscription_args = apply_filters(
		'simpay_get_subscription_args_from_payment_form_request',
		$subscription_args,
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	return $subscription_args;
}

/**
 * Returns additional setup fee line items to add to subscription creation.
 *
 * @since 4.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @throws \Exception Thrown if payment form price cannot be located.
 * @return array
 */
function get_setup_fee_from_payment_form_request(
	$form,
	$form_data,
	$form_values,
	$customer_id
) {
	// Retrieve price option.
	$price = simpay_payment_form_prices_get_price_by_id(
		$form,
		$form_data['price']['id']
	);

	if ( false === $price ) {
		throw new \Exception(
			__( 'Unable to locate payment form price.', 'simple-pay' )
		);
	}

	$quantity = isset( $form_values['simpay_quantity'] )
		? intval( $form_values['simpay_quantity'] )
		: 1;

	if ( false === $price->is_in_stock( $quantity ) ) {
		throw new \Exception(
			__( 'Sorry, this item is no longer in stock.', 'simple-pay' )
		);
	}

	// Fetch price option line items.
	$line_items = isset( $price->line_items )
		? $price->line_items
		: array();

	if ( empty( $line_items ) ) {
		return array();
	}

	// Create Stripe Subscription-compatible line items.
	$subscription_line_items = array();

	// Tax.
	$tax_status   = get_post_meta( $form->id, '_tax_status', true );
	$tax_behavior = get_post_meta( $form->id, '_tax_behavior', true );
	$tax_code     = get_post_meta( $form->id, '_tax_code', true );

	// Tax rates.
	$tax_rates   = simpay_get_payment_form_tax_rates( $form );
	$tax_percent = floatval(
		simpay_get_filtered(
			'tax_percent',
			0,
			$form->id
		)
	);

	// Legacy Plan fee.
	$plan_fee      = isset( $line_items[1] )
		? $line_items[1]['unit_amount']
		: 0;
	$plan_fee_args = array();

	if ( $plan_fee > 0 ) {
		$plan_fee_args = array(
			'quantity'   => 1,
			'price_data' => array(
				'unit_amount' => $plan_fee,
				'currency'    => $line_items[1]['currency'],
			),
		);

		// Static taxes, if needed.
		if ( empty( $tax_status ) || 'fixed-global' === $tax_status ) {
			// Legacy tax percentage.
			if (
				has_filter( 'simpay_form_' . $form->id . '_tax_percent' ) &&
				0 !== $tax_percent
			) {
				$tax = $plan_fee * ( $tax_percent / 100 );

				$plan_fee_args['price_data']['unit_amount'] = $plan_fee + $tax;

				// Tax rates.
			} elseif ( ! empty( $tax_rates ) ) {
				$plan_fee_args['tax_rates'] = wp_list_pluck( $tax_rates, 'id' );
			}
		} elseif ( 'automatic' === $tax_status ) {
			$plan_fee_args['price_data']['tax_behavior'] = $tax_behavior;
		}

		// Use a dynamically created Product with Stripe Checkout for better line item naming.
		if ( 'stripe_checkout' === $form->get_display_type() ) {
			$plan_fee_args['price_data']['product_data'] = array(
				'name' => __( 'Plan Setup Fee', 'simple-pay' ),
			);

			if ( 'automatic' === $tax_status ) {
				$plan_fee_args['price_data']['product_data']['tax_code'] = $tax_code;
			}

			// Otherwise set an existing Product.
		} else {
			$plan_fee_args['price_data']['product'] = $price->product_id;
		}

		/**
		 * Filters the arguments used to create the Plan Setup Fee InvoiceItem.
		 *
		 * @since 3.6.0
		 *
		 * @param array                         $plan_fee_args Arguments used to create the InvoiceItem.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		$plan_fee_args = apply_filters(
			'simpay_get_plan_setup_fee_args_from_payment_form_request',
			$plan_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		$subscription_line_items[] = $plan_fee_args;

		// Fee recovery.
		if (
			$form->has_forced_fee_recovery() ||
			isset( $form_values['fee_recovery_toggle'] )
		) {
			$fee_recovery_amount = Payment_Methods\get_form_payment_method_fee_recovery_amount(
				$form,
				sanitize_text_field( $form_values['payment_method_type'] ),
				$plan_fee_args['price_data']['unit_amount'],
				isset( $price->recurring['trial_period_days'] )
			);

			if ( 0 !== $fee_recovery_amount ) {
				$subscription_line_items[] = array(
					'quantity'   => 1,
					'price_data' => array(
						'unit_amount' => $fee_recovery_amount,
						'currency'    => $line_items[1]['currency'],
						'product'     => $price->product_id,
					),
				);
			}
		}
	}

	// Initial fee.
	$initial_fee = isset( $line_items[0] )
		? $line_items[0]['unit_amount']
		: 0;

	if ( $initial_fee > 0 ) {
		$initial_fee_args = array(
			'quantity'   => 1,
			'price_data' => array(
				'unit_amount' => $initial_fee,
				'currency'    => $line_items[0]['currency'],
			),
		);

		// Static taxes, if needed.
		if ( empty( $tax_status ) || 'fixed-global' === $tax_status ) {
			// Legacy tax percentage.
			if (
				has_filter( 'simpay_form_' . $form->id . '_tax_percent' ) &&
				0 !== $tax_percent
			) {
				$tax = $initial_fee * ( $tax_percent / 100 );

				$initial_fee_args['price_data']['unit_amount'] = $initial_fee + $tax;

				// Tax rates.
			} elseif ( ! empty( $tax_rates ) ) {
				$initial_fee_args['tax_rates'] = wp_list_pluck( $tax_rates, 'id' );
			}
		} elseif ( 'automatic' === $tax_status ) {
			$initial_fee_args['price_data']['tax_behavior'] = $tax_behavior;
		}

		// Use a dynamically created Product with Stripe Checkout for better line item naming.
		if ( 'stripe_checkout' === $form->get_display_type() ) {
			$initial_fee_args['price_data']['product_data'] = array(
				'name' => __( 'Subscription Setup Fee', 'simple-pay' ),
			);

			if ( 'automatic' === $tax_status ) {
				$initial_fee_args['price_data']['product_data']['tax_code'] = $tax_code;
			}

			// Otherwise set an existing Product.
		} else {
			$initial_fee_args['price_data']['product'] = $price->product_id;
		}

		/**
		 * Filters the arguments used to create the Initial Setup Fee InvoiceItem.
		 *
		 * @since 3.6.0
		 *
		 * @param array                         $initial_fee_args Arguments used to create the InvoiceItem.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		$initial_fee_args = apply_filters(
			'simpay_get_initial_setup_fee_args_from_payment_form_request',
			$initial_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		$subscription_line_items[] = $initial_fee_args;

		// Fee recovery.
		if (
			$form->has_forced_fee_recovery() ||
			isset( $form_values['fee_recovery_toggle'] )
		) {
			$fee_recovery_amount = Payment_Methods\get_form_payment_method_fee_recovery_amount(
				$form,
				sanitize_text_field( $form_values['payment_method_type'] ),
				$initial_fee_args['price_data']['unit_amount'],
				isset( $price->recurring['trial_period_days'] )
			);

			if ( 0 !== $fee_recovery_amount ) {
				$subscription_line_items[] = array(
					'quantity'   => 1,
					'price_data' => array(
						'unit_amount' => $fee_recovery_amount,
						'currency'    => $line_items[0]['currency'],
						'product'     => $price->product_id,
					),
				);
			}
		}
	}

	return $subscription_line_items;
}

/**
 * Moves metadata attached to the Subscription to other relevant linked objects.
 *
 * This allows a more natural discovery of the metadata when viewing Payments.
 *
 * @since 3.6.0
 *
 * @param \SimplePay\Vendor\Stripe\Subscription $subscription Stripe Subscription.
 */
function move_metadata_to_linked_objects( $subscription ) {
	$invoice       = $subscription->latest_invoice;
	$paymentintent = $subscription->latest_invoice->payment_intent;

	$metadata = $subscription->metadata->toArray();

	unset( $metadata['simpay_charge_max'] );
	unset( $metadata['simpay_charge_count'] );

	$form_id = isset( $metadata['simpay_form_id'] )
		? $metadata['simpay_form_id']
		: '';

	if ( empty( $form_id ) ) {
		return;
	}

	$form = simpay_get_form( $form_id );

	if ( false === $form ) {
		return;
	}

	if ( $invoice->id ) {
		Stripe_API::request(
			'Invoice',
			'update',
			$invoice->id,
			array(
				'metadata' => $metadata,
			),
			$form->get_api_request_args()
		);
	}

	/**
	 * PaymentIntent does not exist if trialing.
	 *
	 * @link https://stripe.com/docs/billing/subscriptions/payment#handling-trial
	 */
	if ( $paymentintent ) {
		API\PaymentIntents\update(
			$paymentintent->id,
			array(
				'metadata' => $metadata,
			),
			$form->get_api_request_args()
		);
	}
}

/**
 * Moves the original Subscription metadata to other relevant linked objects
 * when an Invoice succeeds.
 *
 * @since 4.0.0
 *
 * @param \SimplePay\Vendor\Stripe\Event        $event Stripe Event object.
 * @param \SimplePay\Vendor\Stripe\Invoice      $invoice Stripe Invoice object.
 * @param \SimplePay\Vendor\Stripe\Subscription $subscription Stripe Subscription object.
 */
function move_metadata_to_linked_objects_on_invoice( $event, $invoice, $subscription ) {
	move_metadata_to_linked_objects( $subscription );
}
add_action(
	'simpay_webhook_invoice_payment_succeeded',
	__NAMESPACE__ . '\\move_metadata_to_linked_objects_on_invoice',
	10,
	3
);

/**
 * Determines if a posted form needs to handle subscription-related processes.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return bool
 */
function payment_form_request_has_subscription( $form, $form_data, $form_values ) {
	$subscription = true === $form_data['isSubscription'];
	$recurring    = payment_form_request_has_recurring_toggle(
		$form,
		$form_data,
		$form_values
	);

	return ( $subscription || $recurring );
}

/**
 * Determines if the Subscription is needed due to "Recurring Amount Toggle" field being checked.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 */
function payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) {
	return isset( $form_data['isRecurring'] ) && true === $form_data['isRecurring'];
}

/**
 * Retrieves a translated label for a Subscription's interval.
 *
 * @param string $interval Billing frequency.
 * @param string $type     Singular or plural label. Accepts `singular` and `plural`.
 * @return string
 */
function get_subscription_interval_label( $interval, $type = 'singular' ) {
	$labels = array(
		'day'   => array(
			'singular' => __( 'Day', 'simple-pay' ),
			'plural'   => __( 'Days', 'simple-pay' ),
		),
		'week'  => array(
			'singular' => __( 'Week', 'simple-pay' ),
			'plural'   => __( 'Weeks', 'simple-pay' ),
		),
		'month' => array(
			'singular' => __( 'Month', 'simple-pay' ),
			'plural'   => __( 'Months', 'simple-pay' ),
		),
		'year'  => array(
			'singular' => __( 'Year', 'simple-pay' ),
			'plural'   => __( 'Years', 'simple-pay' ),
		),
	);

	if ( ! isset( $labels[ $interval ], $labels[ $interval ][ $type ] ) ) {
		return $interval;
	}

	return $labels[ $interval ][ $type ];
}
