<?php
/**
 * REST API: SetupIntent Controller
 *
 * @package SimplePay\Pro\REST_API\v2
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.2.0
 */

namespace SimplePay\Pro\REST_API\v2;

use SimplePay\Core\API;
use SimplePay\Core\Utils;
use SimplePay\Core\REST_API\Controller;

use Exception;
use WP_REST_Response;
use WP_REST_Server;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SetupIntent_Controller.
 */
class SetupIntent_Controller extends Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpsp/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'setupintent';

	/**
	 * Registers the routes for SetupIntents.
	 *
	 * @since 4.2.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/create',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                =>
						$this->get_endpoint_args_for_item_schema(
							WP_REST_Server::CREATABLE
						),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Allows requests originating from a payment form.
	 *
	 * @since 3.6.0
	 *
	 * @param \WP_REST_Request $request Incoming REST API request data.
	 * @return \WP_Error|true Error if a permission check fails.
	 */
	public function create_item_permissions_check( $request ) {
		$checks = array(
			'rate_limit',
			'form_nonce',
			'required_fields',
			'customer_nonce',
		);

		return $this->permission_checks( $checks, $request );
	}

	/**
	 * Handles an incoming request to create a SetupIntent.
	 *
	 * @since 4.2.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REST API request data.
	 *
	 *   @type string $customer_id Customer ID previously generated with Payment Source.
	 *   @type array  $payment_method_types The list of payment method types that this SetupIntent is allowed to set up.
	 *   @type int    $form_id Form ID used to generate PaymentIntent data.
	 *   @type array  $form_data Payment Form state.
	 *   @type array  $form_values Payment form values.
	 * }
	 * @throws \Exception When required data is missing or cannot be verified.
	 * @return \WP_REST_Response
	 */
	public function create_item( $request ) {
		try {
			// Validate form request information.
			$form_id = isset( $request['form_id'] )
				? $request['form_id']
				: false;

			$form = simpay_get_form( $form_id );

			if ( false === $form ) {
				throw new Exception(
					esc_html__( 'Unable to locate payment form.', 'simple-pay' )
				);
			}

			if ( 'stripe_checkout' === $form->get_display_type() ) {
				throw new Exception(
					esc_html__( 'Invalid request. Please try again.', 'simple-pay' )
				);
			}

			$form_data = isset( $request['form_data'] )
				? json_decode( $request['form_data'], true )
				: false;

			if ( false === $form_data ) {
				throw new Exception(
					esc_html__(
						'Unable to locate payment form data.',
						'simple-pay'
					)
				);
			}

			$form_values = isset( $request['form_values'] )
				? $request['form_values']
				: false;

			if ( false === $form_values ) {
				throw new Exception(
					esc_html__(
						'Unable to locate payment form values.',
						'simple-pay'
					)
				);
			}

			// Pass Payment Method types if being used.
			$payment_method_types = isset( $request['payment_method_types'] )
				? array_map(
					'sanitize_text_field',
					$request['payment_method_types']
				)
				: null;

			// Pass Customer if being used.
			$customer_id = isset( $request['customer_id'] )
				? sanitize_text_field( $request['customer_id'] )
				: null;

			$setupintent_args = array(
				'customer'             => $customer_id,
				'payment_method_types' => $payment_method_types,
				'metadata'             => array(
					'simpay_form_id' => $form->id,
				),
			);

			/**
			 * Allow further processing before a SetupIntent is created from a posted form.
			 *
			 * @since 4.6.0
			 *
			 * @param array                         $setupintent_args Arguments used to create a PaymentIntent.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 */
			do_action(
				'simpay_before_setupintent_from_payment_form_request',
				$setupintent_args,
				$form,
				$form_data,
				$form_values
			);

			/**
			 * Filters arguments used to create a SetupIntent from a Payment Form request.
			 *
			 * @since 4.2.0
			 *
			 * @param array $setupintent_args Arguments for PaymentIntent.
			 * @param \SimplePay\Core\Abstracts\Form $form Payment Form instance.
			 * @param array $form_data Payment Form state.
			 * @param array $form_values Payment form values.
			 * @param string $customer_id Stripe Customer ID.
			 */
			$setupintent_args = apply_filters(
				'simpay_get_setupintent_args_from_payment_form_request',
				$setupintent_args,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			$setupintent = API\SetupIntents\create(
				$setupintent_args,
				$form->get_api_request_args()
			);

			/**
			 * Allow further processing after a SetupIntent is created from a posted form.
			 *
			 * @since 4.6.0
			 *
			 * @param array                         $setupintent_args Arguments used to create a PaymentIntent.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 */
			do_action(
				'simpay_after_setupintent_from_payment_form_request',
				$setupintent_args,
				$form,
				$form_data,
				$form_values
			);

			return new WP_REST_Response( $setupintent );
		} catch ( Exception $e ) {
			return new WP_REST_Response(
				array(
					'message' => Utils\handle_exception_message( $e ),
				),
				400
			);
		}
	}
}
