<?php
/**
 * Settings: Emails > Invoice Confirmation
 *
 * @package SimplePay
 * @copyright Copyright (c) 2023, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.7.3
 */

namespace SimplePay\Pro\Settings\Emails\InvoiceConfirmation;

use SimplePay\Core\Emails;
use Simplepay\Core\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Invoice Confirmation" subsection.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\InvoiceConfirmationEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Invoice Confirmation" settings.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\InvoiceConfirmationEmail();
	$id    = $email->get_id();

	// Enable/disable.
	$settings->add(
		new Settings\Setting_Checkbox(
			array(
				'id'          => sprintf( 'email_%s', $id ),
				'section'     => 'emails',
				'subsection'  => $id,
				'label'       => $email->get_label(),
				'input_label' => $email->get_description(),
				'value'       => $email->is_enabled() ? 'yes' : 'no',
				'priority'    => 10,
				'schema'      => array(
					'type'    => 'string',
					'enum'    => array( 'yes', 'no' ),
					'default' => 'yes',
				),
				'toggles'     => array(
					'value'    => 'yes',
					'settings' => array(
						sprintf( 'email_%s_to', $id ),
						sprintf( 'email_%s_subject', $id ),
						sprintf( 'email_%s_body', $id ),
						sprintf( 'email_%s_test', $id ),
					),
				),
			)
		)
	);

	// To.
	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => esc_html__( 'Customer email address', 'simple-pay' ),
				'classes'    => array(
					'regular-text',
					'readonly',
				),
				'priority'   => 20,
				'schema'     => array(
					'type' => 'string',
				),
				'readonly'   => true,
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'Payment Received for %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	// Body.
	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Message',
					'setting label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default(),
						sprintf( 'email_%s_body', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body]',
								$id
							),
							'textarea_rows' => 10,
						)
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);


/**
 * Returns the email body setting or the default.
 *
 * Installations < 4.7.3 do not have the email settings created upon
 * installation so they need fallback values. Installations >= 4.7.3 are activated
 * with default values saved to the database, so these defaults aren't used.
 *
 * @since 4.7.3
 *
 * @return string Email message body.
 */
function get_body_setting_or_default() {
	/** @var string $subscription_management */
	$subscription_management = simpay_get_setting(
		'subscription_management',
		'on-site'
	);

	$default = sprintf(
		/* translators: %1$s: Customer email, %2$s: Form title, %3$s: Charge ID, %4$s: Charge date, %5$s: Total amount, %6$s: Recurring amount */
		__( 'Dear %1$s,<br /><br />A recurring payment for your subscription to &ldquo;%2$s&rdquo; has processed.<br /><ul><li><strong>Payment ID:</strong> %3$s</li><li><strong>Subscription Activation Date:</strong> %4$s</li><li><strong>Initial Payment Amount:</strong> %5$s</li><li><strong>Recurring Payment Amount:</strong> %6$s</li></ul>', 'simple-pay' ),
		'{customer:email}',
		'{form-title}',
		'{charge-id}',
		'{charge-date}',
		'{total-amount}',
		'{recurring-amount}'
	);

	if ( 'none' !== $subscription_management ) {
		$default .= sprintf(
			/* translators: %s: Update payment method URL */
			'<br /><br />' . __( 'You can manage your subscription at any time by visiting: %s', 'simple-pay' ),
			'{update-payment-method-url}'
		);

	}

	$email = new Emails\Email\InvoiceConfirmationEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf( 'email_%s_body', $email->get_id() ),
		$default
	);

	return $body;
}
