<?php
/**
 * Settings: Emails > Manage Subscriptions
 *
 * @package SimplePay
 * @copyright Copyright (c) 2023, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.8.0
 */

namespace SimplePay\Pro\Settings\Emails\ManageSubscriptions;

use SimplePay\Core\Emails;
use Simplepay\Core\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Manage Subscriptions" subsection.
 *
 * @since 4.8.0
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\ManageSubscriptionsEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Manage Subscriptions" settings.
 *
 * @since 4.8.0
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\ManageSubscriptionsEmail();
	$id    = $email->get_id();

	// To.
	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => esc_html__( 'Customer email address', 'simple-pay' ),
				'classes'    => array(
					'regular-text',
					'readonly',
				),
				'priority'   => 20,
				'schema'     => array(
					'type' => 'string',
				),
				'readonly'   => true,
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'Manage subscriptions for %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Message',
					'setting label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default(),
						sprintf( 'email_%s_body', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body]',
								$id
							),
							'textarea_rows' => 5,
						)
					);

					printf(
						'<p class="description">%s</p>',
						__( 'A list of subscriptions associated with the purchaser\'s email address will be added below your custom message.', 'simple-pay' )
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);

/**
 * Returns the email body setting, or a default.
 *
 * Installations < 4.8.0 do not have the email settings created upon
 * installation so they need fallback values. Installations >= 4.8.0 are activated
 * with default values saved to the database, so these defaults aren't used.
 *
 * @since 4.8.0
 *
 * @return string Email message body.
 */
function get_body_setting_or_default() {
	// TODO: A discussion needs to determine the default content.
	$default = 'Use the link(s) below to manage your subscriptions.';

	$email = new Emails\Email\ManageSubscriptionsEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf( 'email_%s_body', $email->get_id() ),
		$default
	);

	return $body;
}
