<?php
/**
 * Settings: Emails > Payment Confirmation
 *
 * @package SimplePay
 * @copyright Copyright (c) 2023, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.7.3
 */

namespace SimplePay\Pro\Settings\Emails\PaymentConfirmation;

use SimplePay\Core\Emails;
use SimplePay\Core\Payments\Payment_Confirmation;
use Simplepay\Core\Settings;
use SimplePay\Pro\Payments\Payment_Confirmation as Pro_Payment_Confirmation;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Payment Confirmation" subsection.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\PaymentConfirmationEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Payment Confirmation" settings.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\PaymentConfirmationEmail();
	$id    = $email->get_id();

	$resend_confirmation_url = Settings\get_url(
		array(
			'section'    => 'emails',
			'subsection' => 'emails-tools',
		)
	);

	// Enable/Disable.
	$settings->add(
		new Settings\Setting_Checkbox(
			array(
				'id'          => sprintf( 'email_%s', $id ),
				'section'     => 'emails',
				'subsection'  => $id,
				'label'       => $email->get_label(),
				'input_label' => $email->get_description(),
				'value'       => $email->is_enabled() ? 'yes' : 'no',
				'priority'    => 10,
				'schema'      => array(
					'type'    => 'string',
					'enum'    => array( 'yes', 'no' ),
					'default' => 'yes',
				),
				'toggles'     => array(
					'value'    => 'yes',
					'settings' => array(
						sprintf( 'email_%s_to', $id ),
						sprintf( 'email_%s_subject', $id ),
						sprintf( 'email_%s_body_one_time', $id ),
						sprintf( 'email_%s_body_subscription', $id ),
						sprintf( 'email_%s_body_trial', $id ),
					),
				),
				'description' => wpautop(
					wp_kses(
						sprintf(
							/* translators: %1$s Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
							__(
								'Use the %1$sResend Payment Receipt tool%2$s to resend a receipt for a previous purchase.',
								'simple-pay'
							),
							'<a href="' . esc_url( $resend_confirmation_url ) . '">',
							'</a>'
						),
						array(
							'a' => array(
								'href'   => true,
								'class'  => true,
								'target' => true,
								'rel'    => true,
							),
						)
					)
				),
			)
		)
	);

	// To.
	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => esc_html__( 'Customer email address', 'simple-pay' ),
				'classes'    => array(
					'regular-text',
					'readonly',
				),
				'priority'   => 20,
				'schema'     => array(
					'type' => 'string',
				),
				'readonly'   => true,
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'Payment Receipt for %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	// Bodies.

	// One Time Payment.
	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body_one_time', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'One Time Payment',
					'settings subsection label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default( 'one_time' ),
						sprintf( 'email_%s_body_one_time', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body_one_time]',
								$id
							),
							'textarea_rows' => 10,
						)
					);

					Payment_Confirmation\Template_Tags\__unstable_print_tag_list(
						'',
						Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions()
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	if ( false === simpay_subscriptions_enabled() ) {
		return;
	}

	// Subscription.
	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body_subscription', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subscription Activation',
					'settings subsection label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default( 'subscription' ),
						sprintf( 'email_%s_body_subscription', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body_subscription]',
								$id
							),
							'textarea_rows' => 10,
						)
					);

					$pro_tags = Pro_Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions();
					unset( $pro_tags['trial-end-date'] );

					Payment_Confirmation\Template_Tags\__unstable_print_tag_list(
						'',
						array_merge(
							Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions(),
							$pro_tags
						)
					);
				},
				'priority'   => 60,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	if ( false === simpay_get_license()->is_enhanced_subscriptions_enabled() ) {
		return;
	}

	// Subscription with Trial.
	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body_trial', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Free Trial Activation',
					'settings subsection label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default( 'trial' ),
						sprintf( 'email_%s_body_trial', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body_trial]',
								$id
							),
							'textarea_rows' => 10,
						)
					);

					Payment_Confirmation\Template_Tags\__unstable_print_tag_list(
						'',
						array_merge(
							Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions(),
							Pro_Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions()
						)
					);
				},
				'priority'   => 80,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);

/**
 * Returns the email body setting for the given type, or the default.
 *
 * Installations < 4.7.3 do not have the email settings created upon
 * installation so they need fallback values. Installations >= 4.7.3 are activated
 * with default values saved to the database, so these defaults aren't used.
 *
 * @since 4.7.0
 *
 * @param string $type Body type (one_time, subscription, trial).
 * @return string Email message body.
 */
function get_body_setting_or_default( $type ) {
	$subscription_management = simpay_get_setting(
		'subscription_management',
		'on-site'
	);

	switch ( $type ) {
		case 'one_time':
			$default = sprintf(
				/* translators: %1$s: Customer email, %2$s: Charge date, %3$s: Form title, %4$s: Charge ID, %5$s: Charge date, %6$s: Total amount */
				__( 'Dear %1$s,<br /><br />Thank you for your payment on %2$s for &ldquo;%3$s&rdquo;.<br /><ul><li><strong>Payment ID:</strong> %4$s</li><li><strong>Payment Date:</strong> %5$s</li><li><strong>Payment Amount:</strong> %6$s</li></ul>', 'simple-pay' ),
				'{customer:email}',
				'{charge-date}',
				'{form-title}',
				'{charge-id}',
				'{charge-date}',
				'{total-amount}'
			);

			break;
		case 'subscription':
			$default = sprintf(
				/* translators: %1$s: Customer email, %2$s: Form title, %3$s: Charge date, %4$s: Charge ID, %5$s: Charge date, %6$s: Total amount, %7$s: Recurring amount */
				__( 'Dear %1$s,<br /><br />Thank you for subscribing to &ldquo;%2$s&rdquo; on %3$s.<br /><ul><li><strong>Payment ID:</strong> %4$s</li><li><strong>Subscription Activation Date:</strong> %5$s</li><li><strong>Initial Payment Amount:</strong> %6$s</li><li><strong>Recurring Payment Amount:</strong> %7$s</li></ul>', 'simple-pay' ),
				'{customer:email}',
				'{form-title}',
				'{charge-date}',
				'{charge-id}',
				'{charge-date}',
				'{total-amount}',
				'{recurring-amount}'
			);

			if ( 'none' !== $subscription_management ) {
				$default .= sprintf(
					/* translators: %s: Update payment method URL */
					__( '<br /><br />You can manage your subscription at any time by visiting: %s', 'simple-pay' ),
					'{update-payment-method-url}'
				);
			}

			break;
		case 'trial':
			$default = sprintf(
				/* translators: %1$s: Customer email, %2$s: Form title, %3$s: Charge date, %4$s: Trial end date, %5$s: Charge date, %6$s: Trial end date, %7$s: Recurring amount */
				__( 'Dear %1$s,<br /><br />Thank you for subscribing to &ldquo;%2$s&rdquo; on %3$s. Your subscription includes a free trial until %4$s. <br /><ul><li><strong>Subscription Activation Date:</strong> %5$s</li><li><strong>Free Trial End Date:</strong> %6$s</li><li><strong>Recurring Payment Amount:</strong> %7$s</li></ul>', 'simple-pay' ),
				'{customer:email}',
				'{form-title}',
				'{charge-date}',
				'{trial-end-date}',
				'{charge-date}',
				'{trial-end-date}',
				'{recurring-amount}'
			);

			if ( 'none' !== $subscription_management ) {
				$default .= sprintf(
					/* translators: %s: Update payment method URL */
					'<br /><br />' . __( 'You can manage your subscription at any time by visiting: %s', 'simple-pay' ),
					'{update-payment-method-url}'
				);
			}

			break;
	}

	$email = new Emails\Email\PaymentConfirmationEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf(
			'email_%s_body_%s',
			$email->get_id(),
			$type
		),
		$default
	);

	return $body;
}
