<?php
/**
 * Settings: Emails > Upcoming Invoice
 *
 * @package SimplePay
 * @copyright Copyright (c) 2023, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.7.3
 */

namespace SimplePay\Pro\Settings\Emails\UpcomingInvoice;

use SimplePay\Core\Emails;
use SimplePay\Core\Payments\Payment_Confirmation;
use Simplepay\Core\Settings;
use SimplePay\Core\Utils;
use SimplePay\Pro\Payments\Payment_Confirmation as Pro_Payment_Confirmation;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Upcoming Invoice" subsection.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\UpcomingInvoiceEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Upcoming Invoice" settings.
 *
 * @since 4.7.3
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\UpcomingInvoiceEmail();
	$id    = $email->get_id();

	$subscription_management = simpay_get_setting(
		'subscription_management',
		'on-site'
	);

	$subscription_management_link = Settings\get_url(
		array(
			'section'    => 'customers',
			'subsection' => 'subscription-management',
			'setting'    => 'subscription_management',
		)
	);

	// Enable/disable.
	$settings->add(
		new Settings\Setting_Checkbox(
			array(
				'id'          => sprintf( 'email_%s', $id ),
				'section'     => 'emails',
				'subsection'  => $id,
				'label'       => $email->get_label(),
				'input_label' => $email->get_description(),
				'value'       => $email->is_enabled() ? 'yes' : 'no',
				'priority'    => 10,
				'schema'      => array(
					'type'    => 'string',
					'enum'    => array( 'yes', 'no' ),
					'default' => 'yes',
				),
				'toggles'     => array(
					'value'    => 'yes',
					'settings' => array(
						sprintf( 'email_%s_to', $id ),
						sprintf( 'email_%s_subject', $id ),
						sprintf( 'email_%s_body', $id ),
					),
				),
				'description' => wpautop(
					wp_kses(
						sprintf(
							/* translators: %1$s Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
							__(
								'Email is scheduled based on the %1$supcoming renewal events%2$s setting in Stripe.',
								'simple-pay'
							),
							'<a href="https://dashboard.stripe.com/settings/billing/automatic" target="_blank" rel="noopener noreferrer" class="simpay-external-link">',
							Utils\get_external_link_markup() . '</a>'
						) . '<br />' .
						sprintf(
							/* translators: %1$s Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
							__(
								'Customers will be able to update their subscription based on the %1$ssubscription management%2$s setting.',
								'simple-pay'
							),
							'<a href="' . esc_url( $subscription_management_link ) . '">',
							'</a>'
						),
						array(
							'br'   => true,
							'a'    => array(
								'href'   => true,
								'class'  => true,
								'target' => true,
								'rel'    => true,
							),
							'span' => array(
								'class' => 'screen-reader-text',
							),
						)
					)
				),
			)
		)
	);

	// To.
	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => esc_html__( 'Customer email address', 'simple-pay' ),
				'classes'    => array(
					'regular-text',
					'readonly',
				),
				'priority'   => 20,
				'schema'     => array(
					'type' => 'string',
				),
				'readonly'   => true,
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'Upcoming Invoice for %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	// Body.
	$default = sprintf(
		/* translators: %1$s: Customer email, %2$s: Form title, %3$s: Next invoice date, %4$s: Charge date, %5$s: Total amount, %6$s: Recurring amount */
		esc_html__( 'Dear %1$s,<br /><br />This is a friendly reminder that the next payment for your subscription to &ldquo;%2$s&rdquo; will automatically process on %3$s. Your payment method on file will be charged at that time.<br /><ul><li><strong>Subscription Activation Date:</strong> %4$s</li><li><strong>Initial Payment Amount:</strong> %5$s</li><li><strong>Recurring Payment Amount:</strong> %6$s</li></ul>', 'simple-pay' ),
		'{customer:email}',
		'{form-title}',
		'{next-invoice-date}',
		'{charge-date}',
		'{total-amount}',
		'{recurring-amount}'
	);

	if ( 'none' !== $subscription_management ) {
		$default .= sprintf(
			/* translators: %s: Update payment method URL */
			'<br /><br />' . esc_html__( 'You can manage your subscription at any time by visiting: %s', 'simple-pay' ),
			'{update-payment-method-url}'
		);

	}

	$body = simpay_get_setting( sprintf( 'email_%s_body', $id ), $default );

	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Message',
					'setting label',
					'simple-pay'
				),
				'output'     => function () use ( $id, $body ) {
					wp_editor(
						$body,
						sprintf( 'email_%s_body', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body]',
								$id
							),
							'textarea_rows' => 10,
						)
					);

					Payment_Confirmation\Template_Tags\__unstable_print_tag_list(
						'',
						array_merge(
							Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions(),
							Pro_Payment_Confirmation\Template_Tags\__unstable_get_tags_and_descriptions()
						)
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);

/**
 * Returns the email body setting, or a default.
 *
 * Installations < 4.7.3 do not have the email settings created upon
 * installation so they need fallback values. Installations >= 4.7.3 are activated
 * with default values saved to the database, so these defaults aren't used.
 *
 * @since 4.7.3
 *
 * @return string Email message body.
 */
function get_body_setting_or_default() {
	$default = sprintf(
		/* translators: %1$s: Customer email, %2$s: Form title, %3$s: Next invoice date, %4$s: Charge date, %5$s: Total amount, %6$s: Recurring amount */
		esc_html__( 'Dear %1$s,<br /><br />This is a friendly reminder that the next payment for your subscription to &ldquo;%2$s&rdquo; will automatically process on %3$s. Your payment method on file will be charged at that time.<br /><ul><li><strong>Subscription Activation Date:</strong> %4$s</li><li><strong>Initial Payment Amount:</strong> %5$s</li><li><strong>Recurring Payment Amount:</strong> %6$s</li></ul>', 'simple-pay' ),
		'{customer:email}',
		'{form-title}',
		'{next-invoice-date}',
		'{charge-date}',
		'{total-amount}',
		'{recurring-amount}'
	);

	$subscription_management = simpay_get_setting(
		'subscription_management',
		'on-site'
	);

	if ( 'none' !== $subscription_management ) {
		$default .= sprintf(
			/* translators: %s: Update payment method URL */
			'<br /><br />' . esc_html__( 'You can manage your subscription at any time by visiting: %s', 'simple-pay' ),
			'{update-payment-method-url}'
		);

	}

	$email = new Emails\Email\UpcomingInvoiceEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf( 'email_%s_body', $email->get_id() ),
		$default
	);

	return $body;
}
