/* global wpforms_builder, wpforms_paypal_standard */

/**
 * WPForms Paypal Standard function.
 *
 * @since 1.5.0
 */
'use strict';

var WPFormsPaypalStandard = window.WPFormsPaypalStandard || ( function( document, window, $ ) {

	/**
	 * Elements holder.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var el = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.5.0
		 */
		init: function() {

			// Cache DOM elements.
			el = {
				$enableToggle: $( '#wpforms-panel-field-paypal_standard-enable' ),
				$modeSelect: $( '#wpforms-panel-field-paypal_standard-mode' ),
				$productionEmailWrap: $( '#wpforms-panel-field-paypal_standard-production_email-wrap' ),
				$sandboxEmailWrap: $( '#wpforms-panel-field-paypal_standard-sandbox_email-wrap' ),
			};

			app.ready();
		},

		/**
		 * Initialized once the DOM is fully loaded.
		 *
		 * @since 1.5.0
		 */
		ready: function() {

			app.bindUIActions();

			app.settingsToggle.call( el.$enableToggle );
		},

		/**
		 * Process various events as a response to UI interactions.
		 *
		 * @since 1.5.0
		 */
		bindUIActions: function() {

			$( document ).on( 'wpformsSaved', app.requiredFieldsCheck );
			$( document ).on( 'wpformsSaved', app.paymentsFieldsCheck );

			el.$enableToggle.on( 'click', app.settingsToggle );
			el.$modeSelect.on( 'change', app.showEmailField );
		},

		/**
		 * On form save notify users about required fields.
		 *
		 * @since 1.5.0
		 */
		requiredFieldsCheck: function() {

			if ( ! el.$enableToggle.is( ':checked' ) ) {
				return;
			}

			var $emailField = app.isProductionMode() ? el.$productionEmailWrap.find( 'input' ) : el.$sandboxEmailWrap.find( 'input' );

			if ( $emailField.val().trim() ) {
				return;
			}

			$.alert( {
				title: wpforms_builder.heads_up,
				content: wpforms_builder.paypal_standard_required_email,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_builder.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		/**
		 * Toggle payments content.
		 *
		 * @since 1.5.0
		 */
		settingsToggle: function() {

			$( '.wpforms-panel-content-section-paypal-standard-body' ).toggle( $( this ).prop( 'checked' ) );
		},

		/**
		 * Show email field based on selected mode.
		 *
		 * @since 1.5.0
		 */
		showEmailField: function() {

			var isProductionMode = app.isProductionMode();

			el.$productionEmailWrap.toggleClass( 'wpforms-hidden', ! isProductionMode );
			el.$sandboxEmailWrap.toggleClass( 'wpforms-hidden', isProductionMode );
		},

		/**
		 * Check if selected mode is in production state.
		 *
		 * @since 1.5.0
		 *
		 * @returns {boolean} True if production is selected.
		 */
		isProductionMode: function() {

			return el.$modeSelect.val() === 'production';
		},

		/**
		 * On form save notify users about required fields.
		 *
		 * @since 1.6.0
		 */
		paymentsFieldsCheck: function() {

			if ( ! el.$enableToggle.is( ':checked' ) ) {
				return;
			}

			var showNotice = true;

			$( '#wpforms-panel-fields .wpforms-panel-content .wpforms-field' ).each( function() {
				if ( wpforms_paypal_standard.payment_fields.indexOf( $( this ).data( 'field-type' ) ) > -1 ) {
					showNotice = false;

					return false;
				}
			} );

			if ( ! showNotice ) {
				return;
			}

			$.alert( {
				title: wpforms_builder.heads_up,
				content: wpforms_builder.paypal_standard_required_payment_field,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_builder.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

// Initialize.
WPFormsPaypalStandard.init();
