<?php

namespace wpbuddy\rich_snippets\pro;

use wpbuddy\rich_snippets\Settings_Section;
use wpbuddy\rich_snippets\Settings_Setting;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly


/**
 * Class Settings.
 *
 * Admin settings actions.
 *
 * @package wpbuddy\rich_snippets
 *
 * @since   2.19.0
 */
class Admin_Settings_Controller extends \wpbuddy\rich_snippets\Admin_Settings_Controller {
	/**
	 * Admin_Settings_Controller constructor.
	 *
	 * @since 2.19.0
	 */
	public function __construct() {
		add_filter( 'wpbuddy/rich_snippets/settings', [ $this, 'pro_settings' ] );
		add_action( 'wpbuddy/rich_snippets/settings/setting/after/wpb_rs/setting/remove_microdata_schema', [
			$this,
			'microdata_info'
		] );
		parent::__construct();
	}

	/**
	 * Add PRO settings.
	 *
	 * @param Settings_Section[] $settings
	 *
	 * @return Settings_Section[]
	 *
	 * @since 2.19.0
	 */
	public function pro_settings( $settings ) {

		$string_to_bool_fcn = function ( $val ) {
			return Helper_Model::instance()->string_to_bool( $val ) ? 1 : 0;
		};

		if ( function_exists( 'WC' ) ) {
			$settings['frontend']->add_setting( array(
				'label'             => __( 'Remove WooCommerce schema', 'rich-snippets-schema' ),
				'title'             => __( 'WooCommerce', 'rich-snippets-schema' ),
				'type'              => 'checkbox',
				'name'              => 'remove_wc_schema',
				'default'           => false,
				'sanitize_callback' => $string_to_bool_fcn,
				'autoload'          => true,
				'description'       => __( 'WooCommerce adds its own schema.org syntax for products. If you don\'t want to use it, the plugin can try to remove it for you so that you can write your own Rich Snippets for products.', 'rich-snippets-schema' ),
			) );

			$settings['frontend']->add_setting( array(
				'label'             => __( 'PreOrder instead of LimitedAvailability', 'rich-snippets-schema' ),
				'title'             => __( 'Item availability', 'rich-snippets-schema' ),
				'type'              => 'checkbox',
				'name'              => 'wc_availability_use_preorder',
				'default'           => false,
				'sanitize_callback' => $string_to_bool_fcn,
				'autoload'          => true,
				'description'       => __( 'Set schema.org/PreOrder instead of schema.org/LimitedAvailability if item is out of stock and the "WooCommerde: Availability" or "WooCommerce: Offers" field types are used.', 'rich-snippets-schema' ),
			) );
		}

		$settings['backend']->add_setting( [
			'label'             => __( 'Inform me about expired support via E-Mail.', 'rich-snippets-schema' ),
			'title'             => __( 'Expired support', 'rich-snippets-schema' ),
			'type'              => 'checkbox',
			'name'              => 'expired_support_info_mail',
			'default'           => true,
			'sanitize_callback' => $string_to_bool_fcn,
			'autoload'          => false,
			'description'       => __( 'Your WordPress website will remind you via email if support for SNIP is expiring soon or has already expired.', 'rich-snippets-schema' ),
		] );

		$microdata_setting           = $settings['frontend']->get_setting( 'remove_microdata_schema' );
		$microdata_setting->disabled = false;
		$microdata_setting->label    = __( 'Remove (old) Microdata schemas from any third party plugins and/or the theme.', 'rich-snippets-schema' );


		return $settings;
	}

	/**
	 * Prints a warning if the creator schemas is active.
	 *
	 * @param Settings_Setting $setting
	 *
	 * @since 2.20.0
	 */
	public function microdata_info( $setting ) {
		if ( ! (bool) $setting->value ) {
			return;
		}

		printf(
			'<div class="notice  notice-alt inline"><p>%s</p></div>',
			__( '<strong>Use this at your own risk</strong> as it is a heavy task to perform on your server. Here are some tips: If possible, you should disable old microdata via the settings page of the third-party plugin or theme. In case of doubt, ask the author of the plugin directly on how you can remove their code. If you really need this functionality, try to use a Caching Plugin.', 'rich-snippets-schema' )
		);
	}
}